package charactermanaj;

import java.awt.Font;
import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.SwingUtilities;
import javax.swing.UIDefaults;
import javax.swing.UIManager;

import charactermanaj.clipboardSupport.ImageSelection;
import charactermanaj.model.AppConfig;
import charactermanaj.model.util.StartupSupport;
import charactermanaj.ui.MainFrame;
import charactermanaj.ui.ProfileListManager;
import charactermanaj.ui.SelectCharatersDirDialog;
import charactermanaj.util.ConfigurationDirUtilities;
import charactermanaj.util.DirectoryConfig;
import charactermanaj.util.ErrorMessageHelper;
import charactermanaj.util.ApplicationLoggerConfigurator;

/**
 * エントリポイント用クラス
 * @author seraphy
 */
public final class Main implements Runnable {
	
	/**
	 * ロガー.<br>
	 */
	private static Logger logger = Logger.getLogger(Main.class.getName());

	/**
	 * Mac OS Xであるか?
	 */
	private static final boolean isMacOSX;
	
	/**
	 * Mac OS XもしくはLinuxであるか?
	 */
	private static final boolean isLinuxOrMacOSX;

	
	/**
	 * クラスイニシャライザ
	 */
	static {
		// Mac OS Xでの実行判定
		// システムプロパティos.nameは、すべてのJVM実装に存在する.
		// 基本ディレクトリの位置の決定に使うため、
		// なによりも、まず、これを判定しないとダメ.(順序が重要)
		String lcOS = System.getProperty("os.name").toLowerCase();
		isMacOSX = lcOS.startsWith("mac os x");
		isLinuxOrMacOSX = isMacOSX || lcOS.indexOf("linux") >= 0;

		// ロガーの準備
		try {
			// ローカルファイルシステム上のユーザ定義ディレクトリから
			// ログの設定を読み取る.(OSにより、設定ファイルの位置が異なることに注意)
			ApplicationLoggerConfigurator.configure();

		} catch (Throwable ex) {
			ex.printStackTrace();
		}

		logger.log(Level.INFO, "os.name=" + lcOS + "/isMacOSX=" + isMacOSX);
	}

	/**
	 * UIをセットアップする.
	 * @throws Exception いろいろな失敗
	 */
	private static void setupUIManager() throws Exception {
		//System.setProperty("swing.aatext", "true");
		//System.setProperty("awt.useSystemAAFontSettings", "on");
		
		// MacOSXであれば、スクリーンメニューを有効化
		if (isMacOSX()) {
			System.setProperty("apple.laf.useScreenMenuBar", "true");
			System.setProperty(
					"com.apple.mrj.application.apple.menu.about.name",
					"CharacterManaJ");
		}

		// 実行プラットフォームのネイティブな外観にする.
		UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		
		// JSpliderのvalueを非表示 (GTKではデフォルトで有効のため)
		UIManager.put("Slider.paintValue", Boolean.FALSE);

		// JTextAreaの既定フォントを固定幅から、標準テキストと同じフォントに変更.
		// (Linuxなどで固定幅フォントでは日本語フォントを持っていないため。)
		Object textFieldFontUI = UIManager.get("TextField.font");
		if (textFieldFontUI == null) {
			// もし無ければダイアログUIフォントを使う.(これは日本語をサポートするであろう。)
			textFieldFontUI = new UIDefaults.ProxyLazyValue(
					"javax.swing.plaf.FontUIResource", null, new Object[] {
							"dialog", new Integer(Font.PLAIN), new Integer(12)});
		}
		UIManager.put("TextArea.font", textFieldFontUI);
	}
	
	/**
	 * 初期処理およびメインフレームを構築する.<br>
	 * SwingのUIスレッドで実行される.<br>
	 */
	public void run() {
		try {
			// UIManagerのセットアップ.
			try {
				setupUIManager();

			} catch (Exception ex) {
				// UIManagerの設定に失敗した場合はログに書いて継続する.
				ex.printStackTrace();
				logger.log(Level.WARNING, "UIManager setup failed.", ex);
			}
			
			// アプリケーション設定の読み込み
			AppConfig appConfig = AppConfig.getInstance();
			appConfig.loadConfig();
			
			// クリップボードサポートの設定
			if ( !ImageSelection.setupSystemFlavorMap()) {
				logger.log(Level.WARNING, "failed to set the clipboard-support.");
			}

			// キャラクターセットディレクトリの選択
			File defaultCharacterDir = ConfigurationDirUtilities.getDefaultCharactersDir();
			File currentCharacterDir = SelectCharatersDirDialog.getCharacterDir(defaultCharacterDir);
			if (currentCharacterDir == null) {
				// キャンセルされたので終了する.
				logger.info("luncher canceled.");
				return;
			}
			DirectoryConfig.getInstance().setCharactersDir(currentCharacterDir);
			
			// スタートアップ時の初期化
			StartupSupport.getInstance().doStartup();
			
			// デフォルトのプロファイルを開く.
			// (最後に使ったプロファイルがあれば、それが開かれる.)
			MainFrame mainFrame = ProfileListManager.openDefaultProfile();
			if (isMacOSX()) {
				// MacOSXであればスクリーンメニューからのイベントをハンドルできるようにする.
				// OSXにしか存在しないクラスを利用するためリフレクションとしている.
				Class<?> clz = Class.forName("charactermanaj.ui.MainFramePartialForMacOSX");
				Method mtd = clz.getMethod("setupScreenMenu", MainFrame.class);
				mtd.invoke(null, mainFrame);
			}
			
			// 表示(および位置あわせ)
			mainFrame.showMainFrame();

		} catch (Throwable ex) {
			// なんらかの致命的な初期化エラーがあった場合、ログとコンソールに表示
			// ダイアログが表示されるかどうかは状況次第.
			ex.printStackTrace();
			logger.log(Level.SEVERE, "Application initiation failed.", ex);
			ErrorMessageHelper.showErrorDialog(null, ex);

			// メインフレームを破棄します.
			MainFrame.closeAllProfiles();
		}
	}
	
	/**
	 * エントリポイント.<br>
	 * 最初のメインフレームを開いたときにMac OS Xであればスクリーンメニューの登録も行う.<br>
	 * @param args 引数(未使用)
	 */
	public static void main(String[] args) {
		try {
			Main main = new Main();
			// SwingのUIスレッドで実行する
			SwingUtilities.invokeAndWait(main);

		} catch (InvocationTargetException ex) {
			ex.printStackTrace();
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}
	
	/**
	 * Mac OS Xで動作しているか?
	 * @return Max OS X上であればtrue
	 */
	public static boolean isMacOSX() {
		return isMacOSX;
	}

	/**
	 * Mac OS X、もしくはlinuxで動作しているか?
	 * @return Mac OS X、もしくはlinuxで動作していればtrue
	 */
	public static boolean isLinuxOrMacOSX() {
		return isLinuxOrMacOSX;
	}
}
