/*
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authn.saml2.web;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.NameIDCache;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.log.exception.CitkSystemException;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;

/**
 * Mule ESBのSAMLアサーションパース用トランスフォーマークラス実装。
 * @author toyota
 *
 */
public class SAML2InfoIntoMuleMessage extends AbstractMessageTransformer {
	
    /**
     * カスタムトランスフォーマーのコンストラクタ。
     */
    public SAML2InfoIntoMuleMessage() {
        super();
        registerSourceType(DataTypeFactory.INPUT_STREAM);
        registerSourceType(DataTypeFactory.MULE_MESSAGE);
        setReturnDataType(DataTypeFactory.MULE_MESSAGE);
    }
	
    /**
     * ログメッセージ出力用オブジェクト。
     */
    private static final CitkLogger logger =
    		CitkLogger.getLog(SAML2InfoIntoMuleMessage.class);
    
    
    /**
     * SAMLアサーションのNameIDタグ開始文字列
     */
    private static final String NAMEID_START_TAG = "<saml:NameID";
    
    /**
     * SAMLアサーションのNameIDタグ終了文字列
     */
    private static final String NAMEID_END_TAG = "</saml:NameID>";
    
    /**
     * SAMLアサーションのNameID内NameQualifierパラメータの開始文字列
     */
    private static final String NAMEID_NAMEQUALIFIER_TAG = "NameQualifier=\"";
   
    /**
     * ログイン済みユーザの情報を収めたキャッシュ。
     * @see NameIDCache
     */
    private NameIDCache cache = null;

    /**
     * cacheのgetter。
     * mule-global-config.xmlにてSpringBeanとして使用している。
     *
     * @return NameIDCache
     */
    public NameIDCache getCache() {
        return cache;
    }

    /**
     * cacheのsetter。
     *
     * @see #getCache
     * @param cache
     */
    public void setCache(NameIDCache cache) {
        this.cache = cache;
    } 
   

    @Override
    /**
     * OpenAMのfedletのAssersion Consumer Service であるfedletapplication
     * からのレスポンス(SAML2 ResponseメッセージのXMLを含むHTMLドキュメント)から
     * ResponseメッセージのXML部分のみを切り出し、さらにNameID、IdPEntityIDを切り出す。
     * MuleMessageのセッションプロパティにそれらをセットし、NameIDをキャッシュする。
     * @param message 変換対象のMuleMessageオブジェクト
     * @param outputEncoding エンコード
     * @return 変換後のMuleMessageオブジェクト
     * @throws TransformerException
     */
    public Object transformMessage(MuleMessage message, String outputEncoding)
                                                   throws TransformerException {
        String encode = message.getEncoding();
        String body = null;
        
		try {
			body = message.getPayloadAsString(encode);
		} catch (Exception e) {
			throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_PAYLOARD,e);
		}

        if (body.equals("{NullPayload}")) {
            throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_PAYLOARD);
        }
        
        logger.info(CitkConstants.LOG_MSG_AUTHN_FEDLET_RESPONSE, body);

        int sp = body.indexOf(NAMEID_START_TAG);
        int ep = body.indexOf(NAMEID_END_TAG) + NAMEID_END_TAG.length();
        
        String tmp = null;
        if (sp > 0 && sp < ep) {
       		tmp = body.substring(sp, ep);
        }
        if (tmp == null) {
            throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_NAEID_TAG);
        } else {
            message.setOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY,
            		message.getInboundProperty(HttpConnector.HTTP_STATUS_PROPERTY));
            
            String nameID = getNameId(tmp);
            
            if(nameID == null || nameID.isEmpty()){
            	throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_NAEID);
            }
            logger.info(CitkConstants.LOG_MSG_GET_NAMEID, nameID);
            
            String idpEntityId = getIdpEntityId(tmp);
            
            if(idpEntityId == null || idpEntityId.isEmpty()){
            	throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_IDPENTITYID);
            }
            logger.info(CitkConstants.LOG_MSG_GET_IDPENTITYID, idpEntityId);
            
            message.setSessionProperty(CitkConstants.NAMEID_KEY, nameID);
            message.setSessionProperty(CitkConstants.IDP_ENTITYID_KEY, idpEntityId);
            
            //Todo:キャッシュのValueには何を入れるべき？
            cache.update(nameID, tmp);
        }


        return message;
    }
    
    /**
     * SAMLアサーションからNameIDの切り出し
     * @param data SAMLアサーション
     * @return NameID
     */
    private String getNameId(String data){
        String nameId = data.replaceAll(NAMEID_END_TAG, "");
        return nameId.replaceAll(NAMEID_START_TAG + ".*\\>", "");
    }
    
    /**
     * SAMLアサーションからIdpEntityIdの切り出し
     * @param data SAMLアサーション
     * @return IdpEntityId
     */
    private String getIdpEntityId(String data){
    	int sp = data.indexOf(NAMEID_NAMEQUALIFIER_TAG) + NAMEID_NAMEQUALIFIER_TAG.length();
        String tmp = data.substring(sp);
        String[] temps = tmp.split("\"");
       
        return temps[0];
    }
    

    
}
