package jp.co.ogis_ri.citk.authn.saml2.web;

import static org.junit.Assert.*;

import java.io.BufferedReader;
import java.io.FileReader;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import jp.co.ogis_ri.citk.authn.saml2.web.SAML2InfoIntoMuleMessage;
import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.NameIDCache;
import jp.co.ogis_ri.citk.common.log.exception.CitkSystemException;

import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;


public class SAML2InfoIntoMuleMessageTestCase extends AbstractMuleContextTestCase{

	private static String testNameID = "Dk0FU/DsJFHZYQBD3wLBS1pQZnia";
	private static String testNameIDValue = "<saml:NameID Format=\"urn:oasis:names:tc:SAML:2.0:nameid-format:transient\" NameQualifier=\"idp201\" SPNameQualifier=\"sp202\">Dk0FU/DsJFHZYQBD3wLBS1pQZnia</saml:NameID>";
	private static String testIdpEntityID = "idp201";
	private static String dummyResponse = "";
	
	private static String encoding = "UTF-8";
	private static String requestPath = "/sp1/app/test/test.html";
	private static Map<String, Object>  messageProperties;
	
	static
	{
		Map<String, Object> map = new HashMap<String, Object>();
		map.put(HttpConnector.HTTP_METHOD_PROPERTY, "GET");
		map.put(HttpConstants.HEADER_ACCEPT, "text/html, application/xhtml+xml, */*");
		map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
		map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
		map.put(HttpConstants.HEADER_HOST,  "www.acme.com");
		map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
		map.put(HttpConnector.HTTP_REQUEST_PROPERTY, requestPath );
		map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, requestPath);
		map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);
		map.put(HttpConnector.HTTP_STATUS_PROPERTY,200);
		
		messageProperties = Collections.unmodifiableMap(map);
	}
	
	static { 
		String uri = "./src/test/resources/dummyResponse.html";
		
		try{
            FileReader f = new FileReader(uri);
            BufferedReader b = new BufferedReader(f);
            String s = null;
            while((s = b.readLine())!=null){
            	dummyResponse = dummyResponse.concat(s);
            }
        }catch(Exception e){
            fail("ファイル読み込み失敗");
        }
			
	}
	
	//setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception
    {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    //tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception 
    {
        super.doTearDown();
        //RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }
	

	public Transformer getTransformer() throws Exception {
		Transformer trans = createObject(SAML2InfoIntoMuleMessage.class);
        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory().getInboundEndpoint(
            builder);
        trans.setEndpoint(endpoint);

 		NameIDCache cache = NameIDCache.getInstance();

 		cache.setCache(new ConcurrentHashMap<String, String>());
 		 ((SAML2InfoIntoMuleMessage) trans).setCache(cache);
 		 
        return trans;
	}



	public Object getTestData() {
		MuleMessage mes = new DefaultMuleMessage(dummyResponse, messageProperties, null, null, muleContext);
		mes.setEncoding(encoding);
		return mes;
	}
	
	public Object getTestData_NullPayloard_Exception() {
		MuleMessage mes = new DefaultMuleMessage(null, messageProperties, null, null, muleContext);
		mes.setEncoding(encoding);
		return mes;
	}
	
	public Object getTestData_InvalidPayloard_Exception() {
		MuleMessage mes = new DefaultMuleMessage("InvalidPayloardInvalidPayloardInvalidPayloard", messageProperties, null, null, muleContext);
		mes.setEncoding(encoding);
		return mes;
	}
	
	public Object getTestData_NullNameID_Exception() {
		String invalidDummyResponse = dummyResponse.replaceAll(testNameID, "");
		
		MuleMessage mes = new DefaultMuleMessage(invalidDummyResponse, messageProperties, null, null, muleContext);
		mes.setEncoding(encoding);
		return mes;
	}
	
	public Object getTestData_NullIdpEntityID_Exception() {
		String invalidDummyResponse = dummyResponse.replaceAll(testIdpEntityID, "");
		
		MuleMessage mes = new DefaultMuleMessage(invalidDummyResponse, messageProperties, null, null, muleContext);
		mes.setEncoding(encoding);
		return mes;
	}


	public Object getResultData() {
		MuleMessage mes = new DefaultMuleMessage(dummyResponse, messageProperties, null, null, muleContext);
		mes.setSessionProperty(CitkConstants.NAMEID_KEY, testNameID);
		mes.setSessionProperty(CitkConstants.IDP_ENTITYID_KEY, testIdpEntityID);
		mes.setOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY,200);
		return mes;
	}

	
	@Test
	public void testTransform() throws Exception {
		Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());
        Object expectedResult = this.getResultData();
        
		MuleMessage expectedMessage = (MuleMessage)expectedResult;
		MuleMessage resultMessage = (MuleMessage)result;
		
		String expectedNameID = expectedMessage.getSessionProperty(CitkConstants.NAMEID_KEY);
		String expectedIdpEntityID = expectedMessage.getSessionProperty(CitkConstants.IDP_ENTITYID_KEY);
		Integer expectedStatus = expectedMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY);
		
		String resultNameID = resultMessage.getSessionProperty(CitkConstants.NAMEID_KEY);
		String resultIdpEntityID = resultMessage.getSessionProperty(CitkConstants.IDP_ENTITYID_KEY);
		Integer resultStatus = resultMessage.getOutboundProperty(HttpConnector.HTTP_STATUS_PROPERTY);
				
		assertEquals(expectedNameID, resultNameID);
		
		assertEquals(expectedIdpEntityID, resultIdpEntityID);
		
		assertEquals(expectedStatus, resultStatus);
		
		NameIDCache cache = NameIDCache.getInstance();
		
		String value = cache.get(resultNameID);
		
		assertEquals(value, testNameIDValue);
		
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_NullPayloard_Exception() throws Exception {
		Transformer trans = this.getTransformer();
		
		//Exception発生
        Object result = trans.transform(getTestData_NullPayloard_Exception());
        
        fail("期待するExceptionが発生しませんでした。");
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_InvalidPayloard_Exception() throws Exception {
		Transformer trans = this.getTransformer();
		
		//Exception発生
        Object result = trans.transform(getTestData_InvalidPayloard_Exception());
        
        fail("期待するExceptionが発生しませんでした。");
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_NullNameID_Exception() throws Exception {
		Transformer trans = this.getTransformer();
		
		//Exception発生
        Object result = trans.transform(getTestData_NullNameID_Exception());
        
        fail("期待するExceptionが発生しませんでした。");
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_NullIdpEntityID_Exception() throws Exception {
		Transformer trans = this.getTransformer();
		
		//Exception発生
        Object result = trans.transform(getTestData_NullIdpEntityID_Exception());
        
        fail("期待するExceptionが発生しませんでした。");
	}

}
