/*
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.authz;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.DataUtil;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.log.exception.CitkSystemException;


import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.api.transport.PropertyScope;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

/**
 * Mule ESBの認可処理用トランスフォーマークラス実装。
 * @author ISP Shiraishi
 *
 */
public class AuthzTransformer extends AbstractMessageTransformer {

	/**
     * カスタムトランスフォーマーのコンストラクタ。
     */
    public AuthzTransformer() {
        super();
    }
	
    /**
     * ログメッセージ出力用オブジェクト。
     */
    private static final CitkLogger logger = CitkLogger.getLog(AuthzTransformer.class);

    /**
     * Fedletのレスポンスhtml内の認可結果開始タグ文字列。
     */
    private static final String POLICY_TAG = "<th>Policy Decision</th>";
    
    /**
     * Fedletへのリクエストパラメータ文字列（idpEntityID）。
     */
    private static final String FEDLET_IDPENTITYID_PARAM = "&idpEntityID=";
    
    /**
     * Fedletへのリクエストパラメータ文字列（resource）。
     */
    private static final String FEDLET_RESOURCE_PARAM = "&resource=";
    
    /**
     * Fedletへのリクエストパラメータ文字列（action）。
     */
    private static final String FEDLET_ACTION_PARAM = "&action=";
    
    /**
     * Fedletへのリクエストパラメータ文字列（nameIDValue）。
     */
    private static final String FEDLET_NAMEID_PARAM = "&nameIDValue=";
    
    /**
     * FedletへのリクエストURL
     */
    private String requestUrl = null;
    
    /**
     * Fedletへのリクエストaddress
     */
    private String address = null;
    
    /**
	 * @return requestUrl
	 */
	public String getRequestUrl() {
		return requestUrl;
	}

	/**
	 * @param requestUrl セットする requestUrl
	 */
	public void setRequestUrl(String requestUrl) {
		this.requestUrl = requestUrl;
	}

	/**
	 * @return address
	 */
	public String getAddress() {
		return address;
	}

	/**
	 * @param address セットする address
	 */
	public void setAddress(String address) {
		this.address = address;
	}


	@Override
    /**
     * 認可処理を行う。(Fedletを利用) 
     * @param message 変換対象のMuleMessageオブジェクト
     * @param outputEncoding エンコード
     * @return 変換後のMuleMessageオブジェクト
     * @throws TransformerException
     */
    public Object transformMessage(MuleMessage message, String outputEncoding)
            throws TransformerException {

    	String requestPath = message.getInboundProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY);
    	String resource = address +  requestPath;
    	String nameId = message.getSessionProperty(CitkConstants.NAMEID_KEY);  	
    	String encodedNameId = null;
		try {
			encodedNameId = URLEncoder.encode(nameId, "UTF-8");
		} catch (UnsupportedEncodingException e1) {
			throw new CitkSystemException(CitkConstants.ERROR_MSG_NAMEID_ENCODE_FAILUER, e1);
		}
		
    	String method = message.getInboundProperty(HttpConnector.HTTP_METHOD_PROPERTY);
    	String idpEntityId = DataUtil.getCookieValue(
                message.getInboundProperty(HttpConnector.HTTP_COOKIES_PROPERTY), CitkConstants.IDP_ENTITYID_KEY);
    	String url = requestUrl + 
    			FEDLET_IDPENTITYID_PARAM + idpEntityId + 
    			FEDLET_RESOURCE_PARAM + resource + 
    			FEDLET_ACTION_PARAM + method + 
    			FEDLET_NAMEID_PARAM + encodedNameId;
    	
    	logger.info(CitkConstants.LOG_MSG_LOGICAL_ADDRESS, resource);
    	logger.info(CitkConstants.LOG_MSG_NADEID_IS, nameId);
    	
    	String data = null;
    	try{
    		data = fedletAuthrization(url);
    	}
    	catch(Exception e){
    		throw new CitkSystemException(CitkConstants.ERROR_MSG_AUTHORIZE_FAILUER, e);
    	}
    	
    	if(data == null){
    		throw new CitkSystemException(CitkConstants.ERROR_MSG_AUTHORIZE_FAILUER);
    	}

        int sp = data.indexOf(POLICY_TAG);
        String tmp = null;
        try{
        	tmp = data.substring(sp);
        }
        catch(Exception e){
        	throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_POLICYDECISION_TAG, e);
        }

        if (tmp == null || tmp.isEmpty()) {
            throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_POLICYDECISION_TAG);
        } else{
            String decision = getPolicyDecision(tmp);
            
            if(decision == null){
            	throw new CitkSystemException(CitkConstants.ERROR_MSG_NOT_CONTAIN_POLICYDECISION);
            }

            logger.info(CitkConstants.LOG_MSG_AUTHZ_DECISION, decision);
            
            message.setProperty(CitkConstants.AUTHZ_RESULT_KEY, decision,
                    PropertyScope.OUTBOUND);
        }

        return message;
    }

	/**
     * Fedletに対し認可リクエストを行う。 
     * @param urlString Fedletの認可リクエストURL文字列
     * @return 認可結果
     * @throws IOException
     */
    private String fedletAuthrization(String urlString) throws IOException{
    	
    	String body = null;
    	URL url = new URL(urlString);

    	HttpURLConnection urlconn = (HttpURLConnection)url.openConnection();
    	urlconn.setRequestMethod(HttpConstants.METHOD_GET);
    	urlconn.setInstanceFollowRedirects(false);
    	
    	urlconn.connect();

    	int responseCode = urlconn.getResponseCode();
    	
    	if(responseCode != HttpConstants.SC_OK){
    		return null;
    	}
    	
    	BufferedReader reader =
    	    new BufferedReader(new InputStreamReader(urlconn.getInputStream()));

    	while (true){
    	    String line = reader.readLine();
    	    if ( line == null ){
    	        break;
    	    }
    	    body = body + line;
    	}
    	
    	reader.close();
    	
    	urlconn.disconnect();
    	
    	return body;
    }
    
    /**
     * Fedletの認可レスポンスから認可結果を抽出。 
     * @param data Fedletの認可レスポンス
     * @return 認可結果
     */
    private String getPolicyDecision(String data) {

        String decision = null;

        if (data.indexOf(CitkConstants.AUTHZ_PERMIT) != -1) {
            decision = CitkConstants.AUTHZ_PERMIT;

        } else if (data.indexOf(CitkConstants.AUTHZ_DENY) != -1) {
            decision = CitkConstants.AUTHZ_DENY;

        } else if (data.indexOf(CitkConstants.AUTHZ_INDETERMINATE) != -1) {
            decision = CitkConstants.AUTHZ_INDETERMINATE;
        }
        return decision;
    }
}
