/*
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.mule.transport.http.CookieHelper;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * CITK で使用するデータに関するユーティリティクラス。
 *
 * @author toyota
 *
 */
public final class DataUtil {
	
	/**
     * SAMLアサーション中のNameIDタグ名
     */
	private static final String NAMEID_TAG = "saml:NameID";
	
	/**
     * SAMLアサーション中のSubjectConfirmationDataタグ名
     */
	private static final String SUBJECT_CONFIRMATION_DATA_TAG = "saml:SubjectConfirmationData";
	
	/**
     * SAMLアサーション中のSubjectConfirmationタグ名
     */
	private static final String SUBJECT_CONFIRMATION_TAG = "saml:SubjectConfirmation";
	
	/**
     * SAMLアサーション中のSubjectタグ名
     */
	private static final String SUBJECT_TAG = "saml:Subject";
	
	/**
     * SAMLアサーション中のNotOnOrAfterパラメータ名
     */
	private static final String NOT_ON_OR_AFTER_NAME = "NotOnOrAfter";
	
	/**
     * 時刻フォーマット
     */
	private static final String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
	
    /**
     * MuleMessageの"cookies"または"Set-Cookie"プロパティから取得した
     * Cookie型配列オブジェクトから特定のCookieの値を取得する。
     * @see org.mule.transport.http.CookieHelper
     * @param cookies org.apache.commons.httpclient.Cookie[]型オブジェクト。
     * @param cookiename 取得するCookieの名前。
     * @return String Cookieの値
     */
    public static String getCookieValue(Object cookies, String cookiename) {
        if (cookies == null || cookiename == null || cookiename.isEmpty()) {
            return null;
        }
        return CookieHelper.getCookieValueFromCookies(cookies, cookiename);
    }

    /**
     * XML文字列をorg.w3c.dom.Document型オブジェクトに変換する。
     * XMLのバリデーションは行わないのでXSD等の定義は使用しない。
     * @param src XML文字列
     * @return XMLのDOM
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    public static Document parseDocument(String src)
            throws ParserConfigurationException, IOException, SAXException {
        if (src == null || src.isEmpty()) {
        	return null;
        }
        ByteArrayInputStream in = new ByteArrayInputStream(src.getBytes());
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(false);
        factory.setExpandEntityReferences(false);
        factory.setNamespaceAware(false);
        factory.setIgnoringComments(true);
        factory.setIgnoringElementContentWhitespace(true);
        DocumentBuilder parser;
        parser = factory.newDocumentBuilder();
        return parser.parse(in);
    }

    /**
     * SAML2のAuthnRequestメッセージに対する応答から、NameID要素の値を取得する。
     * @param document SAML2のResponseメッセージ。
     * @return NameIDの値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getNameId(Document document) throws Exception {
        String value = null;
        NodeList elms = document.getElementsByTagName(NAMEID_TAG);
        if (elms != null && elms.getLength() == 1) {
            if (elms.item(0) != null) {
                value = elms.item(0).getTextContent();
            }
        }
        if (value != null) {
            value = value.trim();
        }
        return value;
    }

    /**
     * @see #getNameId(Document)
     * @param src
     * @return NameIDの値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getNameId(String src) throws Exception {
        Document document = parseDocument(src);
        return getNameId(document);
    }

    /**
     * @see #getNameId(Document)
     * @param value
     * @return NameIDの値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getNameId(Object value) throws Exception {
        if (org.w3c.dom.Document.class.isInstance(value)) {
            return getNameId((Document) value);
        }
        else if (java.lang.String.class.isInstance(value)) {
            return getNameId((String) value);
        }
        return null;
    }

    /**
     * SAML2のAuthnRequestメッセージに対する応答から、Subject要素の
     * NotOnOrAfter属性の値を取得する。
     * @param document SAML2のResponseメッセージ。
     * @return NotOnOrAfter属性の値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getExpierDateTime(Document document)
                                            throws Exception {
        NodeList elms =
            document.getElementsByTagName(SUBJECT_CONFIRMATION_DATA_TAG);
        String value = null;
        Node node = null;
        for (int i = 0; i < elms.getLength(); i++) {
            node = elms.item(i).getParentNode();
            if (node == null) {
                continue;
            }
            if (node.getNodeName().equals(SUBJECT_CONFIRMATION_TAG)) {
                node = node.getParentNode();
                if (node == null) {
                    continue;
                }
                if (node.getNodeName().equals(SUBJECT_TAG)) {
                    node = elms.item(i).getAttributes()
                                       .getNamedItem(NOT_ON_OR_AFTER_NAME);
                    value = node.getNodeValue();
                }
            }
        }
        return value;
    }

    /**
     * @see #getExpierDateTime(Document)
     * @param src
     * @return NotOnOrAfter属性の値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getExpierDateTime(String src) throws Exception {
        Document document = parseDocument(src);
        return getExpierDateTime(document);
    }

    /**
     * @see #getExpierDateTime(Document)
     * @param value
     * @return NotOnOrAfter属性の値、またはnull(取得できなかった場合)
     * @throws Exception
     */
    public static String getExpierDateTime(Object value) throws Exception {
        if (org.w3c.dom.Document.class.isInstance(value)) {
            return getExpierDateTime((Document) value);
        }
        if (java.lang.String.class.isInstance(value)) {
            return getExpierDateTime((String) value);
        }
        return null;
    }

    /**
     * 特定の形式の日時文字列を解析してDateオブジェクトに変換する。
     * SAML2のResponseメッセージに含まれる日付を対象とする。
     * java.text.SimpleDateFormatの"yyyy-MM-dd'T'HH:mm:ssZ"形式の
     * 文字列を受け入れる。(例：2011-03-01T00:00:00+0900)
     * @param str ISO8601フォーマットの日時を示す文字列。
     * @return java.util.Date型オブジェクト
     * @throws ParseException 解析できない場合。
     */
    public static Date parseStringToDate(String str) throws ParseException {
        SimpleDateFormat fmt = new SimpleDateFormat(DATE_FORMAT);
        int len = "yyyy-MM-ddTHH:mm:ss".length();
        Date date = null;
        String data = null;
        String offset = null;

        if (str != null && str.length() > len) {
            data = str.substring(0, len);
            offset = str.substring(len).replace(":", "");
        }
        if ("Z".equals(offset)) {
            offset = "+0000";
        }

        if (data != null && offset != null) {
            date = fmt.parse(data + offset);
        }
        return date;
    }

}
