/*
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import jp.co.ogis_ri.citk.common.log.CitkLogger;


/**
 * キャッシュ用クラス。
 * 認証済みユーザのnameIDクッキーの値をキーとし、その保持と管理を行うクラス。
 * @author ISP Shiraishi
 *
 */
public class NameIDCache {
	
	 /**
     * コンストラクタ。クラスの外部からは利用しない。
     */
    protected NameIDCache() {
        cache = new ConcurrentHashMap<String, String>();
    }
	
    /**
     * ログメッセージ出力用オブジェクト。
     */
//    private static final CitkLogger logger = CitkLogger.getLog(NameIDCache.class);
    
    /**
     * このクラスのシングルトンオブジェクト。
     */
    private static NameIDCache me = new NameIDCache();
    
    /**
     * cacheオブジェクト。
     */
    private Map<String, String> cache;

    /**
     * cacheのsetter。
     * @param cache
     */
    public void setCache(Map<String, String> cache) {
        this.cache = null;
        this.cache = cache;
    }

    /**
     * cacheのgetter。
     * @return cacheオブジェクト
     */
    public Map<String, String> getCache() {
        return cache;
    }
      

    /**
     * 認証済みユーザのNameIDをキャッシュに追加する。
     * @param key 追加するユーザのnameIDクッキーの値
     * @param value 
     */
    public void add(String key, String value) {
        if (key == null || value == null) return;
        if (key.isEmpty()) return;
        
        if (!cache.containsKey(key)) {
            cache.put(key, value);
        }
        return;
    }

    /**
     * キャッシュ内のNameIDを削除する。
     * @param key 削除対象のnameIDクッキーの値
     */
    public void remove(String key) {
    	if (key == null ) return;
        if (cache.containsKey(key)) {
            cache.remove(key);
        }
    }

    /**
     * キャッシュ内のNameIDを更新する。
     * @param key 更新対象のnameIDクッキーの値
     * @param value 更新する値
     */
    public synchronized void update(String key, String value) {
        remove(key);
        add(key, value);
    }

    /**
     * キャッシュ内のNameIDを取得する。
     * @param key nameIDクッキーの値
     * @return 
     */
    public String get(String key) {
    	if (key == null ) return null;
    	String val = null;
        if (cache.containsKey(key)) {
            val = cache.get(key);
        }
        return val;    
    }

    /**
     * このオブジェクトのシングルトンインスタンスを取得する。
     * @return NameIDCacheオブジェクト
     */
    public static NameIDCache getInstance() {
        return me;
    }
}
