/*
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.common.transformer.saml2;

import java.net.URI;
import java.net.URISyntaxException;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.log.CitkLogger;
import jp.co.ogis_ri.citk.common.log.exception.CitkSystemException;

import org.mule.api.MuleMessage;
import org.mule.api.transformer.TransformerException;
import org.mule.transformer.AbstractMessageTransformer;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;


/**
 * Mule ESBのACS処理トランスフォーマークラス実装。
 * @author ISP Shiraishi
 *
 */
public class ReceiveRelayStateTransformer extends AbstractMessageTransformer {
	
	/**
     * カスタムトランスフォーマーのコンストラクタ。
     */
    public ReceiveRelayStateTransformer() {
        super();
    }
	
    /**
     * ログメッセージ出力用オブジェクト。
     */
//    private static final CitkLogger logger =
//    		CitkLogger.getLog(ReceiveRelayStateTransformer.class);
	
	/**
     * RelayStateパラメータのキー
     */
	public static final String KEY = CitkConstants.RELAYSTATE_PARAM + "=";

	@Override
	/**
     * AssertionConsumerServiceとして設けたMuleのエンドポイントに対し
     * 認証結果として送られてきたPOSTリクエストから
     * RelayState値を取り出しMuleMessageのセッションプロパティにセットする。
     * @param message 変換対象のMuleMessageオブジェクト
     * @param outputEncoding エンコード
     * @return 変換後のMuleMessageオブジェクト
     * @throws TransformerException
     */
	public Object transformMessage(MuleMessage message, String outputEncoding)
		throws TransformerException
	{

		// HTTP Inbound Endpoint Transformer
		String method = message.getInboundProperty(HttpConnector.HTTP_METHOD_PROPERTY);
		String queryString = null;
		String relayState = null;
		
		try {
			queryString = getQueryString(message, method);
		} catch (URISyntaxException e) {
			throw new CitkSystemException(CitkConstants.ERROR_MSG_GET_QUERY_FAILUER, e);
		}
		
		if (queryString == null)
		{
			throw new CitkSystemException(CitkConstants.ERROR_MSG_GET_QUERY_FAILUER);
		}
		
		if (queryString.contains(KEY)) 
		{
			int startPosition = queryString.indexOf("&" + KEY) + KEY.length() + 1;
			int endPosition = queryString.indexOf("&", startPosition);
			
			if (endPosition < 0)
			{
				// if RelayState is the last parameter.
				endPosition = queryString.length(); 
			} 
			
			if (startPosition < endPosition) 
			{
				relayState = queryString.substring(startPosition, endPosition);
			}
		}

		if(relayState == null){
			throw new CitkSystemException(CitkConstants.ERROR_MSG_RELAYSTATE_IS_NULL);
		}
		
		message.setSessionProperty(CitkConstants.CITK_RELAYSTATE_KEY, relayState);

		return message;
	}

	
	/**
     * MuleMessageからリクエストクエリを文字列として抽出
     * @param message 対象のMuleMessageオブジェクト
     * @param method リクエストメソッド
     * @return リクエストクエリ文字列
     * @throws TransformerException
	 * @throws URISyntaxException 
     */
	private String getQueryString(MuleMessage message, String method) 
			throws URISyntaxException, TransformerException 
	{
		String query = null;
		if (HttpConstants.METHOD_GET.equals(method)) {
			URI uri = new URI((String) message.getInboundProperty(HttpConnector.HTTP_REQUEST_PROPERTY));

            query = uri.getRawQuery();

		}
		else if (HttpConstants.METHOD_POST.equals(method))
		{
			query = message.getPayload(DataTypeFactory.STRING);
		}
		return query;
	}
}
