package jp.co.ogis_ri.citk.forwarding.transformer;

import static org.junit.Assert.*;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import jp.co.ogis_ri.citk.common.CitkConstants;
import jp.co.ogis_ri.citk.common.log.exception.CitkSystemException;
import jp.co.ogis_ri.citk.forwarding.transformer.ForwardingOutboundTransformer;
import jp.co.ogis_ri.citk.forwarding.transformer.OutboundEndpointResolver;
import jp.co.ogis_ri.citk.forwarding.transformer.impl.EndpointUrlBuilder;
import jp.co.ogis_ri.citk.forwarding.transformer.impl.MemoryOutboundEndpointResolverImpl;

import org.junit.Test;
import org.mule.DefaultMuleMessage;
import org.mule.RequestContext;
import org.mule.api.MuleMessage;
import org.mule.api.endpoint.EndpointBuilder;
import org.mule.api.endpoint.ImmutableEndpoint;
import org.mule.api.transformer.Transformer;
import org.mule.api.transport.PropertyScope;
import org.mule.endpoint.EndpointURIEndpointBuilder;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.transformer.types.DataTypeFactory;
import org.mule.transport.http.HttpConnector;
import org.mule.transport.http.HttpConstants;

public class ForwardingOutboundTransformerTestCase extends AbstractMuleContextTestCase
{

	private static String encoding = "UTF-8";
	private static String path = "/sp1/test/test.html";
	private static String invalidPath = "/sp1";
	private static String notCachedPath = "/sp2/test/test.html";
	
	private static final String PAYLOAD = "http body!";
	private static Map<String, Object>  messageProperties;
	private static Map<String, String>  settingDatas;
	
	private static String resultAddress = "sp1.citk.com:8080/test/test.html?testKey=testValue";
	private static String resultProtocol = "http";
	
	static
	{
		Map<String, Object> map = new HashMap<String, Object>();
		map.put(HttpConnector.HTTP_METHOD_PROPERTY, "GET");
		map.put(HttpConstants.HEADER_ACCEPT, "text/html, application/xhtml+xml, */*");
		map.put(HttpConstants.HEADER_ACCEPT_LANGUAGE, "ja,en-US;q=0.8,en;q=0.6");
		map.put(HttpConstants.HEADER_CONNECTION, "Keep-Alive");
		map.put(HttpConstants.HEADER_HOST,  "www.acme.com");
		map.put(HttpConstants.HEADER_COOKIE, "testCookie=value");
		map.put(HttpConnector.HTTP_CONTEXT_PATH_PROPERTY, "/");
		map.put(HttpConnector.HTTP_REQUEST_PROPERTY, path + "?testKey=testValue");
		map.put(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, path);
		map.put(HttpConnector.HTTP_VERSION_PROPERTY, HttpConstants.HTTP11);
		
		map.put(CitkConstants.AUTHZ_RESULT_KEY, CitkConstants.AUTHZ_PERMIT);
		
		messageProperties = Collections.unmodifiableMap(map);
		
		settingDatas = new HashMap<String, String>();
		settingDatas.put("sp1/test", "http://sp1.citk.com:8080/test/," +
							  "http://sp1.citk.com:8080/test/error.html," +
							  "http://sp1.citk.com:8080/test/error.html");
		
	}
	
	//setup 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    public void doSetUp() throws Exception
    {
        super.doSetUp();
        // トランスフォーマのテストをする際は、必ずRequestContextのイベントが設定されてなければならない。
        RequestContext.setEvent(getTestEvent("test"));
    }

    //tear down 時に呼ばれるメソッド。親クラスのテンプレートメソッドをオーバーライドした。
    @Override
    protected void doTearDown() throws Exception 
    {
        super.doTearDown();
        //RequestContext の値をnullに戻し、次のテストメソッドに備える。
        RequestContext.setEvent(null);
    }
	

	public Transformer getTransformer() throws Exception {
		Transformer trans = createObject(ForwardingOutboundTransformer.class);
        trans.setReturnDataType(DataTypeFactory.create(MuleMessage.class));

        EndpointBuilder builder = new EndpointURIEndpointBuilder("test://test", muleContext);
        builder.setEncoding(encoding);
        ImmutableEndpoint endpoint = muleContext.getEndpointFactory().getInboundEndpoint(
            builder);
        trans.setEndpoint(endpoint);
        
        EndpointUrlBuilder endpointUrlBuilder = new EndpointUrlBuilder();
        endpointUrlBuilder.setSettingDatas(settingDatas);
        OutboundEndpointResolver outboundEndpointResolver = new MemoryOutboundEndpointResolverImpl();
        ((MemoryOutboundEndpointResolverImpl)outboundEndpointResolver).setEndpointUrlBuilder(endpointUrlBuilder);
        
        ((ForwardingOutboundTransformer) trans).setOutboundEndpointResolver(outboundEndpointResolver);

        return trans;
	}

	public Object getTestData() {
		MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties, null, null, muleContext);
		return mes;
	}
	
	public Object getTestData_InvalidTargetKey_Exception() {
		MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties, null, null, muleContext);
		mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, invalidPath,
																	PropertyScope.INBOUND);
		mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, invalidPath + "?testKey=testValue",
																	PropertyScope.INBOUND);
		return mes;
	}
	
	public Object getTestData_NotCachedTargetKey_Exception() {
		MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties, null, null, muleContext);
		mes.setProperty(HttpConnector.HTTP_REQUEST_PATH_PROPERTY, notCachedPath,
																	PropertyScope.INBOUND);
		mes.setProperty(HttpConnector.HTTP_REQUEST_PROPERTY, notCachedPath + "?testKey=testValue",
																	PropertyScope.INBOUND);
		return mes;
	}
	

	public Object getResultData() {
		
		Map<String,Object> map = new HashMap<String, Object>();
		map.put(CitkConstants.OUTBOUND_ADDRESS, resultAddress);
		map.put(CitkConstants.OUTBOUND_PROTOCOL, resultProtocol);
		
		MuleMessage mes = new DefaultMuleMessage(PAYLOAD, messageProperties, map, null, muleContext);
		return mes;
	}
	
	
	@Test
	public void testTransform() throws Exception{
		Transformer trans = this.getTransformer();
        Object result = trans.transform(getTestData());
        Object expectedResult = this.getResultData();
		
		MuleMessage expectedMessage = (MuleMessage)expectedResult;
		MuleMessage resultMessage = (MuleMessage)result;
				
		Map<String, Object> expectedProps = new HashMap<String, Object>();
		Set<String> expectedNames = expectedMessage.getPropertyNames(PropertyScope.OUTBOUND);
			
		for(String expectedName : expectedNames)
		{
			expectedProps.put(expectedName, resultMessage.getOutboundProperty(expectedName));
		}

		Map<String, Object> resultProps = new HashMap<String, Object>();
		Set<String> resultNames = resultMessage.getPropertyNames(PropertyScope.OUTBOUND);
			
		for(String resultName : resultNames)
		{
			resultProps.put(resultName, resultMessage.getOutboundProperty(resultName));
		}
		
		assertEquals(expectedProps, resultProps);
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_InvalidTargetKey_Exception() throws Exception{
		Transformer trans = this.getTransformer();
		//Exception発生
        Object result = trans.transform(getTestData_InvalidTargetKey_Exception());
        fail("期待するExceptionが発生しませんでした。");
	}
	
	@Test(expected=CitkSystemException.class)
	public void testTransform_NotCachedTargetKey_Exception() throws Exception{
		Transformer trans = this.getTransformer();
		//Exception発生
        Object result = trans.transform(getTestData_NotCachedTargetKey_Exception());
        fail("期待するExceptionが発生しませんでした。");
	}
	
}
