/**
 * Copyright (C) 2011 OGIS-RI Co., Ltd. All rights reserved.
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package jp.co.ogis_ri.citk.wsp.cxf.jaxws.handler;

import java.util.HashMap;
import java.util.Map;

import javax.security.auth.Subject;
import javax.xml.soap.SOAPException;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.dom.DOMSource;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.binding.soap.SoapMessage;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.jaxws.handler.soap.SOAPMessageContextImpl;
import org.apache.cxf.staxutils.StaxUtils;

import com.sun.identity.wss.security.handler.SOAPRequestHandlerInterface;

/**
 * JAX-WS Handler(SOAPHandler) core component : SOAPHandlers have access to the entire SOAP
 * message.
 * 
 */
public abstract class AbstractWsHandler {

	private static Log log = LogFactory
			.getLog(AbstractWsHandler.class);

	private static final String WSS_SHARED_MAP = "WSS_SHARED_MAP";

	protected SOAPRequestHandlerInterface requestHandler;

	private String wspProfile;

	/**
	 * The handleMessage method gets invoked for normal processing by the
	 * framework as a message is either sent or received (inbound and outbound).
	 * 
	 * @param message
	 * @param cred
	 * @param subject
	 * @throws Fault
	 */
	public void handleMessage(SoapMessage message, ThreadLocal<Subject> cred,
			Subject subject) throws Fault {

		if (log.isInfoEnabled()) {
			log.info("WSSServerHandlerComponent: handleMessage()");
		}

		// get context from SoapMessage
		SOAPMessageContext context = new SOAPMessageContextImpl(message);

		// WSS SharedMap
		@SuppressWarnings("unchecked")
		Map<Object,Object> sharedMap = (Map<Object,Object>) context.get(WSS_SHARED_MAP);
		if ((sharedMap == null) || (sharedMap.isEmpty())) {
			sharedMap = new HashMap<Object,Object>();
			context.put(WSS_SHARED_MAP, sharedMap);
		}

		try {
			// set WSP Profile name
			Map<String, String> map = new HashMap<String, String>();
			map.put("providername", wspProfile);
			requestHandler.init(map);

		} catch (Exception ex) {
			if (log.isDebugEnabled()) {
				log.debug("WSSServerHandlerComponent.initialization failed : ",
						ex);
			}
			throw new Fault(ex);
		}

		handleSecureMessage(message, context, cred, subject, sharedMap);
	}

	/**
	 * The handleSecureMessage method validate or wrap in SOAP message.
	 * 
	 * @param message
	 * @param context
	 * @param cred
	 * @param subject
	 * @param sharedMap
	 */
	protected abstract void handleSecureMessage(SoapMessage message,
			SOAPMessageContext context, ThreadLocal<Subject> cred,
			Subject subject, Map<Object,Object> sharedMap);

	/**
	 * SOAP message in preparation for XMLStreamReader to get to read.
	 * 
	 * @param context
	 * @return
	 * @throws SOAPException
	 * @throws XMLStreamException
	 */
	protected XMLStreamReader getSOAPReader(SOAPMessageContext context)
			throws SOAPException, XMLStreamException {

		// prepare XMLReader for SOAP
		DOMSource source = new DOMSource(context.getMessage().getSOAPPart()
				.getEnvelope().getBody());
		XMLStreamReader soapReader = StaxUtils.createXMLStreamReader(source);

		// analyze SOAP-Body
		soapReader.nextTag();
		soapReader.nextTag();

		return soapReader;
	}

	public void setRequestHandler(SOAPRequestHandlerInterface requestHandler) {
		this.requestHandler = requestHandler;
	}

	public void setWspProfile(String wspProfile) {
		this.wspProfile = wspProfile;
	}

}