package jp.co.ogis_ri.citk.policytool;

import java.lang.Thread.UncaughtExceptionHandler;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import jp.co.ogis_ri.citk.policytool.common.CommonConstants;
import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.view.policylist.PolicyListView;

import org.apache.commons.logging.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.MessageSourceAccessor;

/**
 * アプリケーション起動クラス.
 */
public class AppLauncher {

	/**
	 * ロガー
	 */
	private final Log logger = LogWrapperFactory.getLog(AppLauncher.class);

	/**
	 * PolicyListView.
	 */
	@Autowired
    private PolicyListView policyListView = null;

	/**
	 * アプリケーションプロパティ.
	 */
	@Autowired
    private MessageSourceAccessor message = null;

    /**
     * main メソッド. アプリケーションのエントリポイント.
     * @param args アプリケーション起動時に指定される引数情報.
     */
    public static void main(String[] args) {
        AppLauncher app = ApplicationContextUtil.getBean(AppLauncher.class);
        app.run(args);
    }
    
    /**
     * アプリケーションを起動します.
     *
     * @param args アプリケーション起動時に指定される引数情報.
     * @param context アプリケーション コンテキスト情報.
     *
     */
    public void run(String[] args) {
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				// EDTに例外ハンドラーを設定し、RuntimeException を受け取る
				Thread.setDefaultUncaughtExceptionHandler(new UncaughtExceptionHandler() {
					public void uncaughtException(Thread t, Throwable e) {
						try {
							if(e instanceof ApplicationException) {
								JOptionPane.showMessageDialog(null,
										e.getMessage(), getMessageTitle((ApplicationException)e),
										getMessageType((ApplicationException)e));
								logger.error(e.getMessage(), e.getCause());
							}
							else {
								JOptionPane.showMessageDialog(null,
										e.getMessage(), message.getMessage(CommonConstants.DIALOG_TITLE_ERROR),
										JOptionPane.ERROR_MESSAGE);
								logger.error(e.getMessage(), e.getCause());
								System.exit(-1);
							}
						} catch (Exception ex) {
							logger.fatal(ex);
							System.exit(-1);
						}
					}
				});
		        policyListView.setVisible(true);
			}
		});
    }
    
    /**
     * 指定された ApplicationException に対する, 表示されるメッセージボックスのタイトル文字列を取得する.
     * @param e ApplicationException.
     * @return 指定された ApplicationException に対する, 表示されるメッセージボックスのタイトル文字列.
     */
    private String getMessageTitle(ApplicationException e) {
    	String code = e.getMessageInfo().getCode();
		if (code.startsWith("E")) {
			return message.getMessage(CommonConstants.DIALOG_TITLE_ERROR);
		} else if (code.startsWith("W")) {
			return message.getMessage(CommonConstants.DIALOG_TITLE_WARN);
		} else if (code.startsWith("I")) {
			return message.getMessage(CommonConstants.DIALOG_TITLE_INFO);
		} else if (code.startsWith("C")) {
			return message.getMessage(CommonConstants.DIALOG_TITLE_CONFIRM);
		} else {
			return null;
		}
    }
    
    /**
     * 指定された ApplicationException に対する, メッセージボックスのメッセエージの種類を取得する.
     * @param e ApplicationException.
     * @return 指定された ApplicationException に対する, メッセージボックスのメッセエージの種類.
     */
    private static int getMessageType(ApplicationException e) {
    	String code = e.getMessageInfo().getCode();
		if (code.startsWith("E")) {
			return JOptionPane.ERROR_MESSAGE;
		} else if (code.startsWith("W")) {
			return JOptionPane.WARNING_MESSAGE;
		} else if (code.startsWith("I")) {
			return JOptionPane.INFORMATION_MESSAGE;
		} else if (code.startsWith("C")) {
			return JOptionPane.QUESTION_MESSAGE;
		} else {
			return Integer.MIN_VALUE;
		}
    }
}
