package jp.co.ogis_ri.citk.policytool.common.api.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

import jp.co.ogis_ri.citk.policytool.common.CommonConstants;
import jp.co.ogis_ri.citk.policytool.common.api.OpenAMAccess;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm.SsoadmClient;
import jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm.SsoadmResult;
import jp.co.ogis_ri.citk.policytool.common.exception.OpenAMAccessException;
import jp.co.ogis_ri.citk.policytool.common.http.HttpClientWrapper;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.StreamUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;

import org.apache.commons.logging.Log;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

/**
 * OpenAMアクセスAPI実装.
 */
@Component
public class OpenAMAccessImpl implements OpenAMAccess {
	/**
	 * ロガー.
	 */
	private final Log operationLogger = LogWrapperFactory.getOperationLog();
	
	/**
	 * アプリケーションプロパティ.
	 */
	@Autowired(required=false)
	@Qualifier(value="policytoolProperty")
    private Properties props = null;

    /**
     * コンストラクタ.
     */
	public OpenAMAccessImpl() {
    }

    /**
     * コンストラクタ.
     * 
     * @param props アプリケーションプロパティ.
     * 
     */
	public OpenAMAccessImpl(Properties props) {
        this.props = props;
    }

    @Override
    public List<Realm> getRealms(String realmName, String filter) { 
    	operationLogger.info("getRealms " + realmName + ", " + filter);
    	
        List<Realm> realms = null;

        RealmsBuilder builder = new RealmsBuilder();

        HttpClientWrapper httpClient = new HttpClientWrapper();
        SsoadmClient ssoadmClient = createSsoadmClient(httpClient);

        try {
            // 全てのレルムを追加
            SsoadmResult realmResult =  ssoadmClient.getRealms(realmName, filter, false);
            if(realmResult.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
    			throw new OpenAMAccessException(realmResult.getExitCode(), "I-0004", realmResult.getExitCode());
            }
            builder.addRealms(realmResult.getData());

            // レルムごとのグループを追加
            for(String name : realmResult.getData()) {
                SsoadmResult groupResult = ssoadmClient.getGroup(name, "*");
                if(groupResult.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
        			throw new OpenAMAccessException(groupResult.getExitCode(), "I-0004", groupResult.getExitCode());
                }
                builder.addGroups(name, groupResult.getData());
            }

            // 参照ポリシーを追加(参照ポリシーは、ルートに存在するため固定)
            SsoadmResult policiesResult = ssoadmClient.getPolicies("/", "*");
            if(policiesResult.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
    			throw new OpenAMAccessException(policiesResult.getExitCode(), "I-0004", policiesResult.getExitCode());
            }
            OpenAMPolicies amPolicies = toOpenAMPolicies(policiesResult.getData().get(0).getBytes());
            builder.addPolicies(amPolicies);
            
            realms = builder.build();
        } finally {
            httpClient.shutdown();
        }
        
        operationLogger.info(realms.toString());        
        return realms;
    }


    @Override
    public List<Policy> getPolicies(String realmName, String policyName) {
    	operationLogger.info("getPolicies " + realmName + ", " + policyName);
    	
        List<Policy> policies = null;
        PoliciesBuilder builder = new PoliciesBuilder();

        HttpClientWrapper httpClient = new HttpClientWrapper();
        SsoadmClient ssoadmClient = createSsoadmClient(httpClient);
        try {
            SsoadmResult result = ssoadmClient.getPolicies(realmName, policyName);
            if(result.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
    			throw new OpenAMAccessException(result.getExitCode(), "I-0004", result.getExitCode());
            }
            OpenAMPolicies amPolicies = toOpenAMPolicies(result.getData().get(0).getBytes());
            builder.add(realmName, amPolicies);
            policies = builder.build();
        } finally {
            httpClient.shutdown();
        }
        
        for (Policy policy : policies) {
        	operationLogger.info(policy.toString());
        }
        return policies;
    }

    @Override
    public void createPolicy(Policy policy) {
    	operationLogger.info("createPolicy " + policy.toString());
    	
        HttpClientWrapper httpClient = new HttpClientWrapper();
        SsoadmClient ssoadmClient = createSsoadmClient(httpClient);
        try {
            String xml = toXml(policy);
            SsoadmResult result =  ssoadmClient.createPolicies(policy.getRealmName(), xml);
            if(result.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
    			throw new OpenAMAccessException(result.getExitCode(), "I-0004", result.getExitCode());
            }
        } finally {
            httpClient.shutdown();
        }
    }

    @Override
    public void updatePolicy(Policy policy) {
    	operationLogger.info("updatePolicy " + policy.toString());
    	
        deletePolicy(policy);
        createPolicy(policy);
    }

    @Override
    public void deletePolicy(Policy policy) {
    	operationLogger.info("deletePolicy " + policy.toString());
    	
        HttpClientWrapper httpClient = new HttpClientWrapper();
        SsoadmClient ssoadmClient = createSsoadmClient(httpClient);
        try {
            SsoadmResult result =  ssoadmClient.deletePolicy(policy.getRealmName(), policy.getPolicyName());
            if(result.getExitCode() != SsoadmResult.EXITCODE_SUCCESS) {
    			throw new OpenAMAccessException(result.getExitCode(), "I-0004", result.getExitCode());
            }
        } finally {
            httpClient.shutdown();
        }
    }
    
    /**
     * ssoadmクライアントを作成する.
     * 
     * @param httpClient 利用するHTTPクライアント.
     * @return ssoadmクライアント.
     */
    private SsoadmClient createSsoadmClient(HttpClientWrapper httpClient) {
        httpClient.setEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
        return new SsoadmClient(
                httpClient,
                props.getProperty(CommonConstants.PROP_KEY_OPENAM_SSOADM_URL),
                props.getProperty(CommonConstants.PROP_KEY_OPENAM_USERID),
                props.getProperty(CommonConstants.PROP_KEY_OPENAM_PASSWORD));        
    }
        
    /**
     * XMLからOpen AMのモデルに変換する.
     * 
     * @param xml XML.
     * @return Open AMのモデル.
     * 
     */
    private OpenAMPolicies toOpenAMPolicies(byte[] xml) {
        OpenAMPolicies  amPolicies = null;
        InputStream is = null;
        try {
            is = new ByteArrayInputStream(xml);
            OpenAMModelContext.setJaxbEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
            amPolicies = OpenAMModelContext.unmarshaller(OpenAMPolicies.class, is);
        } finally {
            if(is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
        return amPolicies;
    }

    /**
     * Policyモデルの内容をXMLに変換する.
     * 
     * @param policy Policyモデル .
     * @return　XML.
     * 
     */
    private String toXml(Policy policy) {
        String xml = null;
        ByteArrayOutputStream os= null;
        
        OpenAMPoliciesBuilder builder = new OpenAMPoliciesBuilder();
        // ドメインモデルからOpen AMのモデルを構築する
        builder.addPolicy(policy);
        OpenAMPolicies amPolicies = builder.build();
        try {
            // Open AMのモデルからXMLへマーシャリングする
            os= new ByteArrayOutputStream();
            OpenAMModelContext.setJaxbEncoding(props.getProperty(CommonConstants.PROP_KEY_OPENAM_ENCODING));
            OpenAMModelContext.marshaller(amPolicies, os);
            xml = StreamUtil.toString(new ByteArrayInputStream(os.toByteArray()));
        } finally {
            if(os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    // NOP
                }
            }
        }
        return xml;
    }    
}
