package jp.co.ogis_ri.citk.policytool.common.api.impl;

import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttribute;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMServiceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubject;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubjects;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;

/**
 * 非公開クラス
 */
class OpenAMPoliciesBuilder {

    /**
     * Policyのリスト.
     */
    private List<Policy> policyList = new ArrayList<Policy>();

    /**
     * デフォルトコンストラクタ.
     * 
     */
    public OpenAMPoliciesBuilder() {
        super();
    }

    /**
     * ポリシーを追加する.
     * 
     * @param policy ポリシー
     * 
     */
    public void addPolicy(Policy policy) {
        policyList.add(policy);
    }

    /**
     * ポリシーモデルの リストからOpenAMPolicies を作成する.<br>
     * OpenAMSubjects　には　OpenAMSubject　は常に1個のみ作成する.
     * 
     * @return OpenAMPolicies.
     * 
     */
    public OpenAMPolicies build() {

        OpenAMPolicies openAMPolicies = new OpenAMPolicies();

        for (Policy policy : policyList) {
            OpenAMPolicy openAMPolicy = createPolicy(policy);
            openAMPolicies.getPolicy().add(openAMPolicy);
        }
        return openAMPolicies;

    }

    /**
     *  ポリシーモデルから OpenAMPolicy を作成する.
     * 
     * @param policy ポリシーモデル.
     * @return OpenAMPolicy.
     * 
     */
    private OpenAMPolicy createPolicy(Policy policy) {

        OpenAMPolicy openAMPolicy = new OpenAMPolicy();
        openAMPolicy.setName(policy.getPolicyName());
        openAMPolicy.setDescription("");
        openAMPolicy.setReferralPolicy(false);
        openAMPolicy.setActive(true);

        // ルール
        for(Resource resource : policy.getResources()) {
            openAMPolicy.getRule().add(createRule(resource));
        }
        
        // サブジェクト
        // ポリシー定義ツールでは、サブジェクトは１つのみ、作成する仕様となっているため、OpenAMに対しても複数のサブジェクトは作成しない。
        openAMPolicy.setSubjects(createSubjects(policy.getSubjects()));
        return openAMPolicy;
    }

    /**
     * リソースモデル から OpenAMRule を作成する.
     * 
     * @param resource リソースモデル.
     * @return OpenAMRule.
     * 
     */
    private OpenAMRule createRule(Resource resource) {
        // <Rule>
        OpenAMRule ruleBean = new OpenAMRule();
        ruleBean.setName(OpenAMNamingUtil.generateRuleName());

        // <ServiceName>
        OpenAMServiceName serviceNameBean = new OpenAMServiceName();
        serviceNameBean.setName(OpenAMNamingUtil.SERVICE_NAME);
        ruleBean.setServiceName(serviceNameBean);

        // <ResourceName>
        OpenAMResourceName resourceNameBean = new OpenAMResourceName();
        resourceNameBean.setName(resource.getResourceUrl());
        ruleBean.setResourceName(resourceNameBean);

        Permit postPermit = resource.getPostPermit();

        if (postPermit != Permit.INDETERMINATE) {
            // POST <AttributeValuePair>
            OpenAMAttributeValuePair postAttributeValuePairBean = new OpenAMAttributeValuePair();
            ruleBean.getAttributeValuePair().add(postAttributeValuePairBean);

            // POST <Attribute>
            OpenAMAttribute postAttributeBean = new OpenAMAttribute();
            postAttributeBean.setName(OpenAMNamingUtil.RULE_ATTR_POSTMETHOD_NAME);
            postAttributeValuePairBean.setAttribute(postAttributeBean);

            // POST <Value>
            String postPermitString = postPermit.getInternalValue();
            postAttributeValuePairBean.getValue().add(postPermitString);
        }

        Permit getPermit = resource.getGetPermit();
        if (getPermit != Permit.INDETERMINATE) {
            // GET <AttributeValuePair>
            OpenAMAttributeValuePair getAttributeValuePairBean = new OpenAMAttributeValuePair();
            ruleBean.getAttributeValuePair().add(getAttributeValuePairBean);

            // GET <Attribute>
            OpenAMAttribute getAttributeBean = new OpenAMAttribute();
            getAttributeBean.setName(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME);
            getAttributeValuePairBean.setAttribute(getAttributeBean);

            // GET <Value>
            String getPermitString = getPermit.getInternalValue();
            getAttributeValuePairBean.getValue().add(getPermitString);
        }
        return ruleBean;
    }

    /**
     *  サブジェクトモデルのリストからOpenAMSubjects を作成する.
     * 
     * @param subjectList サブジェクトモデルのリスト.
     * @return OpenAMSubjects.
     * 
     */
    private OpenAMSubjects createSubjects(List<Subject> subjectList) {
        OpenAMSubjects subjectsBean = new OpenAMSubjects();
        subjectsBean.setName(OpenAMNamingUtil.generateSubjectsName());

        OpenAMSubject subjectBean = createSubjectBySubjectList(subjectList);
        if (subjectBean != null) {
            subjectsBean.getSubject().add(subjectBean);
        }

        return subjectsBean;
    }

    /**
     * OpenAMSubject を作成する.
     * 
     * @param subjectList サブジェクトモデルのリスト.
     * @return OpenAMSubject.
     * 
     */
    private OpenAMSubject createSubjectBySubjectList(List<Subject> subjectList) {
        // <Subject>
        OpenAMSubject subjectBean = new OpenAMSubject();

        if (subjectList.size() != 0) {
            subjectBean.setName(OpenAMNamingUtil.generateSubjectName());
            subjectBean.setType(OpenAMNamingUtil.SUBJECT_TYPE);
            subjectBean.setIncludeType(OpenAMNamingUtil.SUBJECT_INCLUDE_TYPE);

            // <AttributeValuePair>
            OpenAMAttributeValuePair attributeValuePairBean = new OpenAMAttributeValuePair();
            subjectBean.getAttributeValuePair().add(attributeValuePairBean);

            // <Attribute>
            OpenAMAttribute attributeBean = new OpenAMAttribute();
            attributeBean.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
            attributeValuePairBean.setAttribute(attributeBean);

            // <Value>
            for (Subject subject : subjectList) {
                String valueText = subject.getSubjectCode();
                attributeValuePairBean.getValue().add(valueText);
            }
        }
        return subjectBean;
    }
}
