package jp.co.ogis_ri.citk.policytool.common.api.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;

/**
 * 非公開クラス
 */
class PoliciesBuilder {

    /** ﾚﾙﾑに属するポリシーを管理する */
    private Map<String, OpenAMPolicies> openAMpolicies = new HashMap<String, OpenAMPolicies>();
    
    /**
     * コンストラクタ.
     */
    public PoliciesBuilder() {
    }
    
    /**
     * ﾚﾙﾑに属するポリシーを追加する.
     *
     * @param realmName ﾚﾙﾑ名.
     * @param openAMpolicies OpenAMから返却されたポリシー.
     *
     */
    public void add(String realmName, OpenAMPolicies openAMpolicies) {
        this.openAMpolicies.put(realmName, openAMpolicies);
    }

    /**
     * ポリシーを構築する.
     *
     * @return　構築したポリシーのList.
     *
     */
    public List<Policy> build() {
        List<Policy> policies = new ArrayList<Policy>();
        for (String realmName : this.openAMpolicies.keySet()) {
            policies.addAll(buildPolicies(realmName,
                    this.openAMpolicies.get(realmName)));
        }
        return policies;
    }

    /**
     * ﾚﾙﾑに属するポリシーを構築する.
     *
     * @param realmName ﾚﾙﾑ名.
     * @param openAMpolicies　OpenAMから返却されたポリシー.
     * 
     * @return　構築したポリシーのList.
     *
     */
    private List<Policy> buildPolicies(String realmName,
            OpenAMPolicies openAMpolicies) {
        List<Policy> policies = new ArrayList<Policy>();
        for (OpenAMPolicy openAMPolicy : openAMpolicies.getPolicy()) {
            if (!openAMPolicy.isReferralPolicy()) {
            	Policy policy = buildPolicy(realmName, openAMPolicy);
            	if(policy.getResources().size() == 0 || policy.getSubjects().size() == 0) {
            		// ログ
                    //  読み飛ばす(policy.toString()をWarnLogに吐いて、continue)
                	Log logger = LogWrapperFactory.getLog(PoliciesBuilder.class);
                	logger.warn(ApplicationContextUtil.getResources().getMessage("common.policyBuilder.policyNoResOrNoSubjMsg", new Object[] { policy.toString() }));
            		continue;
            	}
                policies.add(policy);
            }
        }
        return policies;
    }

    /**
     * ﾚﾙﾑに属するポリシーを構築する.
     * ポリシーに属するリソース(ルール)、およびサブジェクトを含んでいる。
     *
     * @param realmName ﾚﾙﾑ名.
     * @param openAMpolicies　OpenAMから返却されたポリシー.
     * 
     * @return　構築したポリシー.
     *
     */
    private Policy buildPolicy(String realmName, OpenAMPolicy openAMPolicy) {

        // Policy生成
        Policy policy = new Policy(realmName, openAMPolicy.getName());
        // リソース設定
        for (OpenAMRule openAMRule : openAMPolicy.getRule()) {
            String url = openAMRule.getResourceName().getName();
            Permit get = Permit.INDETERMINATE;
            Permit post = Permit.INDETERMINATE;
            for (OpenAMAttributeValuePair attr : openAMRule
                    .getAttributeValuePair()) {
                if (attr.getAttribute().getName().equalsIgnoreCase(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME)) {
                    get = Permit.internalValueOf(attr.getValue().get(0));
                }
                else if (attr.getAttribute().getName().equalsIgnoreCase(OpenAMNamingUtil.RULE_ATTR_POSTMETHOD_NAME)) {
                    post = Permit.internalValueOf(attr.getValue().get(0));
                }
            }
            policy.addResource(url, get, post);
        }

        // サブジェクト設定
        for (String value : openAMPolicy.getSubjects().getSubject().get(0)
                .getAttributeValuePair().get(0).getValue()) {
            Map<String, String> kv = OpenAMNamingUtil.splitValue(value);
            policy.addSubject(kv.get(OpenAMNamingUtil.SUBJECT_ID_KEY), value);
        }
        return policy;
    }
}
