package jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 非公開クラス
 */
abstract class AbstractSsoadmResultParser {
	/**
	 * &lt;pre&gt; 開始タグ
	 */
	private final String HTML_PRE_PREFIX = "<pre>";
	
	/**
	 * &lt;pre&gt; 終了タグ
	 */
	private final String HTML_PRE_SUFFIX = "</pre>";

	/**
	 * 終了コード文字列の接頭辞.
	 */
	private final String EXITCODE_PREFIX = "<!-- CLI Exit Code:";
	
	/**
	 * 終了コード文字列の接尾辞.
	 */
	private final String EXITCODE_SUFFIX = "-->";

	/**
	 * ssoadmの結果の文字列をパースし, SsoadmResult を生成.
	 * @param s ssoadmの結果の文字列.
	 * @return SsoadmResult.
	 */
	public SsoadmResult parse(String s) {
		String preString = parsePre(s);
		return new SsoadmResult(parseData(preString), parseExitCode(preString));
	}
	
	/**
	 * ssoadmの結果のデータの文字列をパースし, 文字列のリストに格納.
	 * @param s ssoadmの結果のデータの文字列.
	 * @return 文字列のリスト.
	 */
	protected abstract List<String> parseData(String s);

	/**
	 * 文字列から, 指定した接頭辞と接尾辞の間の文字列を切り出す.
	 * @param prefix 接頭辞.
	 * @param suffix 接尾辞.
	 * @param s 切り出し元の文字列.
	 * @param include 切り出した結果文字列に, 接頭辞, 接尾辞を含むかを示す値. true のとき含む.
	 * @return 文字列から切り出した文字列.
	 */
	protected String cutText(String prefix, String suffix, String s, boolean include){
		String text = null;
        Matcher m = Pattern.compile(prefix+ ".*" + suffix, Pattern.DOTALL).matcher(s);
		if(m.find()){
			text =  include ?  s.substring(m.start(), m.end()) : s.substring(m.start() + prefix.length(), m.end() - suffix.length());
		}
		return text != null ? text : "";
	}

	/**
	 * 文字列から, &lt;pre&gt;～&lt;/pre&gt; の中の文字列を切り出して返す.
	 * @param s 切り出し元の文字列.
	 * @return &lt;pre&gt;～&lt;/pre&gt; の中の文字列.
	 */
	private String parsePre(String s) {
		String text = cutText(HTML_PRE_PREFIX, HTML_PRE_SUFFIX, s, false);
		return text;
	}

	/**
	 * 文字列から, 終了コードの数値を切り出して返す.
	 * @param s 切り出し元の文字列.
	 * @return 終了コードの数値.
	 */
	private int parseExitCode(String s) {
		String text = cutText(EXITCODE_PREFIX, EXITCODE_SUFFIX, s, false);
		return Integer.valueOf(text.trim());
	}	
	
}
