package jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm;

import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationSettingException;
import jp.co.ogis_ri.citk.policytool.common.http.HttpClientWrapper;
import jp.co.ogis_ri.citk.policytool.common.http.HttpResponseWrapper;
import jp.co.ogis_ri.citk.policytool.common.util.StringUtil;

import org.apache.http.Header;
import org.apache.http.HttpStatus;

/**
 * ssoadmクライアント.
 * 
 */
public class SsoadmClient {

    /**
     * Location(HTTPヘッダ)
     */
    private static final String HEADER_LOCATION = "Location";
    
    /**
     * Open AM Login画面のユーザーID
     */
    private static final String USERID_PARAM_NAME = "IDToken1";
    
    /**
     * Open AM Login画面のパスワード
     */
    private static final String PASSWORD_PARAM_NAME = "IDToken2";

    /**
     * ssoadm list-realms コマンド用 パラメータ.
     */
    private static final String LIST_REALMS_PARAM = "cmd=list-realms&submit=";

    /**
     * ssoadm list-idntities コマンド用 パラメータ.
     */
    private static final String LIST_IDENTITIES_PARAM = "cmd=list-identities&submit=";

    /**
     * ssoadm list-policies コマンド用 パラメータ.
     */
    private static final String LIST_POLICIES_PARAM = "cmd=list-policies&submit=";

    /**
     * ssoadm create-policies コマンド用 パラメータ.
     */
    private static final String CREATE_POLICIES_PARAM = "cmd=create-policies&submit=";

    /**
     * ssoadm delete-policies コマンド用 パラメータ.
     */
    private static final String DELETE_POLICIES_PARAM = "cmd=delete-policies&submit=";

    /**
     * パラメータ名 realm
     */
    private static final String REALM_PARAM_NAME = "realm";

    /**
     * パラメータ名 filter
     */
    private static final String FILTER_PARAM_NAME = "filter";

    /**
     * パラメータ名 recursive
     */
    private static final String RECURSIVE_PARAM_NAME = "recursive";

    /**
     * パラメータ名 idtype
     */
    private static final String IDTYPE_PARAM_NAME = "idtype";

    /**
     * パラメータ名 policynames
     */
    private static final String POLICYNAMES_PARAM_NAME = "policynames";

    /**
     * パラメータ名 xmlfile
     */
    private static final String XMLFILE_PARAM_NAME = "xmlfile";

    /**
     * idtype名 group
     */
    private static final String GROUP_IDTYPE_NAME = "group";

    /**
     * HTTP1クライアント.
     */
    private HttpClientWrapper httpClient = null;
    
    /**
     * ssoadm.jspのURL. 例 "http://idp.example.com:8080/opensso/ssoadm.jsp"
     */
    private String urlString = null;
    
    /**
     * ユーザーID.
     */
    private String userId = null;

    /**
     * パスワード.
     */
    private String password = null;

    /**
     * エンコーディング.
     */
    private String encoding = StringUtil.CHARSET_UTF8;

    /**
     * SSOADMクライアント内部で利用するHTTPクライアントを取得する.
     * 
     * @return HTTPクライアント.
     * 
     */
    public HttpClientWrapper getHttpClient() {
        return httpClient;
    }

    /**
     * SSOADMクライアント内部で利用するHTTPクライアントを設定する.
     * 
     * @param httpClient　HTTPクライアント.
     * 
     */
    public void setHttpClient(HttpClientWrapper httpClient) {
        this.httpClient = httpClient;
    }

    /**
     * アクセスするssoadmのURL文字列を取得する.
     *  
     * @return アクセスするssoadmのURL文字列.
     * 
     */
    public String getUrlString() {
        return urlString;
    }

    /**
     *  アクセスするssoadmのURL文字列を設定する.
     *  
     * @param urlString アクセスするssoadmのURL文字列.
     * 
     */
    public void setUrlString(String urlString) {
        this.urlString = urlString;
    }

    /**
     * ssoadmにアクセスする際のユーザーIDを取得する.
     * 
     * @return ssoadmにアクセスする際のユーザーID.
     * 
     */
    public String getUserId() {
        return userId;
    }

    /**
     * ssoadmにアクセスする際のユーザーIDを設定する.
     * 
     * @param userId ssoadmにアクセスする際のユーザーID.
     * 
     */
    public void setUserId(String userId) {
        this.userId = userId;
    }

    /**
     * ssoadmにアクセスする際のユーザーパスワードを取得する.
     * 
     * @return ssoadmにアクセスする際のユーザーパスワード.
     * 
     */
    public String getPassword() {
        return password;
    }

    /**
     * ssoadmにアクセスする際のユーザーパスワードを設定する.
     * 
     * @param password ssoadmにアクセスする際のユーザーパスワード.
     * 
     */
    public void setPassword(String password) {
        this.password = password;
    }

    /**
     * ssoadmにアクセスする際のエンコードを取得する.
     * 
     * @return ssoadmにアクセスする際のエンコード.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * ssoadmにアクセスする際のエンコードを設定する.
     * 
     * @param encoding ssoadmにアクセスする際のエンコード.
     */
    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    /**
     * コンストラクタ.
     * 
     */
    public SsoadmClient() {
    }

    /**
     * コンストラクタ.
     * 
     * @param httpClient HTTP Client.
     * @param urlString ssoadm.jspのURL文字列.
     */
    public SsoadmClient(HttpClientWrapper httpClient, String urlString,
            String userId, String password) {
        this.httpClient = httpClient;
        this.urlString = urlString;
        this.userId = userId;
        this.password = password;
    }

    /**
     * レルムを取得する.
     * 
     * @param realmName レルム名.
     * @param filter フィルター.
     * @param recursive Trueの場合、再帰的に取得する.
     * @return 実行結果.
     * 
     */
    public SsoadmResult getRealms(String realmName, String filter, boolean recursive) {
        // ログイン
        login();

        // フォーム編集
        httpClient.addQueryParam(REALM_PARAM_NAME, realmName);
        httpClient.addQueryParam(FILTER_PARAM_NAME, filter);
        if(recursive) {
	        httpClient.addQueryParam(RECURSIVE_PARAM_NAME,
	                Boolean.toString(recursive));
        }
        // POST リクエスト
        HttpResponseWrapper httpResponse = httpClient.post(urlString + "?"
                + LIST_REALMS_PARAM);
        if(httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
        	throw new ApplicationSettingException("I-0002");
        }

        // 実行結果パース
        String content = StringUtil.encoding(httpResponse.getContent(), encoding);
        AbstractSsoadmResultParser parser = new RealmSsoadmResultParser();
        SsoadmResult result = parser.parse(content);
        return result;
    }

    /**
     * グループを取得する.
     * 
     * @param realmName レルム名.
     * @param filter フィルター.
     * @return 実行結果.
     * 
     */
    public SsoadmResult getGroup(String realmName, String filter) {
        // ログイン
        login();

        // フォーム編集
        httpClient.addQueryParam(REALM_PARAM_NAME, realmName);
        httpClient.addQueryParam(FILTER_PARAM_NAME, filter);
        httpClient.addQueryParam(IDTYPE_PARAM_NAME, GROUP_IDTYPE_NAME);

        // POST リクエスト
        HttpResponseWrapper httpResponse = httpClient.post(urlString + "?"
                + LIST_IDENTITIES_PARAM);
        if(httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
        	throw new ApplicationSettingException("I-0002");
        }

        // 実行結果パース
        String content = StringUtil.encoding(httpResponse.getContent(), encoding);
        AbstractSsoadmResultParser parser = new GroupSsoadmResultParser();
        SsoadmResult result = parser.parse(content);
        return result;
    }

    /**
     * ポリシー定義を取得する.
     * 
     * @param realmName レルム名.
     * @param policyNames ポリシー名.
     * @return 実行結果.
     * 
     */
    public SsoadmResult getPolicies(String realmName, String policyNames) {
        // ログイン
        login();

        // フォーム編集
        httpClient.addQueryParam(REALM_PARAM_NAME, realmName);
        httpClient.addQueryParam(POLICYNAMES_PARAM_NAME, policyNames);

        // POST リクエスト
        HttpResponseWrapper httpResponse = httpClient.post(urlString + "?"
                + LIST_POLICIES_PARAM);
        if(httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
        	throw new ApplicationSettingException("I-0002");
        }

        // 実行結果パース
        String content = StringUtil.encoding(httpResponse.getContent(), encoding);
        AbstractSsoadmResultParser parser = new PoliciesSsoadmResultParser();
        SsoadmResult result = parser.parse(content);
        return result;
    }

    /**
     * ポリシー定義を作成する.
     * 
     * @param realmName レルム名.
     * @param xml XML.
     * @return 実行結果.
     * 
     */
    public SsoadmResult createPolicies(String realmName, String xml) {
        // ログイン
        login();

        // フォーム編集
        httpClient.addQueryParam(REALM_PARAM_NAME, realmName);
        httpClient.addQueryParam(XMLFILE_PARAM_NAME, xml);

        // POST リクエスト
        HttpResponseWrapper httpResponse = httpClient.post(urlString + "?"
                + CREATE_POLICIES_PARAM);
        if(httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
        	throw new ApplicationSettingException("I-0002");
        }

        // 実行結果パース
        String content = StringUtil.encoding(httpResponse.getContent(), encoding);
        AbstractSsoadmResultParser parser = new PoliciesSsoadmResultParser();
        SsoadmResult result = parser.parse(content);
        return result;
    }

    /**
     * ポリシー定義を削除する.
     * 
     * @param realmName レルム名.
     * @param policyName ポリシー名.
     * @return 実行結果.
     * 
     */
    public SsoadmResult deletePolicy(String realmName, String policyName) {
        // ログイン
        login();

        // フォーム編集
        httpClient.addQueryParam(REALM_PARAM_NAME, realmName);
        httpClient.addQueryParam(POLICYNAMES_PARAM_NAME, policyName);

        // POST リクエスト
        HttpResponseWrapper httpResponse = httpClient.post(urlString + "?"
                + DELETE_POLICIES_PARAM);
        if(httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
        	throw new ApplicationSettingException("I-0002");
        }

        // 実行結果パース
        String content = StringUtil.encoding(httpResponse.getContent(), encoding);
        AbstractSsoadmResultParser parser = new PoliciesSsoadmResultParser();
        SsoadmResult result = parser.parse(content);
        return result;
    }

    /**
     * ログインを行う.
     *
     * @return 実行結果。
     *
     */
    private HttpResponseWrapper login() {
        HttpResponseWrapper httpResponse = httpClient.post(urlString);
        // ステータスコードが200でない場合は、ログインしていないと判断する
        if (httpResponse.getHttpStatusCode() != HttpStatus.SC_OK) {
            Header loginUrl[] = httpResponse.getHttpResponse().getHeaders(
                    HEADER_LOCATION);
            httpClient.addQueryParam(USERID_PARAM_NAME, userId);
            httpClient.addQueryParam(PASSWORD_PARAM_NAME, password);
            httpResponse = httpClient.post(loginUrl[0].getValue());
        }
        return httpResponse;
    }
}
