package jp.co.ogis_ri.citk.policytool.common.excel.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.exception.ExcelPolicyWriteException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.logging.LogWrapperFactory;

import org.apache.commons.logging.Log;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;

/**
 * ワークブックラッパー.
 * 
 * @author michio
 *
 */
class WorkbookWrapper {
	private static final Log logger = LogWrapperFactory.getLog(WorkbookWrapper.class);
	
	/**
	 * ラップするワークブック.
	 */
	private Workbook workbook = null;
	
	/**
	 * カレントシート. 
	 */
	private Sheet currentSheet = null;

	/**
	 * カレント行番号.
	 */
	private int currentRow = 0;
	
	/**
	 * カレント列番号.
	 */
	private int currentColumn = 0;
	
	/**
	 * オフセット行番号.
	 */
	private int offsetRow = 0;
	
	/**
	 * オフセット列番号.
	 */
	private int offsetColumn = 0;
	
	/**
	 * ワークブックを作成する.
	 * 
	 * @param fileName ファイル名.
	 * @return ワークブック.
	 * @throws IOException ファイルがオープン出来ない場合.
	 */
	public static HSSFWorkbook createHSSFWorkbook(String fileName) throws IOException {
		FileInputStream is = new FileInputStream(fileName);
		HSSFWorkbook workbook = new HSSFWorkbook(is);
		return workbook;
	}

	/**
	 * コンストラクタ.
	 * 
	 */
	public WorkbookWrapper() {
		this(new HSSFWorkbook());
	}

	/**
	 * コンストラクタ.
	 * 
	 * @param workbook ワークブック.
	 */
	public WorkbookWrapper(Workbook workbook) {
		this.workbook = workbook;
		selectSheet(0, 0, 0);
	}
		
	/**
	 * ワークブックを保存する.
	 * 
	 * @param fileName ファイル名.
	 * @param overWrite true ... 上書き許可 / false ... 上書き禁止.
	 */
	public void saveAs(String fileName, boolean overWrite) {
		File file = new File(fileName);
		if(file.exists()) {
			if(!overWrite) {
				throw new ExcelPolicyWriteException(fileName, "I-0403");
			}
		}
		
		OutputStream os = null;
		try {
			os = new FileOutputStream(file);
			this.workbook.write(os);
		} catch (IOException e) {
        	logger.info(fileName + " : " + e.getMessage());
			throw new ExcelPolicyWriteException(fileName, e, "I-0402");
		} finally {
			if(os != null) {
				try {
					os.close();
				} catch (IOException e) {
					// NOP
				}
			}
		}
	}

	/**
	 * シートを選択する.
	 * 
	 * @param index シート番号.
	 * @param offsetRow シートの行オフセット.
	 * @param offsetColumn シートの列オフセット.
	 */
	public void selectSheet(int index, int offsetRow, int offsetColumn) {
		this.currentSheet = this.workbook.getSheetAt(index);
		this.offsetRow = offsetRow;
		this.offsetColumn = offsetColumn;
		resetCurrent();
	}
	
	/**
	 * シートを選択する.
	 * 
	 * @param sheetName シート名.
	 * @param offsetRow シートの行オフセット.
	 * @param offsetColumn シートの列オフセット.
	 */
	public void selectSheet(String sheetName, int offsetRow, int offsetColumn) {
		selectSheet(workbook.getSheetIndex(sheetName), offsetRow, offsetColumn);
	}

	/**
	 * カレント行の存在の有無.
	 * 
	 * @return true ... 有 / false ... 無
	 */
	public boolean existRow() {
		Row currentRow = this.currentSheet.getRow(getCurrentRow());
		if (currentRow == null) {
			return false;
		}
		
		Cell cell = currentRow.getCell(0);
		if (cell == null) {
			return false;
		}
		
		Object cellValue = getCellValue(cell);
		if (cellValue == null || cellValue.equals("")) {
			return false;
		}
		else {
			return true;
		}
	}

	/**
	 * カレント行を読み込む.
	 * 
	 * @return 読み込んだ値リスト.
	 */
	public List<?> readRow() {
		List<Object> values = new ArrayList<Object>();
		Row _row = getRow(currentSheet, getCurrentRow());
		currentRow++;
		currentColumn = 0;
		while(true) {
			Cell _cell = getCell(_row, getCurrentColumn());
			Object value = getCellValue(_cell);
			if(value == null) {
				break;
			}
			values.add(value);
			currentColumn++;
		}
		return values;
	}

	/**
	 * カレント行を読み込む.
	 * 
	 * @param size 読み込む列数.
	 * @return 読み込んだ値リスト.
	 */
	public List<?> readRow(int size) {
		List<Object> values = new ArrayList<Object>();
		Row _row = getRow(currentSheet, getCurrentRow());
		currentRow++;
		currentColumn = 0;
		while(size-- > 0) {
			Cell _cell = getCell(_row, getCurrentColumn());
			Object value = getCellValue(_cell);
			values.add(value);
			currentColumn++;
		}
		return values;
	}
	
	/**
	 * 値リストの内容をカレント行に書き込む.
	 * 
	 * @param values 行に書き込む値リスト.
	 */
	public void writeRow(List<?> values) {
		Row _row = getRow(currentSheet, getCurrentRow());
		currentRow++;
		currentColumn = 0;
		for(Object value : values) {
			Cell _cell = getCell(_row, getCurrentColumn());
			setCellValue(_cell, value);
			currentColumn++;
		}
	}

	/**
	 * シートの行番号と列番号を指定してセルに値を取得する.
	 * 
	 * @param index シート番号.
	 * @param row 行番号.
	 * @param column 列番号.
	 * @return セルの値.
	 */
	public Object getCellValue(int index, int row, int column) {
		Sheet _sheet = this.workbook.getSheetAt(index);
		Row _row = getRow(_sheet, row);
		Cell _cell =getCell(_row, column);
		return getCellValue(_cell);
	}

	/**
	 * シートの行番号と列番号を指定してセルに値を取得する.
	 * 
	 * @param sheetName シート名.
	 * @param row 行番号.
	 * @param column 列番号.
	 * @return セルの値.
	 */
	public Object getCellValue(String sheetName, int row, int column) {
		return getCellValue(workbook.getSheetIndex(sheetName), row, column);
	}

	/**
	 * シートの行番号と列番号を指定してセルに値を設定する.
	 * 
	 * @param index シート番号.
	 * @param row 行番号.
	 * @param column 列番号.
	 * @param value セットする値.
	 */
	public void setCellValue(int index, int row, int column, Object value) {
		Sheet _sheet = this.workbook.getSheetAt(index);
		Row _row = getRow(_sheet, row);
		Cell _cell =getCell(_row, column);
		setCellValue(_cell, value);
	}

	/**
	 * シートの行番号と列番号を指定してセルに値を設定する.
	 * 
	 * @param sheetName シート名.
	 * @param row 行番号.
	 * @param column 列番号.
	 * @param value セットする値.
	 */
	public void setCellValue(String sheetName, int row, int column, Object value) {
		setCellValue(workbook.getSheetIndex(sheetName), row, column, value);
	}
	
	/**
	 * カレントの行番号と列番号をリセットする.
	 * 
	 */
	public void resetCurrent() {
		this.currentRow = 0;
		this.currentColumn = 0;
	}

	/**
	 * カレントの行番号を取得する.
	 * 
	 * @return 行番号.
	 */
	private int getCurrentRow() {
		return this.currentRow + this.offsetRow;
	}

	/**
	 * カレントの列番号を取得する.
	 * 
	 * @return 列番号.
	 */
	private int getCurrentColumn() {
		return this.currentColumn + this.offsetColumn;
	}
	
	/**
	 * 行を取得する.
	 * 
	 * @param sheet シート.
	 * @param row 行番号.
	 * @return 指定した行.
	 */
	private Row getRow(Sheet sheet, int row) {
		return sheet.getRow(row) != null ? sheet.getRow(row) : sheet.createRow(row);
	}

	/**
	 * セルを取得する.
	 * 
	 * @param row 行.
	 * @param column 列番号.
	 * @return 指定したセル.
	 */
	private Cell getCell(Row row, int column) {
		 return row.getCell(column) != null ?  row.getCell(column) : row.createCell(column);
	}
	
	/**
	 * 	セルの値を取得する.
	 * 
	 * @param cell セル.
	 * @return セルの値.(セルが空の場合、null)
	 */
	private Object getCellValue(Cell cell) {
		Object value = null;
		switch(cell.getCellType()) {
		case Cell.CELL_TYPE_BLANK:
			value = null;
			break;
		case Cell.CELL_TYPE_BOOLEAN:
			value = cell.getBooleanCellValue();
			break;
		case Cell.CELL_TYPE_NUMERIC:
			value = cell.getNumericCellValue();
			break;
		case Cell.CELL_TYPE_STRING:
			value = cell.getStringCellValue();
			break;
		case Cell.CELL_TYPE_ERROR:
		case Cell.CELL_TYPE_FORMULA:
		default:
			// 未サポート
			throw new SystemException("not supported cell type: " + cell);
		}
		return value;
	}

	/**
	 * セルに値をセットする.
	 * 
	 * @param cell 値をセットするセル.
	 * @param value セットする値.
	 */
	private void setCellValue(Cell cell, Object value) {
		if(value instanceof String) {
			cell.setCellType(Cell.CELL_TYPE_STRING);
			cell.setCellValue(String.class.cast((value)));
		}
		else if(value instanceof Number) {
			cell.setCellType(Cell.CELL_TYPE_NUMERIC);
			cell.setCellValue(Double.class.cast((value)));
		}
		else if(value instanceof Boolean) {
			cell.setCellType(Cell.CELL_TYPE_BOOLEAN);
			cell.setCellValue(Boolean.class.cast((value)));
		}
		else if(value == null) {
			// 何もしない
		}
		else {
			// 未サポート
			throw new SystemException("not supported value type: " + value);
		}
	}	
}
