package jp.co.ogis_ri.citk.policytool.common.logging;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * commons loggingのLogインターフェイスが実装されているクラスのラッパーを作成する.
 * ラッパーでは、ログ出力の有効確認を行い、ログ出力メソッドを実行する仕組みを実現する.
 * 
 * @author NAKAGAWA michio
 */
public class LogWrapperFactory {
	/**
	 * オペレーションログの名前文字列.
	 */
	private static final String OPERAION_LOG = "operation";
	
	/**
	 * コンストラクタ.
	 */
	private LogWrapperFactory() {
	}

	/**
	 * ロガーを取得する.
	 * 
	 * @param clazz クラス型.
	 * @return commons loggingのLogインターフェイスが実装されているクラスのラッパー.
	 */
	public static Log getLog(Class<?> clazz) {
		return (Log)Proxy.newProxyInstance(Log.class.getClassLoader(), new Class<?>[] {Log.class}, new LogWrapper(LogFactory.getLog(clazz)));
	}

	/**
	 * ロガーを取得する.
	 * 
	 * @param name ロガー名.
	 * @return commons loggingのLogインターフェイスが実装されているクラスのラッパー.
	 */
	public static Log getLog(String name) {
		return (Log)Proxy.newProxyInstance(Log.class.getClassLoader(), new Class<?>[] {Log.class}, new LogWrapper(LogFactory.getLog(name)));
	}

	/**
	 * 操作記録用ロガーを取得する.
	 * 
	 * @return commons loggingのLogインターフェイスが実装されているクラスのラッパー.
	 */
	public static Log getOperationLog() {
		return (Log)Proxy.newProxyInstance(Log.class.getClassLoader(), new Class<?>[] {Log.class}, new LogWrapper(LogFactory.getLog(OPERAION_LOG)));
	}
	
	/**
	 * ロガーラッパークラス(インナークラス).
	 *
	 */
	private static class LogWrapper implements InvocationHandler {
		private Log logger = null;

		/**
		 * コンストラクタ.
		 * 
		 * @param logger  commons loggingのLogインターフェイスが実装されているクラスのインスタンス.
		 */
		public  LogWrapper(Log logger) {
			this.logger = logger;
		}

		@Override
		public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
			Object result  = null;
			Method m = logger.getClass().getDeclaredMethod(method.getName(), method.getParameterTypes());
			if(m.getName().equals("fatal")) {
				if(logger.isFatalEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else if(m.getName().equals("error")) {
				if(logger.isErrorEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else if(m.getName().equals("warn")) {
				if(logger.isWarnEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else if(m.getName().equals("info")) {
				if(logger.isInfoEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else if(m.getName().equals("debug")) {
				if(logger.isDebugEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else if(m.getName().equals("trace")) {
				if(logger.isTraceEnabled()) {
					result = m.invoke(logger, args);
				}
			}
			else {
				result = m.invoke(logger, args);
			}
			return result;
		}
	};
}
