/**
 *
 */

package jp.co.ogis_ri.citk.policytool.domain.realm.model;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.validation.Valid;

import jp.co.ogis_ri.citk.policytool.common.model.AbstractModel;
import jp.co.ogis_ri.citk.policytool.common.validation.annotation.RealmName;

/**
 * レルム(サービス事業者)を管理するクラス
 */
@Entity
@Table(name="t_realm")
public class Realm extends AbstractModel<Long> {
	/**
	 * シリアルバージョンID.
	 */
	private static final long serialVersionUID = 1L;

    /** JPAで永続化するためのサロゲートキー */
    @Id
    @GeneratedValue
    private Long id;
    
    /** レルム名 */
    @RealmName
    private String realmName;
    
    /** 参照ポリシー */
    @OneToMany(mappedBy="realm", fetch = FetchType.LAZY, cascade = {CascadeType.ALL}, orphanRemoval = true)
    @Valid
    private List<Referral> referrals = new ArrayList<Referral>();

    /** グループ */
    @OneToMany(mappedBy="realm", fetch = FetchType.LAZY, cascade = {CascadeType.ALL}, orphanRemoval = true)
    @Valid
    private List<Group> groups = new ArrayList<Group>();

    @Override
    public Long getId() {
        return this.id;
    }

    /**
     * レルム名参照
	 *
	 * @return レルム名
     */
    public String getRealmName() {
        return realmName;
    }

    /**
     * このレルムにリファーラルを追加する.
     * @param referral リファーラルの文字列.
     */
    public void addReferral(String referral) {
    	addReferral(new Referral(referral));
    }

    /**
     * このレルムにリファーラルを追加する.
     * @param referral リファーラル.
     */
    public void addReferral(Referral referral) {
    	referral.setRealm(this);
        this.referrals.add(referral);
    }

    /**
     * このレルムのリファーラルを取得する.
     * @return このレルムのリファーラルのリスト.
     */
    public List<Referral> getReferrals() {
        return referrals;
    }
    
    /**
     * このレルムにグループを追加する.
     * @param groupName グループ名.
     * @param groupCode グループコード.
     */
    public void addGroup(String groupName, String groupCode) {
        addGroup(new Group(groupName, groupCode));
    }

    /**
     * このレルムにグループを追加する.
     * @param group グループ.
     */
    public void addGroup(Group group) {
        group.setRealm(this);
        groups.add(group);
    }
    
    /**
     * このレルムのグループを取得する.
     * @return このレルムのグループのリスト.
     */
    public List<Group> getGroups() {
        return groups;
    }

    /**
     * コンストラクタ.
     *
     */
    public Realm() {
    }

    /**
     * コンストラクタ.
     *
     * @param realName レルム名
     */
    public Realm(String realmName) {
        this.realmName = realmName;
    }
}
