package jp.co.ogis_ri.citk.policytool.view.policylist;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.text.Document;

import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.exception.OpenAMAccessException;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;
import jp.co.ogis_ri.citk.policytool.service.policylist.PolicyListService;
import jp.co.ogis_ri.citk.policytool.service.policylist.model.PolicyView;
import jp.co.ogis_ri.citk.policytool.view.about.AboutView;
import jp.co.ogis_ri.citk.policytool.view.common.AutoCompleteDocumentListener;
import jp.co.ogis_ri.citk.policytool.view.common.JTableColumnWidthCalculator;
import jp.co.ogis_ri.citk.policytool.view.common.JTableUtil;
import jp.co.ogis_ri.citk.policytool.view.common.RealmComboBoxItem;
import jp.co.ogis_ri.citk.policytool.view.common.StringArrayTableCellRenderer;
import jp.co.ogis_ri.citk.policytool.view.exportpolicy.ExportPolicyView;
import jp.co.ogis_ri.citk.policytool.view.importpolicy.ImportPolicyView;
import jp.co.ogis_ri.citk.policytool.view.policyset.PolicySetView;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.support.MessageSourceAccessor;

/**
 * ポリシー一覧画面の View です.<br>
 * (★注) グループコンボボックスの項目リストは, レルムコンボボックスの選択項目が変更されたときにのみ, 変更される実装となっています.
 */
public class PolicyListView extends JFrame {
	/** シリアルバージョン番号. */
    private static final long serialVersionUID = 1L;

    /** メッセージアクセス オブジェクト */
    private MessageSourceAccessor messages = null;

    /** ポリシー設定画面の View. */
    @Autowired
    private PolicySetView policySetView = null;

    /** インポート画面の View. */
    @Autowired
    private ImportPolicyView importPolicyView = null;

    /** エクスポート画面の View. */
    @Autowired
    private ExportPolicyView exportView = null;

    /** バージョン情報画面の View. */
    @Autowired
    private AboutView aboutView = null;
    
    /** ポリシー一覧画面のサービス. */
    @Autowired
    private PolicyListService service = null;

    /** メニューバー. */
    private JMenuBar menuBar;

    /** ファイルメニュー. */
    private JMenu fileMenu;

    /** ツールメニュー. */
	private JMenu toolMenu;

    /** インポート メニュー項目. */
    private JMenuItem importMenuItem;

    /** エクスポート メニュー項目. */
    private JMenuItem exportMenuItem;

    /** 終了 メニュー項目. */
    private JMenuItem quit;

    /** ヘルプメニュー. */
    private JMenu helpMenu;

    /** バージョン情報 メニュー項目. */
    private JMenuItem aboutMenuItem;

    /** メインの JPanel. */
    private JPanel mainPanel;

    /** ポリシーリスト テーブルの JScrollPane. */
    private JScrollPane policyTableScrollPane;

    /** ポリシーリスト テーブル. */
    private JTable policyTable;

    /** ポリシー検索条件 パネル. */
    private JPanel searchPanel;

    /** ポリシー検索条件の内容パネル. */
    private JPanel searchContentPanel;

    /** レルムコンボボックス タイトル ラベル. */
    private JLabel realmLabel;

    /** レルムコンボボックス. */
    private JComboBox realmComboBox;

    /** グループコンボボックス タイトルラベル. */
    private JLabel groupLabel;

    /** グループコンボボックス. */
    private JComboBox groupComboBox;

    /** リソースコンボボックス タイトルラベル. */
    private JLabel resourceLabel;

    /** リソースコンボボックス. */
    private JComboBox resourceComboBox;

    /** 検索ボタン. */
    private JButton searchButton;

    /** 検索条件パネル タイトル ラベル. */
    private JLabel searchPanelLabel;

    /** 動作ボタン パネル. */
    private JPanel actionButtonsPanel;

    /** 追加ボタン. */
    private JButton addButton;

    /** 修正ボタン. */
    private JButton modifyButton;

    /** 削除ボタン. */
    private JButton deleteButton;

    /** 更新ボタン. */
    private JButton refreshButton;
    
    /** ポリシーテーブルのうち, Policy を持つ列. ユーザーに対しては表示されない. */
    private TableColumn modelColumn;

    /**
     * コンテンツの初期化を行います.
     * 
     */
    public void init() {

        setSize(new Dimension(800, 600));
        setMinimumSize(new Dimension(800, 600));
        setLocationRelativeTo(null);
        setTitle(messages.getMessage("view.policyList.title"));
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        addWindowListener(new WindowAdapter() {
            @Override
            public void windowOpened(WindowEvent e) {
                boolean refreshResult = refresh();
                if (!refreshResult) {
                	System.exit(1);
                }

                // 各列幅の自動設定
                JTableColumnWidthCalculator policyTableCalculator = new JTableColumnWidthCalculator(
                        policyTable);
                policyTableCalculator.setAutomatically();

            }
        });

        GridBagLayout gridBagLayout = new GridBagLayout();
        gridBagLayout.columnWidths = new int[] { 792, 0 };
        gridBagLayout.rowHeights = new int[] { 30, 544, 0 };
        gridBagLayout.columnWeights = new double[] { 0.0, Double.MIN_VALUE };
        gridBagLayout.rowWeights = new double[] { 0.0, 0.0, Double.MIN_VALUE };
        getContentPane().setLayout(gridBagLayout);

        menuBar = new JMenuBar();
        fileMenu = new JMenu(
                messages.getMessage("view.policyList.fileMenuTitle"));
        fileMenu.setMnemonic(KeyEvent.VK_F);

        quit = new JMenuItem(
                messages.getMessage("view.policyList.quitMenuButton"));
        quit.setMnemonic(KeyEvent.VK_X);
        quit.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                System.exit(0);
            }
        });
        fileMenu.add(quit);
        menuBar.add(fileMenu);

		toolMenu = new JMenu(messages.getMessage("view.policyList.toolMenuTitle"));
		toolMenu.setMnemonic(KeyEvent.VK_T);

        importMenuItem = new JMenuItem(messages.getMessage("view.policyList.importMenuButton"));
        importMenuItem.setMnemonic(KeyEvent.VK_I);
        importMenuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                importPolicy();
            }
        });
        toolMenu.add(importMenuItem);

        exportMenuItem = new JMenuItem(messages.getMessage("view.policyList.exportMenuButton"));
        exportMenuItem.setMnemonic(KeyEvent.VK_E);
        exportMenuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                exportPolicy();
            }
        });
        toolMenu.add(exportMenuItem);

		menuBar.add(toolMenu);

        helpMenu = new JMenu(
                messages.getMessage("view.policyList.helpMenuTitle"));
        helpMenu.setMnemonic(KeyEvent.VK_H);
        aboutMenuItem = new JMenuItem(
                messages.getMessage("view.policyList.aboutMenuButton"));
        aboutMenuItem.setMnemonic(KeyEvent.VK_V);
        aboutMenuItem.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
            	about();
            }
        });
        helpMenu.add(aboutMenuItem);
        menuBar.add(helpMenu);

        GridBagConstraints gbc_menuBar = new GridBagConstraints();
        gbc_menuBar.weightx = 1.0;
        gbc_menuBar.fill = GridBagConstraints.BOTH;
        gbc_menuBar.gridx = 0;
        gbc_menuBar.gridy = 0;
        getContentPane().add(menuBar, gbc_menuBar);

        mainPanel = new JPanel();
        // mainPanel.setSize(800, 700);
        GridBagConstraints gbc_mainPanel = new GridBagConstraints();
        gbc_mainPanel.insets = new Insets(5, 5, 5, 5);
        gbc_mainPanel.fill = GridBagConstraints.BOTH;
        gbc_mainPanel.weightx = 1.0;
        gbc_mainPanel.weighty = 1.0;
        gbc_mainPanel.gridx = 0;
        gbc_mainPanel.gridy = 1;
        getContentPane().add(mainPanel, gbc_mainPanel);
        GridBagLayout gbl_mainPanel = new GridBagLayout();
        gbl_mainPanel.columnWidths = new int[] { 0, 0 };
        gbl_mainPanel.rowHeights = new int[] { 0, 0, 0, 0 };
        gbl_mainPanel.columnWeights = new double[] { 0.0, Double.MIN_VALUE };
        gbl_mainPanel.rowWeights = new double[] { 0.0, 0.0, 0.0,
                Double.MIN_VALUE };
        mainPanel.setLayout(gbl_mainPanel);

        policyTableScrollPane = new JScrollPane();
        policyTableScrollPane.setViewportBorder(new EtchedBorder(
                EtchedBorder.LOWERED, null, null));

        policyTable = new JTable();

        policyTable.setFocusable(false);
        policyTable.setFocusTraversalKeysEnabled(false);
        policyTable
                .setBorder(new EtchedBorder(EtchedBorder.LOWERED, null, null));
        policyTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        policyTable
                .setModel(new DefaultTableModel(
                        new Object[][] {},
                        new String[] {
                                messages.getMessage("view.policyList.tableRealmTitle"),
                                messages.getMessage("view.policyList.tablePolicyTitle"),
                                messages.getMessage("view.policyList.tableResourceTitle"),
                                messages.getMessage("view.policyList.tableSubjectTitle"),
                                "" }) {
                    /**
                     * 
                     */
                    private static final long serialVersionUID = 1L;

                    @Override
                    public boolean isCellEditable(int row, int column) {
                        return false;
                    }
                });
        policyTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        policyTable.getTableHeader().setReorderingAllowed(false);

        TableColumnModel policyTableColumnModel = policyTable.getColumnModel();
        TableColumn personColumn = policyTableColumnModel.getColumn(0);
        DefaultTableCellRenderer personColumnCellRenderer = new DefaultTableCellRenderer();
        personColumnCellRenderer.setVerticalAlignment(JLabel.TOP);
        personColumn.setCellRenderer(personColumnCellRenderer);

        TableColumn applicationIDColumn = policyTableColumnModel.getColumn(1);
        DefaultTableCellRenderer applicationIDColumnCellRenderer = new DefaultTableCellRenderer();
        applicationIDColumnCellRenderer.setVerticalAlignment(JLabel.TOP);
        applicationIDColumn.setCellRenderer(applicationIDColumnCellRenderer);

        TableColumn policyTableUrlColumn = policyTableColumnModel.getColumn(2);
        policyTableUrlColumn
                .setCellRenderer(new StringArrayTableCellRenderer());

        TableColumn policyTableRoleColumn = policyTableColumnModel.getColumn(3);
        policyTableRoleColumn
                .setCellRenderer(new StringArrayTableCellRenderer());
        
        /* 不可視にできないため、普段は削除しておく */
        modelColumn = policyTableColumnModel.getColumn(4);
        policyTable.removeColumn(modelColumn);

        searchPanel = new JPanel();
        GridBagLayout gbl_policySearchPanel = new GridBagLayout();
        gbl_policySearchPanel.columnWidths = new int[] { 0, 0 };
        gbl_policySearchPanel.rowHeights = new int[] { 0, 0, 0 };
        gbl_policySearchPanel.columnWeights = new double[] { 0.0,
                Double.MIN_VALUE };
        gbl_policySearchPanel.rowWeights = new double[] { 0.0, 0.0,
                Double.MIN_VALUE };
        searchPanel.setLayout(gbl_policySearchPanel);

        searchContentPanel = new JPanel();
        searchContentPanel.setBorder(new EtchedBorder(EtchedBorder.LOWERED,
                null, null));
        GridBagLayout gbl_policyListSearch = new GridBagLayout();
        gbl_policyListSearch.columnWidths = new int[] { 0, 0, 0, 0, 0, 0, 0, 0 };
        gbl_policyListSearch.rowHeights = new int[] { 22, 0 };
        gbl_policyListSearch.columnWeights = new double[] { 0.0, 0.0, 0.0, 0.0,
                0.0, 0.0, 0.0, Double.MIN_VALUE };
        gbl_policyListSearch.rowWeights = new double[] { 0.0, Double.MIN_VALUE };
        searchContentPanel.setLayout(gbl_policyListSearch);
        resourceComboBox = new JComboBox();
        resourceComboBox.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        resourceComboBox.setPreferredSize(new Dimension(192, 22));
        resourceComboBox.setModel(new DefaultComboBoxModel(new String[] {}));

        resourceComboBox.setEditable(true);

        JTextField resourceComboBoxTextEditor = (JTextField) resourceComboBox
                .getEditor().getEditorComponent();
        Document resourceComboBoxDocument = resourceComboBoxTextEditor
                .getDocument();

        realmComboBox = new JComboBox();
        realmComboBox.setModel(new DefaultComboBoxModel(new Object[] {}));
        realmComboBox.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        realmComboBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                RealmComboBoxItem selectedItem = (RealmComboBoxItem) e
                        .getItem();
                setGroupComboBox(selectedItem);
                setReferralComboBox(selectedItem);
            }
        });

        realmLabel = new JLabel(
                messages.getMessage("view.policyList.listRealmTitle"));
        realmLabel.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        GridBagConstraints gbc_RealmListName = new GridBagConstraints();
        gbc_RealmListName.insets = new Insets(0, 0, 0, 5);
        gbc_RealmListName.gridx = 0;
        gbc_RealmListName.gridy = 0;
        searchContentPanel.add(realmLabel, gbc_RealmListName);

        GridBagConstraints gbc_realmListComboBox = new GridBagConstraints();
        gbc_realmListComboBox.fill = GridBagConstraints.HORIZONTAL;
        gbc_realmListComboBox.weightx = 0.5;
        gbc_realmListComboBox.insets = new Insets(0, 0, 0, 5);
        gbc_realmListComboBox.gridx = 1;
        gbc_realmListComboBox.gridy = 0;
        searchContentPanel.add(realmComboBox, gbc_realmListComboBox);

        groupLabel = new JLabel(
                messages.getMessage("view.policyList.listGroupTitle"));
        groupLabel.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        GridBagConstraints gbc_GroupListName = new GridBagConstraints();
        gbc_GroupListName.insets = new Insets(0, 0, 0, 5);
        gbc_GroupListName.gridx = 2;
        gbc_GroupListName.gridy = 0;
        searchContentPanel.add(groupLabel, gbc_GroupListName);
        groupComboBox = new JComboBox();
        groupComboBox.setModel(new DefaultComboBoxModel(new String[] {}));
        groupComboBox.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        GridBagConstraints gbc_groupListComboBox = new GridBagConstraints();
        gbc_groupListComboBox.fill = GridBagConstraints.HORIZONTAL;
        gbc_groupListComboBox.weightx = 0.5;
        gbc_groupListComboBox.insets = new Insets(0, 0, 0, 5);
        gbc_groupListComboBox.gridx = 3;
        gbc_groupListComboBox.gridy = 0;
        searchContentPanel.add(groupComboBox, gbc_groupListComboBox);

        resourceLabel = new JLabel(
                messages.getMessage("view.policyList.resourceTitle"));
        resourceLabel.setFont(new Font("MS UI Gothic", Font.PLAIN, 12));
        GridBagConstraints gbc_resourceName = new GridBagConstraints();
        gbc_resourceName.insets = new Insets(0, 0, 0, 5);
        gbc_resourceName.gridx = 4;
        gbc_resourceName.gridy = 0;
        searchContentPanel.add(resourceLabel, gbc_resourceName);
        resourceComboBoxDocument
                .addDocumentListener(new AutoCompleteDocumentListener(
                        resourceComboBox));

        GridBagConstraints gbc_resourceComboBox = new GridBagConstraints();
        gbc_resourceComboBox.fill = GridBagConstraints.HORIZONTAL;
        gbc_resourceComboBox.weightx = 1.0;
        gbc_resourceComboBox.insets = new Insets(0, 0, 0, 5);
        gbc_resourceComboBox.gridx = 5;
        gbc_resourceComboBox.gridy = 0;
        searchContentPanel.add(resourceComboBox, gbc_resourceComboBox);
        searchButton = new JButton(
                messages.getMessage("view.policyList.search"));
        searchButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                search();
            }
        });
        searchButton.setHorizontalTextPosition(SwingConstants.CENTER);
        GridBagConstraints gbc_SearchButton = new GridBagConstraints();
        gbc_SearchButton.insets = new Insets(5, 0, 5, 3);
        gbc_SearchButton.gridx = 6;
        gbc_SearchButton.gridy = 0;
        searchContentPanel.add(searchButton, gbc_SearchButton);

        searchPanelLabel = new JLabel(
                messages.getMessage("view.policyList.searchAreaTitle"));
        searchPanelLabel.setAlignmentX(Component.CENTER_ALIGNMENT);
        searchPanelLabel.setHorizontalTextPosition(SwingConstants.CENTER);
        searchPanelLabel.setHorizontalAlignment(SwingConstants.CENTER);
        GridBagConstraints gbc_policySearchPanelTitle = new GridBagConstraints();
        gbc_policySearchPanelTitle.weightx = 1.0;
        gbc_policySearchPanelTitle.gridx = 0;
        gbc_policySearchPanelTitle.gridy = 0;
        searchPanel.add(searchPanelLabel, gbc_policySearchPanelTitle);

        GridBagConstraints gbc_policyListSearch = new GridBagConstraints();
        gbc_policyListSearch.fill = GridBagConstraints.BOTH;
        gbc_policyListSearch.weightx = 1.0;
        gbc_policyListSearch.gridx = 0;
        gbc_policyListSearch.gridy = 1;
        searchPanel.add(searchContentPanel, gbc_policyListSearch);
        GridBagConstraints gbc_policySearchPanel = new GridBagConstraints();
        gbc_policySearchPanel.insets = new Insets(0, 0, 5, 0);
        gbc_policySearchPanel.weightx = 1.0;
        gbc_policySearchPanel.fill = GridBagConstraints.BOTH;
        gbc_policySearchPanel.gridx = 0;
        gbc_policySearchPanel.gridy = 0;
        mainPanel.add(searchPanel, gbc_policySearchPanel);
        policyTableScrollPane.setViewportView(policyTable);

        GridBagConstraints gbc_policyListTable = new GridBagConstraints();
        gbc_policyListTable.insets = new Insets(0, 0, 5, 0);
        gbc_policyListTable.weightx = 1.0;
        gbc_policyListTable.weighty = 1.0;
        gbc_policyListTable.fill = GridBagConstraints.BOTH;
        gbc_policyListTable.gridx = 0;
        gbc_policyListTable.gridy = 1;
        mainPanel.add(policyTableScrollPane, gbc_policyListTable);
        actionButtonsPanel = new JPanel();
        actionButtonsPanel.setBorder(null);

        addButton = new JButton(
                messages.getMessage("view.policyList.addPolicy"));
        addButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                // ★WindowBuilder Editor で編集すると, getContentPane().add()
                // に変わるので注意. add(); に戻すこと.
                add();
            }
        });
        actionButtonsPanel.add(addButton);

        modifyButton = new JButton(
                messages.getMessage("view.policyList.modPolicy"));
        modifyButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                modify();
            }
        });
        actionButtonsPanel.add(modifyButton);

        deleteButton = new JButton(
                messages.getMessage("view.policyList.delPolicy"));
        deleteButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                delete();
            }
        });
        actionButtonsPanel.add(deleteButton);

        refreshButton = new JButton(
                messages.getMessage("view.policyList.refresh"));
        refreshButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                wrapRefresh();
            }
        });
        actionButtonsPanel.add(refreshButton);

        GridBagConstraints gbc_policyListAction = new GridBagConstraints();
        gbc_policyListAction.fill = GridBagConstraints.BOTH;
        gbc_policyListAction.weightx = 1.0;
        gbc_policyListAction.gridx = 0;
        gbc_policyListAction.gridy = 2;
        mainPanel.add(actionButtonsPanel, gbc_policyListAction);

    }

    /**
     * 新しい PolicyListView を構築します.
     * 
     * @param mes MessageSourceAccessor.
     */
    private PolicyListView(MessageSourceAccessor mes) {
        messages = mes;
        init();
    }

    /**
     * インポート処理.
     * 
     */
    private void importPolicy() {
        // 画面表示
    	importPolicyView.setModal(true);
    	importPolicyView.setVisible(true);

    	if (importPolicyView.isRefresh()) {
    		refresh();
    	}
    }

    /**
     * エクスポート処理.
     * 
     */
    private void exportPolicy() {

    	// 検索条件取得
        RealmComboBoxItem selectedRealmItem = (RealmComboBoxItem) realmComboBox.getSelectedItem();
        GroupComboBoxItem selectedGroupItem = (GroupComboBoxItem) groupComboBox.getSelectedItem();
        String selectedResourceItem = (String) resourceComboBox.getSelectedItem();

        String selectedRealmName = selectedRealmItem.getRealmModel() == null ? null
                : selectedRealmItem.getRealmName();
        String selectedGroupName = selectedGroupItem.getGroupName();
        String selectedResourceUrl = selectedResourceItem.trim();
        if (selectedResourceUrl.length() == 0) {
        	selectedResourceUrl = null;
        }
    	
        // 画面表示内容設定
        exportView.initData(selectedRealmName, selectedGroupName, selectedResourceUrl);

        // 画面表示
        exportView.setModal(true);
        exportView.setVisible(true);
    }

    /**
     * 追加ボタン押下時の処理です.
     * 
     */
    private void add() {
    	try {
	    	startEvent();

	        // 画面表示内容設定
	        policySetView.initData();
	
	        // 画面表示
	        policySetView.setModal(true);
	        policySetView.setVisible(true);
	
	        // 追加が行われた場合は検索.
	        if (policySetView.getResult()) {
	        	search();
	        }

    	} finally {
    		endEvent();
    	}
    }

    /**
     * 修正ボタン押下時の処理です.
     * 
     */
    private void modify() {
    	try {
	    	startEvent();
	
	        // PolicyTable で行が選択されている場合, その行についての修正画面を表示
	        if (policyTable.getSelectedRow() == -1) {
	            JOptionPane.showMessageDialog(this, messages.getMessage("view.policyList.noPolicySelectedMessage"));
	        } else {

	            // 選択ポリシーを取得
	        	policyTable.addColumn(modelColumn);        	
	        	Policy selectedPolicy = (Policy)policyTable.getValueAt(policyTable.getSelectedRow(), 4);
	        	policySetView.initData(selectedPolicy);
	        	policyTable.removeColumn(modelColumn);
	
	            // 画面表示
	            policySetView.setModal(true);
	            policySetView.setVisible(true);
	
	            // 修正が行われた場合は検索.
	            if (policySetView.getResult()) {
	                search();
	            }
	        }
    	} finally {
    		endEvent();
    	}
    }

    /**
     * 削除ボタン押下時の処理です.
     * 
     */
    private void delete() {
    	try {
	    	startEvent();
	
	        // PolicyTable で行が選択されている場合, その行のポリシーを削除
	        if (policyTable.getSelectedRow() == -1) {
	            JOptionPane.showMessageDialog(this, messages.getMessage("view.policyList.noPolicySelectedMessage"));
	        } else {

	            // 選択ポリシーを取得
	        	policyTable.addColumn(modelColumn);
	            Policy policy = (Policy)policyTable.getValueAt(policyTable.getSelectedRow(), 4);
	            policyTable.removeColumn(modelColumn);
	
	            // 確認ダイアログ
	            Object[] objs = { policy.getRealmName(), policy.getPolicyName() };
	            String confirmMessage = messages.getMessage("view.policyList.deleteConfirmMessage", objs);
	            int confirmResult = JOptionPane.showConfirmDialog(this, confirmMessage,
	            		messages.getMessage("view.policyList.confirm"),
	                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE,
	                    null);
	
	            if (confirmResult == JOptionPane.YES_OPTION) {
	            	try {
	            		service.delete(policy);
	            	} catch (OpenAMAccessException e) {
	        			JOptionPane.showMessageDialog(this, e.getMessage());
	        			return;
	            	} catch (ApplicationException e) {
	        			JOptionPane.showMessageDialog(this, e.getMessage());
	        			return;
	            	}
	            	
	                refresh();
	                search();
	            }
	        }
    	} finally {
    		endEvent();
    	}
    }

    /**
     * バージョン情報メニュー選択時の処理.
     */
    private void about() {
        // 画面表示
    	aboutView.setModal(true);
	  	aboutView.setVisible(true);
    }
    
    /**
     * 更新ボタン押下時の処理です.
     * 
     */
    private void wrapRefresh() {
    	try {
    		startEvent();
    		refresh();
    	} finally {
    		endEvent();
    	}
    }

    /**
     * 画面更新処理です.
     * 
     * @return 更新に成功した場合 true, そうでない場合 false.
     */
    private boolean refresh() {
    	try {
    		service.refresh();
    	} catch (OpenAMAccessException e) {
        	JOptionPane.showMessageDialog(this, e.getMessage());
    		return false;
    	} catch (ApplicationException e) {
        	JOptionPane.showMessageDialog(this, e.getMessage());
    		return false;
    	}
    	
        setRealmComboBox();
        
        // 一覧画面をクリア
        DefaultTableModel policyTableModel = (DefaultTableModel) policyTable
                .getModel();
        policyTableModel.setRowCount(0);
        
        return true;
    }
    
    /**
     * レルムコンボボックスに選択項目を設定します.
     */
    private void setRealmComboBox() {
    	realmComboBox.setVisible(false);
    	List<Realm> realms = service.findRealms();
    	realmComboBox.removeAllItems();
    	realmComboBox.addItem(RealmComboBoxItem.ALL_REALM);
    	for (Realm realm : realms) {
    		realmComboBox.addItem(new RealmComboBoxItem(realm.getRealmName(), realm));
    	}
    	realmComboBox.setSelectedItem(RealmComboBoxItem.ALL_REALM);
    	realmComboBox.setVisible(true);
    }

    /**
     * グループリスト コンボボックスの項目を, 指定されたレルム項目のものに設定します.
     * 
     * @param realmItem レルム項目
     * 
     */
    private void setGroupComboBox(RealmComboBoxItem realmItem) {
    	groupComboBox.setVisible(false);
        // グループリストコンボボックス項目作成
        List<String> groupList = service.findGroupNames(realmItem.getRealmModel());

        // グループ コンボボックスの項目更新
        groupComboBox.removeAllItems();
        // 空項目を加える
        groupComboBox.addItem(new GroupComboBoxItem(null, null));
        for (String groupName : groupList) {
            GroupComboBoxItem groupItem = new GroupComboBoxItem(groupName, groupName);
            groupComboBox.addItem(groupItem);
        }
    	groupComboBox.setVisible(true);
    }

    /**
     * リソースリスト コンボボックスの項目を, 指定されたレルム項目のものに設定します.
     * 
     * @param realmItem レルム項目
     * 
     */
    private void setReferralComboBox(RealmComboBoxItem realmItem) {
    	resourceComboBox.setVisible(false);
        // グループリストコンボボックス項目作成
        List<Referral> referrals = service.findReferrals(realmItem.getRealmModel());

        resourceComboBox.removeAllItems();
        // 空項目を加える
        resourceComboBox.addItem("");
        for (Referral referral : referrals) {
        	resourceComboBox.addItem(referral.getRefpolicy());
        }
    	resourceComboBox.setVisible(true);
    }

    /**
     * 検索ボタン押下時の処理です.
     * 
     */
    private void search() {
    	try {
	    	startEvent();
	
	        // 検索処理
	        RealmComboBoxItem selectedRealmItem = (RealmComboBoxItem) realmComboBox.getSelectedItem();
	        GroupComboBoxItem selectedGroupItem = (GroupComboBoxItem) groupComboBox.getSelectedItem();
	        String selectedResourceItem = (String) resourceComboBox.getSelectedItem();
	
	        String selectedRealmName = selectedRealmItem.getRealmModel() == null ? null
	                : selectedRealmItem.getRealmName();
	        String selectedGroupName = selectedGroupItem.getGroupName();
	        String selectedResourceUrl = selectedResourceItem.trim();
	        if (selectedResourceUrl.length() == 0) {
	        	selectedResourceUrl = null;
	        }
	
	        List<PolicyView> resultPolicyList = service.findPolicies(selectedRealmName,
	                selectedGroupName, selectedResourceUrl);
	
	        // 表示設定
	        setDataToPolicyTable(resultPolicyList);
	        
	        // 列幅調整
	        JTableColumnWidthCalculator policyTableCalculator = new JTableColumnWidthCalculator(
	                policyTable);
	        policyTableCalculator.setAutomatically();
    	} finally {
    		endEvent();
    	}
    }

    /**
     * 指定されたポリシーのリストを, ポリシーテーブルに設定します.
     * 
     * @param policyList ポリシーのリスト.
     * 
     */
    private void setDataToPolicyTable(List<PolicyView> policyList) {
    	policyTable.setVisible(false);
    	policyTable.addColumn(modelColumn);
        DefaultTableModel policyTableModel = (DefaultTableModel) policyTable
                .getModel();
        policyTableModel.setRowCount(0);

        for (PolicyView policy : policyList) {
        	Object[] rowDataArray = new Object[] {
        			policy.getRealmName(),
        			policy.getPolicyName(),
        			policy.getResourceUrls(),
        			policy.getSubjects(),
        			policy.getPolicy()
        	};
            policyTableModel.addRow(rowDataArray);
        }

        policyTable.removeColumn(modelColumn);
        // 各行の高さを計算
        JTableUtil.refreshTableRowHeight(policyTable);
    	policyTable.setVisible(true);
    }
    
    /**
     * 処理開始時に, マウスカーソルを砂時計にし, 各ボタンの有効/無効状態を無効にします.
     */
    private void startEvent() {
    	setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    	
    	searchButton.setEnabled(false);
    	addButton.setEnabled(false);
    	modifyButton.setEnabled(false);
    	deleteButton.setEnabled(false);
    	refreshButton.setEnabled(false);
    }

    /**
     * 処理終了時に, マウスカーソルを元に戻し, 各ボタンの有効/無効状態を元に戻します.
     */
    private void endEvent() {
    	setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    	
    	searchButton.setEnabled(true);
    	addButton.setEnabled(true);
    	modifyButton.setEnabled(true);
    	deleteButton.setEnabled(true);
    	refreshButton.setEnabled(true);
    }
}
