package jp.co.ogis_ri.citk.policytool.common.api.impl;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMConditions;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;

import org.junit.Test;

public class OpenAMModelContextTest {
    @Test
    public void testMarshaller_UTF8_FALSE() throws UnsupportedEncodingException {
        OpenAMPolicies policies = createOpenAMPolicies();
        
        OpenAMModelContext.setJaxbEncoding("UTF-8");
        OpenAMModelContext.setJaxbFormatOutput(false);
        
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        OpenAMModelContext.marshaller(policies, outputStream);
        
        String actual = outputStream.toString("UTF-8");
        String lineSeparator  = System.getProperty("line.separator");
        assertEquals(
                "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + lineSeparator
                        + "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">" + lineSeparator
                        + "<Policies><Policy active=\"true\" description=\"あいうえお\" name=\"pol\"><Conditions description=\"説明\" name=\"cond\"/></Policy></Policies>"
                , actual);
    }

    @Test
    public void testMarshaller_SJIS_FALSE() throws UnsupportedEncodingException {
        OpenAMPolicies policies = createOpenAMPolicies();
        
        OpenAMModelContext.setJaxbEncoding("Shift_JIS");
        OpenAMModelContext.setJaxbFormatOutput(false);
        
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        OpenAMModelContext.marshaller(policies, outputStream);
        
        String actual = outputStream.toString("Shift_JIS");
        String lineSeparator  = System.getProperty("line.separator");
        assertEquals( 
                "<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" + lineSeparator
                        + "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">" + lineSeparator
                        + "<Policies><Policy active=\"true\" description=\"あいうえお\" name=\"pol\"><Conditions description=\"説明\" name=\"cond\"/></Policy></Policies>"
                , actual);
    }

    @Test
    public void testMarshaller_UTF8_TRUE() throws UnsupportedEncodingException {
        OpenAMPolicies policies = createOpenAMPolicies();
        
        OpenAMModelContext.setJaxbEncoding("UTF-8");
        OpenAMModelContext.setJaxbFormatOutput(true);
        
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        OpenAMModelContext.marshaller(policies, outputStream);
        
        String actual = outputStream.toString("UTF-8");
        String lineSeparator  = System.getProperty("line.separator");
        assertEquals(
                "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + lineSeparator
                + "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">" + lineSeparator
                + "\n"
                + "<Policies>\n"
                + "    <Policy active=\"true\" description=\"あいうえお\" name=\"pol\">\n"
                + "        <Conditions description=\"説明\" name=\"cond\"/>\n"
                + "    </Policy>\n"
                + "</Policies>\n"
                , actual);
    }

    @Test
    public void testMarshaller_SJIS_TRUE() throws UnsupportedEncodingException {
        OpenAMPolicies policies = createOpenAMPolicies();
        
        OpenAMModelContext.setJaxbEncoding("Shift_JIS");
        OpenAMModelContext.setJaxbFormatOutput(true);
        
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        OpenAMModelContext.marshaller(policies, outputStream);
        
        String actual = outputStream.toString("Shift_JIS");
        String lineSeparator  = System.getProperty("line.separator");
        assertEquals(
                "<?xml version=\"1.0\" encoding=\"Shift_JIS\"?>" + lineSeparator
                + "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">" + lineSeparator
                + "<Policies>\n"
                + "    <Policy active=\"true\" description=\"あいうえお\" name=\"pol\">\n"
                + "        <Conditions description=\"説明\" name=\"cond\"/>\n"
                + "    </Policy>\n"
                + "</Policies>"
                , actual);
    }

    @Test
    public void testMarshaller_JAXBException(final Marshaller marshaller) throws JAXBException {
    	OpenAMModelContext.setJaxbEncoding("pppp");
    	
        OpenAMPolicies policies = createOpenAMPolicies();
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        
        try {
        	OpenAMModelContext.marshaller(policies, outputStream);
        	fail();
        } catch (SystemException e) {
        	assertTrue(e.getCause() instanceof JAXBException);
        }
    }
    
    @Test
    public void testMarshaller_IOException(final Marshaller marshaller) throws JAXBException, IOException {
    	OpenAMModelContext.setJaxbEncoding("UTF-8");
    	
        OpenAMPolicies policies = createOpenAMPolicies();
        OutputStream outputStream = new IOExceptionThrowingOutputStream();
        
        try {
        	OpenAMModelContext.marshaller(policies, outputStream);
        	fail();
        } catch (SystemException e) {
        	assertTrue(e.getCause() instanceof IOException);
        }
    }
    
    @Test
    public void testUnmarshallerClassOfTInputStream_UTF8() throws UnsupportedEncodingException {
        String xmlString = createXMLString("UTF-8");
        
        OpenAMModelContext.setJaxbEncoding("UTF-8");
        OpenAMModelContext.setJaxbFormatOutput(false);
        
        InputStream inputStream = new ByteArrayInputStream(xmlString.getBytes("UTF-8"));
        
        OpenAMPolicies policies = OpenAMModelContext.unmarshaller(OpenAMPolicies.class, inputStream);
        assertOpenAMPolicies(policies);
    }

    @Test
    public void testUnmarshallerClassOfTInputStream_SJIS() throws UnsupportedEncodingException  {
        String xmlString = createXMLString("Shift_JIS");
        
        OpenAMModelContext.setJaxbEncoding("Shift_JIS");
        OpenAMModelContext.setJaxbFormatOutput(false);
        
        InputStream inputStream = new ByteArrayInputStream(xmlString.getBytes("Shift_JIS"));
        
        OpenAMPolicies policies = OpenAMModelContext.unmarshaller(OpenAMPolicies.class, inputStream);
        
        assertOpenAMPolicies(policies);
    }

    @Test
    public void testUnmarshallerClassOfTInputStream_JAXBException() throws UnsupportedEncodingException, JAXBException {
    	// 不正なXML文字列を設定することにより、JAXBExceptionをスローさせる
        InputStream inputStream = new ByteArrayInputStream("123".getBytes("UTF-8"));

        try {
        	OpenAMModelContext.unmarshaller(OpenAMPolicies.class, inputStream);
        	fail();
        } catch (SystemException e) {
        	assertTrue(e.getCause() instanceof JAXBException);
        }
    }

    private void assertOpenAMPolicies(OpenAMPolicies policies) {
        assertThat(policies.getPolicy().size(), is(1));
        
        OpenAMPolicy policy = policies.getPolicy().get(0);
        assertThat(policy.getName(), is("pol"));
        assertThat(policy.isActive(), is(true));
        assertThat(policy.getDescription(), is("あいうえお"));
        
        OpenAMConditions conditions = policy.getConditions();
        assertThat(conditions.getName(), is("cond"));
        assertThat(conditions.getDescription(), is("説明"));
    }
    
    private OpenAMPolicies createOpenAMPolicies() {
        OpenAMPolicies policies = new OpenAMPolicies();
        
        OpenAMPolicy policy1 = new OpenAMPolicy();
        policy1.setName("pol");
        policy1.setActive(true);
        policy1.setDescription("あいうえお");
        policies.getPolicy().add(policy1);
        
        OpenAMConditions conditions = new OpenAMConditions();
        conditions.setName("cond");
        conditions.setDescription("説明");
        policy1.setConditions(conditions);
        
        return policies;
    }
    
    private String createXMLString(String encodingStr) {
        String lineSeparator  = System.getProperty("line.separator");
        return 
                "<?xml version=\"1.0\" encoding=\"" + encodingStr + "\"?>" + lineSeparator
                + "<!DOCTYPE Policies PUBLIC \"-//OpenSSO Policy Administration DTD//EN\" \"jar://com/sun/identity/policy/policyAdmin.dtd\">" + lineSeparator
                + "\n"
                + "<Policies>\n"
                + "    <Policy active=\"true\" description=\"あいうえお\" name=\"pol\">\n"
                + "        <Conditions description=\"説明\" name=\"cond\"/>\n"
                + "    </Policy>\n"
                + "</Policies>\n";
    }
    
    /**
     * IOExceptionをスローするテスト用OutputAStream
     *
     */
    private static class IOExceptionThrowingOutputStream extends OutputStream {
		@Override
		public void write(int b) throws IOException {
		}
		
		public void close() throws IOException {
			throw new IOException();
		}
    }
}
