package jp.co.ogis_ri.citk.policytool.common.api.impl;

import static org.junit.Assert.assertEquals;

import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttribute;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMAttributeValuePair;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicies;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMPolicy;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMResourceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMRule;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMServiceName;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubject;
import jp.co.ogis_ri.citk.policytool.common.api.impl.model.OpenAMSubjects;
import jp.co.ogis_ri.citk.policytool.common.util.ApplicationContextUtil;
import jp.co.ogis_ri.citk.policytool.common.util.OpenAMNamingUtil;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Permit;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Policy;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Resource;
import jp.co.ogis_ri.citk.policytool.domain.policy.model.Subject;

import mockit.Deencapsulation;

import org.junit.Before;
import org.junit.Test;

public class PoliciesBuilderTest {
    @Before
    public void setUp() {
        Deencapsulation.setField(ApplicationContextUtil.class, "context", null);
        Deencapsulation.setField(
                ApplicationContextUtil.class,
                "FILE_APPLICATION_CONTEXT",
                "jp/co/ogis_ri/citk/policytool/common/api/impl/PoliciesBuilderTest-context.xml");
    }
    
	/**
	 * 通常パターン - ポリシー基本情報, Subject を確認するパターン
	 *
	 */
    @Test
    public void testBuild() {
        //-----
        // Policies その1
        //-----
        OpenAMPolicies policies = new OpenAMPolicies();

        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("policyname");
        policy.setDescription("description");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin123,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1319176315017L);
        policy.setLastmodifieddate(1319176315027L);
        policy.setReferralPolicy(false);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        // Subjects
        OpenAMSubjects subjects = new OpenAMSubjects();
        subjects.setName("subjectsname");
        subjects.setDescription("description");
        policy.setSubjects(subjects);
        
    	// Subject 1 個目
        OpenAMSubject subject1 = createSubject1();
        subjects.getSubject().add(subject1);

        // 2 個目以降のSubjectは読み込まれないことを確認
        OpenAMSubject subject2 = createSubject2();
        subjects.getSubject().add(subject2);
        
        // リソースがない場合はポリシーが作れないため、1つだけポリシーを作成する
        OpenAMRule resource = createRule1();
        policy.getRule().add(resource);
        
    	// ポリシー2 参照ポリシーであるケース
        OpenAMPolicy policy2 = new OpenAMPolicy();
        policy2.setReferralPolicy(true);
        policies.getPolicy().add(policy2);

        //--------------------------------
        // テスト実行
        //--------------------------------
        PoliciesBuilder builder = new PoliciesBuilder();
        builder.add("testRealm", policies);
        List<Policy> policyList = builder.build();
        
        //--------------------------------
        // 結果確認
        //--------------------------------
        // Policy リストの確認
        assertEquals(1, policyList.size());
        Policy resultPolicy = policyList.get(0);
        
        //-----
        // Policies その1
        //-----
        assertEquals("policyname", resultPolicy.getPolicyName());
        
        assertEquals(3, resultPolicy.getSubjects().size());
        assertEquals(1, resultPolicy.getResources().size());
        
        // Subject の確認
        Subject resultSubject1 = resultPolicy.getSubjects().get(0);
        assertEquals("username10", resultSubject1.getSubjectName());
        assertEquals("id=username10,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net", resultSubject1.getSubjectCode());
        
        Subject resultSubject2 = resultPolicy.getSubjects().get(1);
        assertEquals("username11", resultSubject2.getSubjectName());
        assertEquals("id=username11,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net", resultSubject2.getSubjectCode());
        
        Subject resultSubject3 = resultPolicy.getSubjects().get(2);
        assertEquals("username12", resultSubject3.getSubjectName());
        assertEquals("id=username12,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net", resultSubject3.getSubjectCode());
    }
    
    /**
     * 通常パターン - Resource を確認するパターン
     *
     */
    @Test
    public void testBuild_Resource() {
        //-----
        // Policies その1
        //-----
        OpenAMPolicies policies = new OpenAMPolicies();

        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("policyname");
        policy.setDescription("description");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin123,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1319176315017L);
        policy.setLastmodifieddate(1319176315027L);
        policy.setReferralPolicy(false);
        policy.setActive(true);
        policies.getPolicy().add(policy);
        
        OpenAMRule rule1 = createRule1();
        policy.getRule().add(rule1);
        OpenAMRule rule2 = createRule2();
        policy.getRule().add(rule2);
        OpenAMRule rule3 = createRule3();
        policy.getRule().add(rule3);
        
    	// Subjects
        OpenAMSubjects subjects = new OpenAMSubjects();
        subjects.setName("subjectsname");
        subjects.setDescription("description");
        policy.setSubjects(subjects);
     
        OpenAMSubject subject = new OpenAMSubject();
        subject.setName("subjectname1");
        subject.setType(OpenAMNamingUtil.SUBJECT_TYPE);
        subject.setIncludeType(OpenAMNamingUtil.SUBJECT_INCLUDE_TYPE);
        subjects.getSubject().add(subject);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        subject.getAttributeValuePair().add(attributeValuePair);

        // ポリシーは最低ひとつはサブジェクトを持ってない作成されないため、サブジェクトを一つ作る
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        attributeValuePair.getValue().add("id=username10,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        
    	// ポリシー2 参照ポリシーであるケース
        OpenAMPolicy policy2 = new OpenAMPolicy();
        policy2.setReferralPolicy(true);
        policies.getPolicy().add(policy2);

        //--------------------------------
        // テスト実行
        //--------------------------------
        PoliciesBuilder builder = new PoliciesBuilder();
        builder.add("testRealm", policies);
        List<Policy> resultPolicyList = builder.build();
        
        //--------------------------------
        // 結果確認
        //--------------------------------
        // Policy リストの確認
        assertEquals(1, resultPolicyList.size());
        Policy resultPolicy = resultPolicyList.get(0);
        assertEquals(1, resultPolicy.getSubjects().size());
        assertEquals(3, resultPolicy.getResources().size());
        
        //-----
        // Policies その1
        //-----
        // Resource の確認
        Resource resultResource1 = resultPolicy.getResources().get(0);
        assertEquals("http://host/resource/*", resultResource1.getResourceUrl());
        assertEquals(Permit.INDETERMINATE, resultResource1.getPostPermit());
        assertEquals(Permit.ALLOW, resultResource1.getGetPermit());
        
        Resource resultResource2 = resultPolicy.getResources().get(1);
        assertEquals("http://host1/resource1/*", resultResource2.getResourceUrl());
        assertEquals(Permit.ALLOW, resultResource2.getPostPermit());
        assertEquals(Permit.DENY, resultResource2.getGetPermit());
        
        Resource resultResource3 = resultPolicy.getResources().get(2);
        assertEquals("http://host1/resource1/333333/*", resultResource3.getResourceUrl());
        assertEquals(Permit.INDETERMINATE, resultResource3.getPostPermit());
        assertEquals(Permit.INDETERMINATE, resultResource3.getGetPermit());
    }

    /**
     * Policies その2 Subject, Resource が 0 個
     *
     */
    @Test
    public void testBuild_SubjectZero_ResourceZero() {
        OpenAMPolicies openAMPolicies2 = new OpenAMPolicies();

        OpenAMPolicy policy = new OpenAMPolicy();
        policy.setName("policyname");
        policy.setDescription("description");
        policy.setCreatedby("id=amadmin,ou=user,dc=opensso,dc=java,dc=net");
        policy.setLastmodifiedby("id=amadmin123,ou=user,dc=opensso,dc=java,dc=net");
        policy.setCreationdate(1319176315017L);
        policy.setLastmodifieddate(1319176315027L);
        policy.setReferralPolicy(false);
        policy.setActive(true);
        openAMPolicies2.getPolicy().add(policy);
        
        OpenAMSubjects subjects = new OpenAMSubjects();
        subjects.setName("subjectsname");
        subjects.setDescription("description");
        policy.setSubjects(subjects);
        
        OpenAMSubject subject = new OpenAMSubject();
        subjects.getSubject().add(subject);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        subject.getAttributeValuePair().add(attributeValuePair);
        
        // テスト実行
        PoliciesBuilder builder = new PoliciesBuilder();
        builder.add("testRealm22", openAMPolicies2);
        List<Policy> policyList = builder.build();

        // 結果確認
        // サブジェクトかリソースがないポリシーは取り込まない使用としたため、テスト結果が変更となった。
//        Policy resultPolicy = policyList.get(0);
//        
//        assertEquals(0, resultPolicy.getSubjects().size());
//        assertEquals(0, resultPolicy.getResources().size());
        assertEquals(0, policyList.size());
    }
    
    /**
     * ポリシーが 0 個
     *
     *
     */
    @Test
    public void testBuild_PolicyZero() {
        OpenAMPolicies openAMPolicies3 = new OpenAMPolicies();
    	
        PoliciesBuilder builder = new PoliciesBuilder();
        builder.add("testRealm33", openAMPolicies3);
        List<Policy> policyList = builder.build();
        
        assertEquals(0, policyList.size());
    }
    
    /**
     * 通常パターン 1番目 Subject作成
     *
     * @return 通常パターン 1番目 Subject
     *
     */
    private OpenAMSubject createSubject1() {
        OpenAMSubject subject = new OpenAMSubject();
        subject.setName("subjectname1");
        subject.setType(OpenAMNamingUtil.SUBJECT_TYPE);
        subject.setIncludeType(OpenAMNamingUtil.SUBJECT_INCLUDE_TYPE);
        
        OpenAMAttributeValuePair attributeValuePair = new OpenAMAttributeValuePair();
        subject.getAttributeValuePair().add(attributeValuePair);
        
        OpenAMAttribute attribute = new OpenAMAttribute();
        attribute.setName(OpenAMNamingUtil.VALUES_ATTR_NAME);
        attributeValuePair.setAttribute(attribute);
        
        attributeValuePair.getValue().add("id=username10,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        attributeValuePair.getValue().add("id=username11,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        attributeValuePair.getValue().add("id=username12,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        
        return subject;
    }
    
    /**
     * 通常パターン 2番目 Subject作成
     *
     * @return 通常パターン 2番目 Subject
     *
     */
    private OpenAMSubject createSubject2() {
        OpenAMSubject subject2 = new OpenAMSubject();
        subject2.setName("subjectname3");
        subject2.setType(OpenAMNamingUtil.SUBJECT_TYPE);
        subject2.setIncludeType(OpenAMNamingUtil.SUBJECT_INCLUDE_TYPE);
        
        OpenAMAttributeValuePair attributeValuePair2 = new OpenAMAttributeValuePair();
        subject2.getAttributeValuePair().add(attributeValuePair2);
        
        OpenAMAttribute attribute2 = new OpenAMAttribute();
        attribute2.setName("ValuesX");
        attributeValuePair2.setAttribute(attribute2);
        
        attributeValuePair2.getValue().add("id=username20,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        attributeValuePair2.getValue().add("id=username21,ou=user,o=relmname,ou=services,dc=opensso,dc=java,dc=net");
        
        return subject2;
    }
    
    /**
     * 通常パターン 1番目 ルール作成
     *
     */
    private OpenAMRule createRule1() {
        OpenAMRule rule1 = new OpenAMRule();
        rule1.setName("rulename1");
        
        OpenAMServiceName serviceName11 = new OpenAMServiceName();
        serviceName11.setName("iPlanetAMWebAgentServiceZZZ");
        rule1.setServiceName(serviceName11);
        
        OpenAMResourceName resourceName11 = new OpenAMResourceName();
        resourceName11.setName("http://host/resource/*");
        rule1.setResourceName(resourceName11);
        
        OpenAMAttributeValuePair attributeValuePair11 = new OpenAMAttributeValuePair();
        rule1.getAttributeValuePair().add(attributeValuePair11);
        
        OpenAMAttribute attribute11 = new OpenAMAttribute();
        attribute11.setName("POST123");
        attributeValuePair11.setAttribute(attribute11);
        
        attributeValuePair11.getValue().add("abc");
        
        OpenAMAttributeValuePair attributeValuePair12 = new OpenAMAttributeValuePair();
        rule1.getAttributeValuePair().add(attributeValuePair12);
        
        OpenAMAttribute attribute12 = new OpenAMAttribute();
        attribute12.setName(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME);
        attributeValuePair12.setAttribute(attribute12);
    
        attributeValuePair12.getValue().add("allow");
        
        return rule1;
    }
    
    /**
     * 通常パターン 2番目 ルール作成
     *
     */
    private OpenAMRule createRule2() {
        OpenAMRule rule2 = new OpenAMRule();
        rule2.setName("rulename2");
        
        OpenAMServiceName serviceName21 = new OpenAMServiceName();
        serviceName21.setName(OpenAMNamingUtil.SERVICE_NAME);
        rule2.setServiceName(serviceName21);
        
        OpenAMResourceName resourceName21 = new OpenAMResourceName();
        resourceName21.setName("http://host1/resource1/*");
        rule2.setResourceName(resourceName21);
        
        OpenAMAttributeValuePair attributeValuePair21 = new OpenAMAttributeValuePair();
        rule2.getAttributeValuePair().add(attributeValuePair21);
        
        OpenAMAttribute attribute21 = new OpenAMAttribute();
        attribute21.setName(OpenAMNamingUtil.RULE_ATTR_POSTMETHOD_NAME);
        attributeValuePair21.setAttribute(attribute21);
        
        attributeValuePair21.getValue().add("allow");
        
        OpenAMAttributeValuePair attributeValuePair22 = new OpenAMAttributeValuePair();
        rule2.getAttributeValuePair().add(attributeValuePair22);
        
        OpenAMAttribute attribute22 = new OpenAMAttribute();
        attribute22.setName(OpenAMNamingUtil.RULE_ATTR_GETMETHOD_NAME);
        attributeValuePair22.setAttribute(attribute22);
    
        attributeValuePair22.getValue().add("deny");
        
        return rule2;
    }
    
    /**
     * 通常パターン 3番目 ルール作成
     *
     */
    private OpenAMRule createRule3() {
        OpenAMRule rule3 = new OpenAMRule();
        rule3.setName("rulename3");
        
        OpenAMServiceName serviceName31 = new OpenAMServiceName();
        serviceName31.setName("iPlanetAMWebAgentService33");
        rule3.setServiceName(serviceName31);
        
        OpenAMResourceName resourceName31 = new OpenAMResourceName();
        resourceName31.setName("http://host1/resource1/333333/*");
        rule3.setResourceName(resourceName31);
        
        return rule3;
    }
}
