package jp.co.ogis_ri.citk.policytool.common.api.impl.ssoadm;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationSettingException;
import jp.co.ogis_ri.citk.policytool.common.http.HttpClientWrapper;
import jp.co.ogis_ri.citk.policytool.common.http.HttpResponseWrapper;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import mockit.Expectations;
import mockit.Mocked;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.message.BasicHeader;
import org.junit.Test;

public class SsoadmClientTest {
	private SsoadmClient ssoadmClient = new SsoadmClient();
	
	@Mocked
    private HttpClientWrapper httpClientWrapper;
	
	@Mocked
	private HttpResponseWrapper loginHttpResponseWrapper;
	
	@Mocked
	private HttpResponse loginHttpResponse;
	
	@Mocked
	private HttpResponseWrapper getRealmsHttpResponseWrapper;
	
	@Mocked
	private RealmSsoadmResultParser realmSsoadmResultParser;
    
	@Mocked
	private GroupSsoadmResultParser groupSsoadmResultParser;
	
	@Mocked
	private PoliciesSsoadmResultParser policiesSsoadmResultParser;
    
	@SuppressWarnings("unused")
	@Mocked
	private MessageInfo messageInfo;
	
    /**
     * getRealms のテスト<br>
     * ログインされていないときにログインされていることの確認も行う
     *
     */
    @Test
    public void testGetRealms() throws UnsupportedEncodingException {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setUserId("USER_ID");
    	ssoadmClient.setPassword("PASS");
    	ssoadmClient.setEncoding("UTF-8");
    	
    	final SsoadmResult ssoadmResult = new SsoadmResult(new ArrayList<String>(), 5);
    	
    	new Expectations() {
    		{
    	    	// ログイン確認
    			httpClientWrapper.post("http://12345/"); result = loginHttpResponseWrapper;
    			
    	    	// ログイン
    			loginHttpResponseWrapper.getHttpStatusCode(); result = 100;
    			loginHttpResponseWrapper.getHttpResponse(); result = loginHttpResponse;
    			loginHttpResponse.getHeaders("Location"); result = new Header[] { new BasicHeader("k123", "v123") };
    			httpClientWrapper.addQueryParam("IDToken1", "USER_ID");
    			httpClientWrapper.addQueryParam("IDToken2", "PASS");
    			httpClientWrapper.post("v123"); result = loginHttpResponseWrapper;
    			
    			// list-realms
    			httpClientWrapper.addQueryParam("realm", "rname");
    			httpClientWrapper.addQueryParam("filter", "123*");
    			httpClientWrapper.addQueryParam("recursive", "true");
    			httpClientWrapper.post("http://12345/?cmd=list-realms&submit="); result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK;
    			getRealmsHttpResponseWrapper.getContent(); result = "さしすせそ".getBytes("UTF-8");
    			realmSsoadmResultParser.parse("さしすせそ"); result = ssoadmResult;
    		}
    	};
    	
    	SsoadmResult actualResult = ssoadmClient.getRealms("rname", "123*", true);
    	
    	assertThat(actualResult, is(ssoadmResult));
    }

    @Test
    public void testGetRealms_HttpStatusCodeNotOK() {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	
    	new Expectations() {
    		{
    	    	// ログイン確認のPOST. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post(anyString);
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    			// list-realms
    			httpClientWrapper.addQueryParam("realm", anyString);
    			httpClientWrapper.addQueryParam("filter", anyString);
    			httpClientWrapper.addQueryParam("recursive", anyString);
    			httpClientWrapper.post(anyString); result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK + 5;
    			
            	// MessageInfo 作成される
            	new MessageInfo("I-0002");
    		}
    	};
    	
    	try {
    		ssoadmClient.getRealms("rname", "123*", true);
    		fail();
    	} catch (ApplicationSettingException e) {
    		
    	}
    }
    
    /**
     * getGroup のテスト<br>
     * 文字コードをSJISにして確認(他はすべてUTF8)<br>
     * 既にログインされていることを仮定<br>
     *
     */
    @Test
    public void testGetGroup() throws UnsupportedEncodingException {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setEncoding("Shift_JIS");
    	
    	final SsoadmResult ssoadmResult = new SsoadmResult(new ArrayList<String>(), 5);
    	
    	new Expectations() {
    		{
    	    	// ログイン確認のPOST. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post("http://12345/");
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", "rname2");
    			httpClientWrapper.addQueryParam("filter", "2*");
    			httpClientWrapper.addQueryParam("idtype", "group");
    			httpClientWrapper.post("http://12345/?cmd=list-identities&submit=");
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK;
    			getRealmsHttpResponseWrapper.getContent();
    			result = "かきくけこ".getBytes("Shift_JIS");
    			groupSsoadmResultParser.parse("かきくけこ");
    			result = ssoadmResult;
    		}
    	};
    	
    	SsoadmResult actualResult = ssoadmClient.getGroup("rname2", "2*");
    	
    	assertThat(actualResult, is(ssoadmResult));
    }

    @Test
    public void testGetGroup_HttpStatusCodeNotOK() {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	
    	new Expectations() {
    		{
    	    	// ログイン確認のPOST. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post(anyString);
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", anyString);
    			httpClientWrapper.addQueryParam("filter", anyString);
    			httpClientWrapper.addQueryParam("idtype", anyString);
    			httpClientWrapper.post(anyString);
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK + 4;
    			
            	// MessageInfo 作成される
            	new MessageInfo("I-0002");
    		}
    	};
    	
    	try {
    		ssoadmClient.getGroup("rname2", "2*");
    		fail();
    	} catch (ApplicationSettingException e) {
    		
    	}
    }
    
    @Test
    public void testGetPolicies() throws UnsupportedEncodingException {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setEncoding("UTF-8");
    	
    	final SsoadmResult ssoadmResult = new SsoadmResult(new ArrayList<String>(), 5);
    	
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post("http://12345/");
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", "rname2");
    			httpClientWrapper.addQueryParam("policynames", "2*");
    			httpClientWrapper.post("http://12345/?cmd=list-policies&submit=");
    			result = getRealmsHttpResponseWrapper;
    			
    			// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK;
    			getRealmsHttpResponseWrapper.getContent();
    			result = "あいうえお".getBytes("UTF-8");
    			policiesSsoadmResultParser.parse("あいうえお");
    			result = ssoadmResult;
    		}
    	};

    	SsoadmResult actualResult = ssoadmClient.getPolicies("rname2", "2*");
    	
    	assertThat(actualResult, is(ssoadmResult));
    }

    @Test
    public void testGetPolicies_HttpStatusCodeNotOK() {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setEncoding("UTF-8");
    	
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post(anyString);
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", anyString);
    			httpClientWrapper.addQueryParam("policynames", anyString);
    			httpClientWrapper.post(anyString);
    			result = getRealmsHttpResponseWrapper;
    			
    			// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK - 1;
    			
            	// MessageInfo 作成される
            	new MessageInfo("I-0002");
    		}
    	};
    	
    	try {
        	ssoadmClient.getPolicies("rname2", "2*");
    		fail();
    	} catch (ApplicationSettingException e) {
    		
    	}
    }
    
    @Test
    public void testCreatePolicies() throws UnsupportedEncodingException {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setEncoding("UTF-8");
    	final SsoadmResult ssoadmResult = new SsoadmResult(new ArrayList<String>(), 5);
    	
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post("http://12345/");
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = 200;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", "tttname");
    			httpClientWrapper.addQueryParam("xmlfile", "xml5");
    			httpClientWrapper.post("http://12345/?cmd=create-policies&submit=");
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK;
    			getRealmsHttpResponseWrapper.getContent();
    			result = "らりるれろ".getBytes("UTF-8");
    			policiesSsoadmResultParser.parse("らりるれろ");
    			result = ssoadmResult;
    		}
    	};
    	
    	SsoadmResult actualResult = ssoadmClient.createPolicies("tttname", "xml5");
    	
    	assertThat(actualResult, is(ssoadmResult));
    }

    @Test
    public void testCreatePolicies_HttpStatusCodeNotOK() {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post(anyString);
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", anyString);
    			httpClientWrapper.addQueryParam("xmlfile", anyString);
    			httpClientWrapper.post(anyString);
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK + 1;
    			
            	// MessageInfo 作成される
            	new MessageInfo("I-0002");
    		}
    	};
    	
    	try {
    		ssoadmClient.createPolicies("tttname", "xml5");
    		fail();
    	} catch (ApplicationSettingException e) {
    		
    	}
    }
    
    @Test
    public void testDeletePolicy() throws UnsupportedEncodingException {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	ssoadmClient.setUrlString("http://12345/");
    	ssoadmClient.setEncoding("UTF-8");
    	
    	final SsoadmResult ssoadmResult = new SsoadmResult(new ArrayList<String>(), 5);
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post("http://12345/");
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", "realmname");
    			httpClientWrapper.addQueryParam("policynames", "pol");
    			httpClientWrapper.post("http://12345/?cmd=delete-policies&submit=");
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK;
    			getRealmsHttpResponseWrapper.getContent();
    			result = "やゆよ".getBytes("UTF-8");
    			policiesSsoadmResultParser.parse("やゆよ");
    			result = ssoadmResult;
    		}
    	};
    	
    	SsoadmResult actualResult = ssoadmClient.deletePolicy("realmname", "pol");
    	
    	assertThat(actualResult, is(ssoadmResult));
    }
    
    @Test
    public void testDeletePolicy_HttpStatusCodeNotOK() {
    	ssoadmClient.setHttpClient(new HttpClientWrapper());
    	
    	new Expectations() {
    		{
    	    	// ログイン確認. ここでは, ログインされている状態であると仮定
    			httpClientWrapper.post(anyString);
    			result = loginHttpResponseWrapper;
    			loginHttpResponseWrapper.getHttpStatusCode();
    			result = HttpStatus.SC_OK;
    			
    	    	// list-realms のPOST
    			httpClientWrapper.addQueryParam("realm", anyString);
    			httpClientWrapper.addQueryParam("policynames", anyString);
    			httpClientWrapper.post(anyString);
    			result = getRealmsHttpResponseWrapper;
    			
    	    	// 実行結果パース コンテント取得
    			getRealmsHttpResponseWrapper.getHttpStatusCode(); result = HttpStatus.SC_OK + 4;
    			
            	// MessageInfo 作成される
            	new MessageInfo("I-0002");
    		}
    	};
    	
    	try {
    		ssoadmClient.deletePolicy("realmname", "pol");
    		fail();
    	} catch (ApplicationSettingException e) {
    		
    	}
    }
}
