package jp.co.ogis_ri.citk.policytool.common.excel.impl;

import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.exception.ExcelPolicyWriteException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import mockit.Expectations;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Workbook;
import org.junit.After;
import org.junit.Test;

public class WorkbookWrapperTest {
	@After
	public void tearDown() {
		File file = new File("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls");
		file.delete();
	}
	
	/**
	 * writeRow, saveAs のテスト
	 * @throws IOException 
	 */
	@Test
	public void testWriteRow_SaveAs() throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		workbookWrapper.selectSheet(0, 1, 0);
		
		List<Object> values = new ArrayList<Object>();
		values.add("あ");
		values.add(0.25d);
		values.add(false);
		workbookWrapper.writeRow(values);
		workbookWrapper.saveAs("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls", true);

		// 読み込んでみて値を確認
		HSSFWorkbook readWorkbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(readWorkbook);
		wrapper.selectSheet(0, 1, 0);
		List<?> readRow = wrapper.readRow();
		assertThat(readRow.size(), is(3));
		assertThat(readRow.get(0), is((Object)"あ"));
		assertThat(readRow.get(1), is((Object)0.25d));
		assertThat(readRow.get(2), is((Object)false));
	}

	public void testWriteRow_SystemException() throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		workbookWrapper.selectSheet(0, 1, 0);
		
		List<Object> values = new ArrayList<Object>();
		values.add(new Object());
		try {
			workbookWrapper.writeRow(values);
			fail();
		} catch (SystemException e) {
			assertTrue(e.getMessage().startsWith("not supported cell type: "));
		}
	}
	
	@Test
	public void testSaveAs_FileAlreadyExists(final MessageInfo messageInfo) throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		
		new Expectations() {
			{
				new MessageInfo("I-0403");
			}
		};
		
		String fileSavedTo = "src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls";
		try {
			workbookWrapper.saveAs(fileSavedTo, false);
		} catch (ExcelPolicyWriteException e) {
			assertThat(e.getFileName(), is(fileSavedTo));
		}
	}
	
	@Test
	public void testSaveAs_IOException(final MessageInfo messageInfo) throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		
		new Expectations() {
			{
				new MessageInfo("I-0402");
			}
		};
		
		String fileSavedTo = "src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE_readonly.xls";
		// 読み取り専用として、書き出そうとするとIOExceptionがスローされるようにする
		File file = new File(fileSavedTo);
		file.setReadOnly();
		try {
			workbookWrapper.saveAs(fileSavedTo, true);
		} catch (ExcelPolicyWriteException e) {
			assertThat(e.getFileName(), is(fileSavedTo));
		}
	}
	
	@Test
	public void testSelectSheetIntIntInt() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		
		wrapper.selectSheet(1, 0, 2);
		List<?> row = wrapper.readRow();
		assertThat(row.size(), is(2));
		assertThat(row.get(0), is((Object)"千葉"));
		assertThat(row.get(1), is((Object)"埼玉"));
	}

	@Test
	public void testSelectSheetStringIntInt() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		
		wrapper.selectSheet("new", 1, 3);
		List<?> row = wrapper.readRow();
		assertThat(row.size(), is(2));
		assertThat(row.get(0), is((Object)"山梨"));
		assertThat(row.get(1), is((Object)"長野"));
	}

	@Test
	public void testExistRow_TRUE() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		
		boolean actual = wrapper.existRow();
		assertThat(actual, is(true));
	}

	@Test
	public void testExistRow_FALSE_currentRowNULL() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		wrapper.selectSheet(1, 0, 0);
		wrapper.readRow();
		wrapper.readRow();
		wrapper.readRow();
		
		boolean actual = wrapper.existRow();
		assertThat(actual, is(false));
	}

	@Test
	public void testExistRow_FALSE_cellValueNULL() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		wrapper.selectSheet(2, 1, 1);
		
		boolean actual = wrapper.existRow();
		assertThat(actual, is(false));
	}

	@Test
	public void testReadRow() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		List<?> row1 = wrapper.readRow();
		List<?> row2 = wrapper.readRow();
		
		assertThat(row1.size(), is(3));
		assertThat(row1.get(0), is((Object)true));
		assertThat(row1.get(1), is((Object)0.5d));
		assertThat(row1.get(2), is((Object)"文字"));
		
		assertThat(row2.size(), is(4));
		assertThat(row2.get(0), is((Object)"あ"));
		assertThat(row2.get(1), is((Object)"い"));
		assertThat(row2.get(2), is((Object)"う"));
		assertThat(row2.get(3), is((Object)"え"));
	}

	@Test
	public void testReadRow_SystemException() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		wrapper.readRow();
		wrapper.readRow();
		try {
			wrapper.readRow();
			fail();
		} catch (SystemException e) {
			assertTrue(e.getMessage().startsWith("not supported cell type: "));
		}
	}
	
	@Test
	public void testReadRowInt() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		List<?> row1 = wrapper.readRow(2);
		List<?> row2 = wrapper.readRow(2);
		
		assertThat(row1.size(), is(2));
		assertThat(row1.get(0), is((Object)true));
		assertThat(row1.get(1), is((Object)0.5d));
		
		assertThat(row2.size(), is(2));
		assertThat(row2.get(0), is((Object)"あ"));
		assertThat(row2.get(1), is((Object)"い"));
	}

	@Test
	public void testGetCellValueIntIntInt() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		
		Object actual = wrapper.getCellValue(1, 2, 3);
		assertThat(actual, is((Object)"岐阜"));
	}

	@Test
	public void testGetCellValueStringIntInt() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		
		Object actual = wrapper.getCellValue("new", 2, 4);
		assertThat(actual, is((Object)"愛知"));
	}

	/**
	 * writeRow, saveAs のテスト
	 * @throws IOException 
	 */
	@Test
	public void testSetCellValue() throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		workbookWrapper.setCellValue(1, 2, 3, "ABC");
		workbookWrapper.saveAs("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls", true);

		// 読み込んでみて値を確認
		HSSFWorkbook readWorkbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls");
		WorkbookWrapper readWrapper = new WorkbookWrapper(readWorkbook);
		Object readCellValue = readWrapper.getCellValue(1, 2, 3);
		assertThat(readCellValue, is((Object)"ABC"));
	}

	/**
	 * writeRow, saveAs のテスト
	 * @throws IOException 
	 */
	@Test
	public void testSetCellValue_StringIntIntObject() throws IOException {
		InputStream template = new FileInputStream("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xlt");
		Workbook workbook = new HSSFWorkbook(template);
		WorkbookWrapper workbookWrapper = new WorkbookWrapper(workbook);
		workbookWrapper.setCellValue("a2", 2, 3, "DEF");
		workbookWrapper.saveAs("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls", true);

		// 読み込んでみて値を確認
		HSSFWorkbook readWorkbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest_SAVE.xls");
		WorkbookWrapper readWrapper = new WorkbookWrapper(readWorkbook);
		Object readCellValue = readWrapper.getCellValue(1, 2, 3);
		assertThat(readCellValue, is((Object)"DEF"));
	}

	@Test
	public void testResetCurrent() throws IOException {
		HSSFWorkbook workbook = WorkbookWrapper.createHSSFWorkbook("src/test/resources/jp/co/ogis_ri/citk/policytool/common/excel/impl/WorkbookWrapperTest.xls");
		WorkbookWrapper wrapper = new WorkbookWrapper(workbook);
		List<?> row1 = wrapper.readRow();
		wrapper.resetCurrent();
		List<?> row2 = wrapper.readRow();
		
		assertThat(row1.size(), is(3));
		assertThat(row1.get(1), is((Object)0.5d));
		
		// もう1回同じ行が読まれていることを確認
		assertThat(row2.size(), is(3));
		assertThat(row2.get(1), is((Object)0.5d));
	}
}
