package jp.co.ogis_ri.citk.policytool.common.http;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.exception.ApplicationException;
import jp.co.ogis_ri.citk.policytool.common.exception.SystemException;
import jp.co.ogis_ri.citk.policytool.common.resource.MessageInfo;
import jp.co.ogis_ri.citk.policytool.common.util.ExceptionUtil;

import mockit.Deencapsulation;
import mockit.Delegate;
import mockit.Expectations;
import mockit.Mock;
import mockit.MockClass;
import mockit.Mocked;
import mockit.Mockit;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.ProtocolVersion;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.AbstractHttpClient;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.SingleClientConnManager;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.message.BasicStatusLine;
import org.junit.Test;

public class HttpClientWrapperTest {
    private HttpClientWrapper httpClientWrapper = new HttpClientWrapper();
    
    @Mocked
	private SingleClientConnManager connManager;
	
    @SuppressWarnings("unchecked")
    @Test
    public void testAddQueryParam() {
        List<NameValuePair> queryParams = (List<NameValuePair>)Deencapsulation.getField(httpClientWrapper, "queryParams");
        queryParams.clear();
        
        httpClientWrapper.addQueryParam("TestName", "TestValue");
        
        NameValuePair nameValuePair = queryParams.get(0);
        assertThat(nameValuePair.getName(), is("TestName"));
        assertThat(nameValuePair.getValue(), is("TestValue"));
    }

    @SuppressWarnings("unchecked")
    @Test
    public void testClearQueryParam() {
        List<NameValuePair> queryParams = (List<NameValuePair>)Deencapsulation.getField(httpClientWrapper, "queryParams");
        queryParams.add(new BasicNameValuePair("abc", "123"));
        
        httpClientWrapper.clearQueryParam();
        
        assertThat(queryParams.size(), is(0));
    }

    @Test
    public void testAddCookie(final DefaultHttpClient httpClient) {
        final CookieStore cookieStore = new BasicCookieStore();
        final Cookie cookie = new BasicClientCookie("name", "val");
        
        new Expectations() {
            {
                httpClient.getCookieStore();
                result = cookieStore;
            }
        };
        httpClientWrapper.addCookie(cookie);
        
        assertThat(cookieStore.getCookies().size(), is(1));
        assertThat(cookieStore.getCookies().get(0), is(cookie));
    }

    @Test
    public void testClearCookieStore(final DefaultHttpClient httpClient) {
        final CookieStore cookieStore = new BasicCookieStore();
        final Cookie cookie = new BasicClientCookie("name", "val");
        cookieStore.addCookie(cookie);
        
        new Expectations() {
            {
                httpClient.getCookieStore();
                result = cookieStore;
            }
        };
        httpClientWrapper.clearCookieStore();
        
        assertThat(cookieStore.getCookies().size(), is(0));
    }

    @Test
    public void testShutdown(final DefaultHttpClient httpClient) {
    	new Expectations() {
    		{
    			httpClient.getConnectionManager();
    			result = connManager;
    			connManager.shutdown();
    			result = null;
    		}
    	};
    	
    	httpClientWrapper.shutdown();
    }
    
    @Test
    public void testGetString() throws IOException {
        Mockit.setUpMock(GetAbstractHttpClientMock.class);
        
        httpClientWrapper = new HttpClientWrapper();
        HttpResponseWrapper httpResponseWrapper = httpClientWrapper.get("https://www.example.com:80/ssoadm?p=1&ppp=so");
        assertThat(httpResponseWrapper, notNullValue());
        
        Mockit.tearDownMocks(GetAbstractHttpClientMock.class);
    }

    @Test
    public void testGetString_MalformedURLException(final URL url, final ExceptionUtil exceptionUtil) throws IOException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		new URL(anyString);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void construct() throws MalformedURLException {
        				throw new MalformedURLException();
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        	}
        };
        
        try {
        	httpClientWrapper.get("https://www.example.com:80/ssoadm?p=1&ppp=so");
        	fail();
        } catch (SystemException e) {
        	
        }
    }
    
    @Test
    public void testGetString_URISyntaxException(final URIUtils uriUtils, final ExceptionUtil exceptionUtil) throws URISyntaxException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		URIUtils.createURI(anyString, anyString, anyInt, anyString, anyString, anyString);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void construct() throws URISyntaxException {
        				throw new URISyntaxException("", "");
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        	}
        };
        
        try {
        	httpClientWrapper.get("https://www.example.com:80/ssoadm?p=1&ppp=so");
        	fail();
        } catch (SystemException e) {
        	
        }
    }
    
    @Test
    public void testGetString_ClientProtocolException(final DefaultHttpClient httpClient, final MessageInfo messageInfo) throws IOException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		httpClient.execute((HttpUriRequest) any);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					HttpResponse execute(HttpUriRequest req) throws ClientProtocolException {
        				throw new ClientProtocolException();
        			}
        		};

        		new MessageInfo("I-0005");
        	}
        };
        
        try {
        	httpClientWrapper.get("https://www.example.com:80/ssoadm?p=1&ppp=so");
        	fail();
        } catch (ApplicationException e) {
        	
        }
    }
    
    @Test
    public void testGetString_IOException(final DefaultHttpClient httpClient, final MessageInfo messageInfo) throws IOException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		httpClient.execute((HttpUriRequest) any);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					HttpResponse execute(HttpUriRequest req) throws IOException {
        				throw new IOException();
        			}
        		};
        		
        		new MessageInfo("I-0006");
        	}
        };
        
        try {
        	httpClientWrapper.get("https://www.example.com:80/ssoadm?p=1&ppp=so");
        	fail();
        } catch (ApplicationException e) {
        	
        }
    }
    
    @Test
    public void testGetURL() throws IOException {
        Mockit.setUpMock(GetAbstractHttpClientMock.class);
        
        httpClientWrapper = new HttpClientWrapper();
        URL url = new URL("https://www.example.com:80/ssoadm?p=1&ppp=so");
        HttpResponseWrapper httpResponseWrapper = httpClientWrapper.get(url);
        assertThat(httpResponseWrapper, notNullValue());
        
        Mockit.tearDownMocks(GetAbstractHttpClientMock.class);
    }

    @Test
    public void testPostString() {
        Mockit.setUpMock(PostAbstractHttpClientMock.class);
    	
        httpClientWrapper = new HttpClientWrapper();
        HttpResponseWrapper httpResponseWrapper = httpClientWrapper.post("http://www.example.jp:8080/ssoadm2?p=2&ppp=sb");
        assertThat(httpResponseWrapper, notNullValue());
  
        Mockit.tearDownMocks(PostAbstractHttpClientMock.class);
    }
    
    @SuppressWarnings("unchecked")
	@Test
    public void testPostString_UnsupportedEncodingException(final UrlEncodedFormEntity entity, final ExceptionUtil exceptionUtil) throws UnsupportedEncodingException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		new UrlEncodedFormEntity((List<? extends NameValuePair>) any, (String) any);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void construct(final List <? extends NameValuePair> parameters,
					        final String encoding) throws UnsupportedEncodingException {
        				throw new UnsupportedEncodingException();
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        	}
        };
    	
        try {
        	httpClientWrapper.post("http://www.example.jp:8080/ssoadm2?p=2&ppp=sb");
        	fail();
        } catch (SystemException e) {
        	
        }
    }
    
    @Test
    public void testPostURI() throws URISyntaxException {
        Mockit.setUpMock(PostAbstractHttpClientMock.class);
    	
        httpClientWrapper = new HttpClientWrapper();
        URI uri = new URI("http://www.example.jp:8080/ssoadm2?p=2&ppp=sb");
        HttpResponseWrapper httpResponseWrapper = httpClientWrapper.post(uri);
        assertThat(httpResponseWrapper, notNullValue());
  
        Mockit.tearDownMocks(PostAbstractHttpClientMock.class);
    }
    
    @SuppressWarnings("unchecked")
	@Test
    public void testPostURI_UnsupportedEncodingException(final UrlEncodedFormEntity entity, final ExceptionUtil exceptionUtil) throws UnsupportedEncodingException, URISyntaxException {
        httpClientWrapper = new HttpClientWrapper();
        
        new Expectations() {
        	{
        		new UrlEncodedFormEntity((List<? extends NameValuePair>) any, (String) any);
        		result = new Delegate() {
        			@SuppressWarnings("unused")
					void construct(final List <? extends NameValuePair> parameters,
					        final String encoding) throws UnsupportedEncodingException {
        				throw new UnsupportedEncodingException();
        			}
        		};
        		
        		ExceptionUtil.convertRuntimeException((Exception)any);
        		result = new SystemException();
        	}
        };
    	
        try {
        	httpClientWrapper.post(new URI("http://www.example.jp:8080/ssoadm2?p=2&ppp=sb"));
        	fail();
        } catch (SystemException e) {
        	
        }
    }
    
    @MockClass(realClass = AbstractHttpClient.class)
    public static class GetAbstractHttpClientMock {
        @Mock
        public final HttpResponse execute(HttpUriRequest request)
                throws IOException, ClientProtocolException {
            StatusLine statusLine = new BasicStatusLine(new ProtocolVersion("A", 123, 456), 92, "rrr");
            
            HttpResponse httpResponse = new BasicHttpResponse(statusLine);
            httpResponse.setEntity(new StringEntity("xyz"));

            // リクエストの確認
            URI uri = request.getURI();
            assertThat(uri.getScheme(), is("https"));
            assertThat(uri.getHost(), is("www.example.com"));
            assertThat(uri.getPort(), is(80));
            assertThat(uri.getPath(), is("/ssoadm"));
            assertThat(uri.getQuery(), is("p=1&ppp=so"));
            
            HttpGet httpGet = (HttpGet)request;
            assertThat(httpGet, notNullValue());
            
            return httpResponse;
        }        
        
        @Mock
        public synchronized final CookieStore getCookieStore() {
            
            final CookieStore cookieStore = new BasicCookieStore();
            final Cookie cookie = new BasicClientCookie("name", "val");
            cookieStore.addCookie(cookie);
            
            return cookieStore;
        }
    }
    
    @MockClass(realClass = AbstractHttpClient.class)
    public static class PostAbstractHttpClientMock {
        @Mock
        public final HttpResponse execute(HttpUriRequest request)
                throws IOException, ClientProtocolException {
            StatusLine statusLine = new BasicStatusLine(new ProtocolVersion("A", 123, 456), 92, "rrr");
            
            HttpResponse httpResponse = new BasicHttpResponse(statusLine);
            httpResponse.setEntity(new StringEntity("xyz"));

            // リクエストの確認
            URI uri = request.getURI();
            assertThat(uri.getScheme(), is("http"));
            assertThat(uri.getHost(), is("www.example.jp"));
            assertThat(uri.getPort(), is(8080));
            assertThat(uri.getPath(), is("/ssoadm2"));
            assertThat(uri.getQuery(), is("p=2&ppp=sb"));
            
            HttpPost httpPost = (HttpPost)request;
            assertThat(httpPost, notNullValue());
            
            UrlEncodedFormEntity entity = (UrlEncodedFormEntity)httpPost.getEntity();
            assertThat(entity.getContentType().getValue(), is("application/x-www-form-urlencoded; charset=UTF-8"));
            
            return httpResponse;
        }        
        
        @Mock
        public synchronized final CookieStore getCookieStore() {
            
            final CookieStore cookieStore = new BasicCookieStore();
            final Cookie cookie = new BasicClientCookie("name", "val");
            cookieStore.addCookie(cookie);
            
            return cookieStore;
        }
    }
}
