package jp.co.ogis_ri.citk.policytool.domain.realm.impl;

import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.hamcrest.core.IsNull.nullValue;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertThat;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import jp.co.ogis_ri.citk.policytool.common.api.OpenAMAccess;
import jp.co.ogis_ri.citk.policytool.domain.realm.RealmRepository;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Group;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Realm;
import jp.co.ogis_ri.citk.policytool.domain.realm.model.Referral;
import jp.co.ogis_ri.citk.policytool.domain.util.DbUnitDataLoader;
import mockit.Expectations;
import mockit.Mocked;

import org.apache.commons.dbcp.BasicDataSource;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.transaction.TransactionConfiguration;
import org.springframework.transaction.annotation.Transactional;

@RunWith(SpringJUnit4ClassRunner.class)
@ContextConfiguration({"JpaRealmRepositoryImplTest-context.xml"})
@TransactionConfiguration(defaultRollback=true)
@Transactional
public class JpaRealmRepositoryImplTest {
	@Autowired
	BasicDataSource datasource;
	
	@Autowired
	private RealmRepository repository;
	
	@Mocked(capture=1)
	OpenAMAccess amAccess = null;

	@Before
	public void setUp() throws Exception {
        InputStream testData = Realm.class.getResourceAsStream("/realm.db.xml");
        DbUnitDataLoader loader = new DbUnitDataLoader(testData, datasource.getConnection());
		loader.populateTestData();
	}

	@Test
	public void testSync_SUCCESS() {
		final List<Realm> realms = new ArrayList<Realm>();

		new Expectations() {{
			amAccess.getRealms("/", "*"); result=realms;
		}};
        repository.sync();
        assertThat(repository.findAll().size(), is(0));

        Realm realm = createSyncRealm();
        realms.add(realm);
        new Expectations() {{
			amAccess.getRealms("/", "*"); result=realms;
		}};
        repository.sync();
        assertThat(repository.findAll().size(), is(1));
	}

	@Test
	public void testFindById_SUCCESS() {
        Realm actual = repository.findById(100L);
        assertThat(actual, is(notNullValue()));
        assertThat(actual.getId(), is(100L));
        assertThat(actual.getReferrals().size(), is(1));
        assertThat(actual.getGroups().size(), is(2));
	}

	@Test
	public void testFindById_SUCCESS_NOT_FOUND() {
		Realm actual = repository.findById(99999L);
		assertThat(actual, is(nullValue()));
	}
    @Test
    public void testFindAll_SUCCESS() {
        List<Realm> actual = repository.findAll();
        assertThat(actual.size(), is(2));
    }
    
	public void testPersist_SUCCESS() {
	    Realm realm = new Realm("new_sp");
	    realm.addGroup(new Group("g1","g1code"));
        realm.addReferral(new Referral("http://"));
        repository.persist(realm);

        final Realm actual = repository.findById(realm.getId());

        assertThat(actual.getId(), is(realm.getId()));
        assertThat(actual.getReferrals().size(), is(realm.getReferrals().size()));
        assertThat(actual.getGroups().size(), is(realm.getGroups().size()));
 	}

	@Test
	public void testMerge_SUCCESS() {
        Realm realm = repository.findById(100L);

        realm.addGroup(new Group("g1", "o=g1"));
        realm.addReferral(new Referral("http://www.example.com/"));
        repository.merge(realm);

        final Realm actual = repository.findById(100L);

        assertThat(actual.getId(), is(realm.getId()));
        assertThat(actual.getReferrals().size(), is(realm.getReferrals().size()));
        assertThat(actual.getGroups().size(), is(realm.getGroups().size()));
	}

    @Test
    public void testRemove_SUCCESS() {
        Realm realm = repository.findById(100L);
        repository.remove(realm);

        final Realm actual = repository.findById(100L);
        assertThat(actual, is(nullValue()));
    }
    
    @Test
    public void testRemoveAll_SUCCESS() {
    	List<Realm> realms = repository.findAll();
    	assertNotSame(realms.size(), 0);
    	
    	repository.removeAll();
    	
    	List<Realm> actual = repository.findAll();
    	assertThat(actual.size(), is(0));
    }
    
    @Test
    public void testFindGroupNames_SUCCESS() {
    	List<String> actual = repository.findGroupNames(null);
    	assertThat(actual.size(), is(2));
    }
    
    @Test
    public void testFindReferrals_SUCCESS() {
    	List<Referral> actual = repository.findReferrals(null);
    	assertThat(actual.size(), is(2));
    }
    
	/**
	 * Sync テスト用 Realm 作成.
	 * @return Sync テスト用 Realm.
	 */
	private Realm createSyncRealm() {
        Realm realm = new Realm("realm1");
        Group group = new Group();
        group.setGroupName("ggg");
        group.setGroupCode("o=ggg,i=1");
        group.setRealm(realm);
        Referral referral = new Referral();
        referral.setRefpolicy("http://www.example.com/");
        referral.setRealm(realm);
        realm.addGroup(group);
        realm.addReferral(referral);
        
        return realm;
	}
}
