﻿using System;
using System.Text;
using System.Collections.Generic;
using System.IO;
using System.Security.AccessControl;
using System.Security.Principal;
using System.Linq;
using Trinet.Core.IO.Ntfs;
using CompleteEraser.Properties;

namespace CompleteEraser
{
    class FileBreaker
    {
        const long ShirnkLength = 1;

        public static void BreakFileOrFolder(string path)
        {
            if (Directory.Exists(path))
                BreakFolder(path);
            else if (File.Exists(path))
                BreakFile(path);
        }

        public static void BreakFile(string file)
        {
            FileAttributes attr = File.GetAttributes(file);

            if ((attr & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)
                File.SetAttributes(file, attr & ~FileAttributes.ReadOnly);

            if ((attr & FileAttributes.ReparsePoint) != FileAttributes.ReparsePoint &&
                (attr & FileAttributes.SparseFile) != FileAttributes.SparseFile)
                FillFileContent(file);
            
            string newFile = GenerateRandoName(Path.GetDirectoryName(file), Path.GetFileName(file).Length);
            if(File.Exists(newFile) == false)
                File.Move(file, newFile);
            
            File.Delete(newFile);
        }

        private static void FillFileContent(string file)
        {
            FileStream fs;
            FileInfo info = new FileInfo(file);

            foreach (AlternateDataStreamInfo alt in info.ListAlternateDataStreams())
            {
                fs = alt.Open(FileMode.Open);
                BreakFileSlowSpeed(fs);
                fs.Close();
            }

            string ext = Path.GetExtension(file);

            fs = new FileStream(file, FileMode.Open, FileAccess.Write, FileShare.None, 8, FileOptions.WriteThrough);
            if (Settings.Default.AlwaysSlowSpeed || Settings.Default.SlowSpeedExt.IndexOf(ext) != -1)
                BreakFileSlowSpeed(fs);
            else
                BreakFileHiSpeed(fs);
            fs.Close();
        }

        private static void BreakFileSlowSpeed(FileStream fs)
        {
            fs.Seek(0, SeekOrigin.Begin);
            byte[] data = new byte[Settings.Default.FillLengthAtHiSpeed];
            for (long i = 0; i < fs.Length; i += data.Length)
                fs.Write(data,0,data.Length);
            fs.Flush(true);
            fs.SetLength(ShirnkLength);
        }

        private static void BreakFileHiSpeed(FileStream fs)
        {
            fs.Seek(0, SeekOrigin.Begin);
            byte[] data = new byte[Settings.Default.FillLengthAtHiSpeed];
            fs.Write(data, 0, data.Length);
            fs.Seek(-data.Length, SeekOrigin.End);
            fs.Write(data, 0, data.Length);
            fs.Flush(true);
            fs.SetLength(ShirnkLength);
        }

        public static void BreakFolder(string folder)
        {
            DirectoryInfo info = new DirectoryInfo(folder);
            if ((info.Attributes & FileAttributes.ReparsePoint) != FileAttributes.ReparsePoint)
            {
                IEnumerable<string> files = Directory.EnumerateFiles(folder, "*.*", SearchOption.TopDirectoryOnly);
                foreach (string file in files)
                    BreakFile(file);
                IEnumerable<string> dirs = Directory.EnumerateDirectories(folder, "*.*", SearchOption.TopDirectoryOnly);
                foreach (string dir in dirs)
                    BreakFolder(dir);
            }
            string newName = RenameFolderName(folder);
            Directory.Delete(newName);
        }

        public static string RenameFolderName(string dir)
        {
            string newName = GenerateRandoName(Path.GetDirectoryName(dir), Path.GetFileName(dir).Length);
            if (Directory.Exists(newName))
                return dir;
            if ((FileAccessPermissionHelper.GetCurrentAccessRule(dir).FileSystemRights & FileSystemRights.Modify) == FileSystemRights.Modify)
                Directory.Move(dir, newName);
            else
                throw new UnauthorizedAccessException();
            return newName;
        }
        
        private static string GenerateRandoName(string dirpath, int length)
        {
            string list = "abcdefghijklmnopqrstuvwxyz0123456789";
            Random rnd = new Random();
            StringBuilder output = new StringBuilder();
            for (int i = 0; i < length; i++)
                output.Append(list[rnd.Next(list.Length - 1)]);
            return dirpath + "\\" + output.ToString();
        }
    }

    public static class FileAccessPermissionHelper
    {
        // 書き込み権限持ってる？
        public static bool IsGotPermission(string path)
        {
            var rule = GetCurrentAccessRule(path);
            return ((rule != null) && ((rule.FileSystemRights & FileSystemRights.Write) == FileSystemRights.Write));
        }

        // 現在のユーザーが持っている指定パスのFileSystemAccessRuleを得る
        public static FileSystemAccessRule GetCurrentAccessRule(string path)
        {
            var fileSecurity = File.GetAccessControl(path);
            var rules = fileSecurity.GetAccessRules(true, true, typeof(SecurityIdentifier)).OfType<FileSystemAccessRule>();
            var currentIdentity = WindowsIdentity.GetCurrent();
            var sids = new[] { currentIdentity.User }.Concat(currentIdentity.Groups);

            // アクセスルール内にユーザーSIDがある？無ければグループSIDも探す
            return rules.FirstOrDefault(rule => sids.Contains(rule.IdentityReference));
        }
    }
}
