package jp.cssj.sakae.font.otf;

import java.io.IOException;

import jp.cssj.cr.compat.XMatrix;
import jp.cssj.cr.compat.XPath;
import jp.cssj.sakae.font.FontSource;
import jp.cssj.sakae.font.ShapedFont;
import jp.cssj.sakae.gc.GC;
import jp.cssj.sakae.gc.GraphicsException;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.font.util.FontUtils;
import jp.cssj.sakae.gc.text.Text;
import net.zamasoft.font.Glyph;
import net.zamasoft.font.table.Feature;
import net.zamasoft.font.table.FeatureList;
import net.zamasoft.font.table.FeatureTags;
import net.zamasoft.font.table.GsubTable;
import net.zamasoft.font.table.LangSys;
import net.zamasoft.font.table.Lookup;
import net.zamasoft.font.table.LookupList;
import net.zamasoft.font.table.Script;
import net.zamasoft.font.table.ScriptList;
import net.zamasoft.font.table.ScriptTags;
import net.zamasoft.font.table.SingleSubst;
import net.zamasoft.font.table.Table;
import net.zamasoft.font.table.XmtxTable;
import android.graphics.RectF;

public class OpenTypeFont implements ShapedFont {
	private static final long serialVersionUID = 2L;

	protected static final boolean ADJUST_VERTICAL = false;

	protected static final int DEFAULT_VERTICAL_ORIGIN = 880;

	protected final OpenTypeFontSource source;

	transient protected SingleSubst vSubst;

	transient protected XmtxTable vmtx, hmtx;

	protected OpenTypeFont(OpenTypeFontSource source) {
		this.source = source;
	}

	private void init() {
		if (this.hmtx != null) {
			return;
		}
		net.zamasoft.font.OpenTypeFont ttfFont = this.source.getOpenTypeFont();
		this.hmtx = (XmtxTable) ttfFont.getTable(Table.hmtx);
		if (this.source.getDirection() == FontStyle.DIRECTION_TB) {
			// 縦書モード
			{
				GsubTable gsub = (GsubTable) ttfFont.getTable(Table.GSUB);
				ScriptList scriptList = gsub.getScriptList();
				Script script = scriptList
						.findScript(ScriptTags.SCRIPT_TAG_KANA);
				if (script == null) {
					script = scriptList.findScript(ScriptTags.SCRIPT_TAG_HANI);
				}
				if (script == null) {
					script = scriptList.findScript(ScriptTags.SCRIPT_TAG_LATN);
				}
				LangSys langSys = script.getDefaultLangSys();
				FeatureList featureList = gsub.getFeatureList();
				Feature feature = featureList.findFeature(langSys,
						FeatureTags.FEATURE_TAG_VERT);
				if (feature != null) {
					LookupList lookupList = gsub.getLookupList();
					Lookup lookup = lookupList.getLookup(feature, 0);
					this.vSubst = (SingleSubst) lookup.getSubtable(0);
					this.vmtx = (XmtxTable) ttfFont.getTable(Table.vmtx);
					return;
				}
			}
		}
		this.vSubst = null;
		this.vmtx = null;

	}

	protected final boolean isVertical() {
		this.init();
		return this.vSubst != null;
	}

	protected final XPath adjustShape(XPath shape, double advance) {
		RectF bounds = new RectF();
		shape.computeBounds(bounds, false);
		double bottom = bounds.bottom + DEFAULT_VERTICAL_ORIGIN;
		if (bottom > advance) {
			// 字面が衝突しないように調整する
			XPath path = new XPath(shape);
			path.transform(XMatrix.getTranslateInstance(0, advance - bottom));
			shape = path;
		}
		return shape;
	}

	protected final short getHAdvance(int gid) {
		final OpenTypeFontSource source = (OpenTypeFontSource) this
				.getFontSource();
		this.init();
		final short advance = (short) (this.hmtx.getAdvanceWidth(gid)
				* FontSource.UNITS_PER_EM / source.getUnitsPerEm());
		return advance;
	}

	protected final short getVAdvance(int gid) {
		this.init();
		if (this.vmtx == null) {
			return (short) FontSource.UNITS_PER_EM;
		}
		final OpenTypeFontSource source = (OpenTypeFontSource) this
				.getFontSource();
		final short advance = (short) (this.vmtx.getAdvanceWidth(gid)
				* FontSource.UNITS_PER_EM / source.getUnitsPerEm());
		return advance;
	}

	public FontSource getFontSource() {
		return this.source;
	}

	public int toGID(int c) {
		OpenTypeFontSource source = (OpenTypeFontSource) this.getFontSource();
		int gid = source.getCmapFormat().mapCharCode(c);
		this.init();
		if (this.vSubst != null) {
			gid = this.vSubst.substitute(gid);
		}
		return gid;
	}

	public XPath getShapeByGID(int gid) {
		OpenTypeFontSource source = (OpenTypeFontSource) this.getFontSource();
		Glyph glyph = source.getOpenTypeFont().getGlyph(gid);
		if (glyph == null) {
			return null;
		}
		XPath shape = glyph.getPath();
		if (ADJUST_VERTICAL) {
			if (this.isVertical()) {
				double advance = this.getAdvance(gid);
				shape = this.adjustShape(shape, advance);
			}
		}
		return shape;
	}

	public short getAdvance(int gid) {
		if (this.isVertical()) {
			return this.getVAdvance(gid);
		}
		return this.getHAdvance(gid);
	}

	public short getWidth(int gid) {
		return this.getHAdvance(gid);
	}

	public void drawTo(GC gc, Text text) throws IOException, GraphicsException {
		FontUtils.drawPDFFont(gc, this, text);
	}

	public short getKerning(int scid, int cid) {
		return 0;
	}

	public int getLigature(int scid, int cid) {
		return -1;
	}
}