package jp.cssj.sakae.font;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import jp.cssj.sakae.gc.font.FontFace;
import jp.cssj.sakae.gc.font.FontListMetrics;
import jp.cssj.sakae.gc.font.FontManager;
import jp.cssj.sakae.gc.font.FontMetrics;
import jp.cssj.sakae.gc.font.FontStyle;
import jp.cssj.sakae.gc.text.GlyphHandler;
import jp.cssj.sakae.gc.text.Glypher;
import jp.cssj.sakae.gc.text.Quad;

/**
 * Implementation of FontManager for PDF.
 * 
 * @author <a href="mailto:tatsuhiko at miya.be">MIYABE Tatsuhiko </a>
 * @version $Id: PDFFontManager.java,v 1.1 2007-05-06 15:37:19 miyabe Exp $
 */
public class FontManagerImpl implements FontManager {
	private static final long serialVersionUID = 1L;

	protected final FontSourceManager globaldb;

	protected final FontStore fontStore;

	protected final Map fontListMetricsCache = new HashMap();

	public FontManagerImpl(FontSourceManager fontdb, FontStore fontStore) {
		this.globaldb = fontdb;
		this.fontStore = fontStore;
	}

	public FontManagerImpl(FontSourceManager fontdb) {
		this(fontdb, new DefaultFontStore());
	}

	public void addFontFace(FontFace face) throws IOException {
		// ignore
	}

	public FontListMetrics getFontListMetrics(FontStyle fontStyle) {
		FontListMetrics flm = (FontListMetrics) this.fontListMetricsCache
				.get(fontStyle);
		if (flm != null) {
			return flm;
		}
		FontSource[] fonts = this.globaldb.lookup(fontStyle);
		FontMetrics[] fms = new FontMetrics[fonts.length];
		for (int i = 0; i < fonts.length; ++i) {
			fms[i] = new FontMetricsImpl(this.fontStore, fonts[i], fontStyle);
		}
		flm = new FontListMetrics(fms);
		this.fontListMetricsCache.put(fontStyle, flm);
		return flm;
	}

	public Glypher getGlypher() {
		return new CharacterHandler();
	}

	protected class CharacterHandler implements Glypher {
		private GlyphHandler glyphHandler;

		private final char[] ch = new char[3];

		private int off;

		private byte len;

		private int gid;

		private FontListMetrics fontListMetrics = null;

		private int fontBound = 0;

		private FontStyle fontStyle;

		private FontMetricsImpl fontMetrics = null;

		private int pgid = -1;

		private boolean outOfRun = true;

		public CharacterHandler() {
			// ignore
		}

		public void setGlyphHander(GlyphHandler glyphHandler) {
			this.glyphHandler = glyphHandler;
		}

		public void fontStyle(FontStyle fontStyle) {
			this.glyphBreak();
			this.fontListMetrics = null;
			this.fontStyle = fontStyle;
		}

		private void glyphBreak() {
			if (!this.outOfRun) {
				this.glyph();
				this.endRun();
			}
		}

		public void characters(char[] ch, int off, int len) {
			if (this.fontListMetrics == null) {
				this.fontListMetrics = FontManagerImpl.this
						.getFontListMetrics(this.fontStyle);
				this.initFont();
			}

			for (int k = 0; k < len; ++k) {
				char c = ch[k + off];

				// \A0は空白に変換
				char cc = c == '\u00A0' ? '\u0020' : c;

				// ランの範囲を作成
				if (!this.fontMetrics.canDisplay(cc)) {
					this.glyphBreak();
					this.initFont();
				}

				// 利用可能な文字を検出
				for (int j = 0; j < this.fontBound; ++j) {
					FontMetricsImpl metrics = (FontMetricsImpl) this.fontListMetrics
							.getFontMetrics(j);
					if (metrics.canDisplay(cc)) {
						this.glyphBreak();
						this.fontMetrics = (FontMetricsImpl) this.fontListMetrics
								.getFontMetrics(j);
						this.fontBound = j;
						break;
					}
				}

				if (this.outOfRun) {
					this.outOfRun = false;
					this.gid = -1;
					this.pgid = -1;
					this.len = 0;
					this.glyphHandler.startTextRun(this.fontStyle,
							this.fontMetrics);
				}

				// 通常の文字として処理
				Font font = this.fontMetrics.getFont();
				int gid = font.toGID(cc);

				// 連字のチェック
				int lgid = this.fontMetrics.getLigature(this.gid, gid);
				if (lgid != -1) {
					// 連字にできる
					this.gid = this.pgid;
					gid = lgid;
					this.ch[this.len] = c;
					++this.len;
				} else {
					// 連字にできない
					if (this.gid != -1) {
						this.glyph();
					}
					this.ch[0] = c;
					this.len = 1;
				}
				this.pgid = this.gid;
				this.gid = gid;
			}
		}

		public void quad(Quad quad) {
			this.glyphBreak();
			this.glyphHandler.quad(quad);
		}

		private void glyph() {
			this.glyphHandler.glyph(this.ch, this.off, this.len, this.gid);
		}

		private void endRun() {
			assert this.outOfRun == false;
			this.glyphHandler.endTextRun();
			this.outOfRun = true;
		}

		private void initFont() {
			this.fontBound = this.fontListMetrics.getLength();
			this.fontMetrics = (FontMetricsImpl) this.fontListMetrics
					.getFontMetrics(this.fontBound - 1);
		}

		public void flush() {
			this.glyphBreak();
			this.glyphHandler.flush();
		}
	}
}