/*$Id$*/
package nicobrowser;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.Persistence;
import nicobrowser.entity.NicoContent;
import nicobrowser.entity.NicoContent.Status;
import nicobrowser.search.SearchKind;
import nicobrowser.search.SearchResult;
import nicobrowser.search.SortKind;
import nicobrowser.search.SortOrder;
import org.apache.http.HttpException;
import org.junit.After;
import org.junit.AfterClass;
import static org.junit.Assert.*;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 *
 * @author yuki
 */
public class NicoHttpClientTest {

    static final String OK_MAIL = "niconico.senyou@live.jp";
    static final String OK_PASS = "piyopiyo";
    static final String OK_LIST_NO = "1802157";
    static final String NORMAL_VIDEO = "sm1359820";
    static final String OFFICIAL_VIDEO = "so8799877";
    private NicoHttpClient instance;

    public NicoHttpClientTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
        instance = new NicoHttpClient();
    }

    @After
    public void tearDown() {
    }

    /**
     * Test of login method, of class NicoHttpClient.
     */
    @Test
    public void login() throws HttpException, URISyntaxException, InterruptedException {
        System.out.println("login");

        // ログイン失敗ケース
        instance.logout();
        String mail = "test@example.com";
        String password = "xxxx";
        boolean result = instance.login(mail, password);
        assertEquals(false, result);

        // ログイン成功ケース
        instance.logout();
        mail = OK_MAIL;
        password = OK_PASS;
        result = instance.login(mail, password);
        assertEquals(true, result);
    }

    public void logout() throws URISyntaxException, HttpException, InterruptedException {
        System.out.println("logout");

        boolean result;
        result = instance.logout();
        assertEquals(true, result);

        result = instance.logout();
        assertEquals(true, result);
    }

    @Test
    public void loadMyList() throws URISyntaxException, HttpException, InterruptedException {
        System.out.println("loadMyList");

        List<NicoContent> list;

        list = instance.loadMyList(OK_LIST_NO);
        assertNotNull(list);
        assertNotSame(0, list.size());
//        for (Object o : list) {
//            System.out.println(o.toString());
//        }

        list = instance.loadMyList("XXXX");
        assertNotNull(list);
        assertSame(0, list.size());
    }

//
    public void loadMyListDaily() throws URISyntaxException, HttpException, InterruptedException {
        System.out.println("loadMyListDaily");
        List<NicoContent> list = instance.loadMyListDaily();
        assertNotNull(list);
        assertNotSame(0, list.size());
//        System.out.println(list.size());
//        for (Object o : list) {
//            System.out.println(o.toString());
//        }

        EntityManagerFactory factory;
        EntityManager manager;

        factory = Persistence.createEntityManagerFactory("NicoBrowserPU");
        manager = factory.createEntityManager();

        EntityTransaction transaction = manager.getTransaction();

        transaction.begin();
        try {
            for (NicoContent c : list) {
                manager.persist(c);
            }

            transaction.commit();
        } catch (Exception ex) {
            ex.printStackTrace();
            transaction.rollback();
        } finally {
            manager.close();
            factory.close();
        }

    }

    @Test
    public void getVideoInfo() throws URISyntaxException, HttpException, InterruptedException, IOException {
        System.out.println("getFlv");

        instance.login(OK_MAIL, OK_PASS);
        try {
            URL str = instance.getVideoInfo(NORMAL_VIDEO).getVideoUrl();
            System.out.println(str);
        } catch (IOException ex) {
            fail();
            Logger.getLogger(NicoHttpClientTest.class.getName()).log(Level.SEVERE, null, ex);
        }

    }

    public void getVideoInfo_Official() throws URISyntaxException, HttpException, InterruptedException, IOException {
        System.out.println("getVideoInfo_Official");
        instance.login(OK_MAIL, OK_PASS);
        VideoInfo vi = instance.getVideoInfo(OFFICIAL_VIDEO);
        assertNotSame(OFFICIAL_VIDEO, vi.getRealVideoId());
    }

    public void downLoad() throws URISyntaxException, IOException, HttpException, InterruptedException {
        System.out.println("downLoad");

        instance.login(OK_MAIL, OK_PASS);

        try {
            VideoInfo vi = instance.getVideoInfo("sm183036");
            instance.getFlvFile(vi, "sm183036", NicoContent.Status.GET_INFO, false, ProgressListener.EMPTY_LISTENER);
        } catch (IOException ex) {
            Logger.getLogger(NicoHttpClientTest.class.getName()).log(Level.SEVERE, null, ex);
            fail();
        }
    }

    /**
     * Test of getFlvFile method, of class NicoHttpClient.
     */
    public void getFlvFile_String() throws HttpException, InterruptedException, URISyntaxException {
        System.out.println("getFlvFile");
        String videoID = "sm1097445";
        instance.login(OK_MAIL, OK_PASS);
        GetFlvResult result;
        try {
            VideoInfo vi = instance.getVideoInfo(videoID);
            result = instance.getFlvFile(vi);
            System.out.println(result);
            assertNotSame(Status.GET_INFO, result);
        } catch (Exception ex) {
            Logger.getLogger(NicoHttpClientTest.class.getName()).log(Level.SEVERE, null, ex);
            fail();
        }

        // 権限が無い動画？の取得
        // http://www.nicovideo.jp/watch/1231042440
        // 【亡き王女の為のセプテット・ツェペシュの幼き末裔】Priere -プリエール-
        try {
            VideoInfo vi = instance.getVideoInfo("1231042440");
            result = instance.getFlvFile(vi);
            fail("権限が無い動画を取得しようとした場合は例外が送出される");
        } catch (IOException ex) {
            Logger.getLogger(NicoHttpClientTest.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    @Test
    public void getFlvFile_Official() throws IOException, URISyntaxException, HttpException, InterruptedException {
        // 公式動画(ビデオIDが実際と異なる)のダウンロード
        instance.login(OK_MAIL, OK_PASS);
        VideoInfo vi = instance.getVideoInfo(OFFICIAL_VIDEO);
        GetFlvResult result = instance.getFlvFile(vi);
        assertNotSame(Status.GET_INFO, result.getStatus());
        assertNull("公式動画からは著者は取得できない", result.getAuthor());
    }

    @Test
    public void search() throws URISyntaxException, InterruptedException, HttpException, IOException {
        System.out.println("search");
        instance.login(OK_MAIL, OK_PASS);

        SearchResult result = instance.search(SearchKind.KeyWord, "初音ミク", SortKind.PlayTimes, SortOrder.Descending, 1);
        ArrayList<NicoContent> conts = result.getContents();
        System.out.println("検索結果件数： " + conts.size());
        assertEquals("1ページの上限32件がヒット", 32, conts.size());
        assertTrue(result.getPages().size() > 0);
    }

    public void loadMyMovie() throws URISyntaxException, InterruptedException, HttpException, IOException {
        System.out.println("search");
        instance.login(OK_MAIL, OK_PASS);

        NicoContent cont;

        // 通常の動画
        cont = instance.loadMyMovie("sm9");
        assertNotNull(cont);
        assertEquals("sm9", cont.getNicoId());

        // チャンネル動画はnull(statusがfailなので)
        cont = instance.loadMyMovie("1228201771");
        assertNull(cont);

        // 削除済み
        cont = instance.loadMyMovie("sm1");
        assertNull(cont);

        // 存在しない
        cont = instance.loadMyMovie("xxx");
        assertNull(cont);
    }

    /**
     * マイリスト登録するテストケースだが, 不完全. 登録できなかった場合もテストケースは正常終了する.
     */
    @Test
    public void addMyList() throws URISyntaxException, HttpException, InterruptedException, IOException {
        instance.login(OK_MAIL, OK_PASS);

        instance.addMyList("5681871", "sm9477049");
    }

    @Test
    public void getCommentFile() throws URISyntaxException, HttpException, InterruptedException, Exception {
        final String comm = NORMAL_VIDEO + "comment";
        final String tcomm = NORMAL_VIDEO + "tcomment.xml";
        File commFile = new File(comm + ".xml");
        File tcommFile = new File(tcomm);

        assertFalse("ファイルを削除してください", commFile.exists());
        assertFalse("ファイルを削除してください", tcommFile.exists());

        instance.login(OK_MAIL, OK_PASS);
        VideoInfo vi = instance.getVideoInfo(NORMAL_VIDEO);
        instance.getCommentFile(vi, comm);
        instance.getTCommentFile(vi, tcomm);

        assertTrue(commFile.exists());
        assertTrue(tcommFile.exists());
    }

    @Test
    public void getCommentFile_Official() throws URISyntaxException, HttpException, InterruptedException, Exception {
        final String comm = OFFICIAL_VIDEO + "comment";
        final String tcomm = OFFICIAL_VIDEO + "tcomment.xml";
        File commFile = new File(comm + ".xml");
        File tcommFile = new File(tcomm);

        assertFalse("ファイルを削除してください", commFile.exists());
        assertFalse("ファイルを削除してください", tcommFile.exists());

        instance.login(OK_MAIL, OK_PASS);
        VideoInfo vi = instance.getVideoInfo(OFFICIAL_VIDEO);
        instance.getCommentFile(vi, comm);
        instance.getTCommentFile(vi, tcomm);

        assertTrue(commFile.exists());
        assertTrue(tcommFile.exists());
    }

    /**
     *
     */
    @Test
    public void testGetTitleInWatchPageZero() throws NoSuchMethodException, IOException, IllegalAccessException,
            IllegalArgumentException, InvocationTargetException {
        final String title = "【MMD】 WAVEFILE fullver. 【モーション完成】";
        final Method m = NicoHttpClient.class.getDeclaredMethod("getTitleInWatchPage", InputStream.class);
        m.setAccessible(true);

        final FileInputStream fisHarajuku = new FileInputStream(new File("testdata/hara_watch.html"));
        final String titleHarajuku = (String) m.invoke(instance, fisHarajuku);
        fisHarajuku.close();
        assertEquals(title, titleHarajuku);

        final FileInputStream fisZero = new FileInputStream(new File("testdata/zero_watch.html"));
        final String titleZero = (String) m.invoke(instance, fisZero);
        fisZero.close();
        assertEquals(title, titleZero);
    }
}
