using System;
using System.IO;

using NUnit.Framework;

using Uguu.IO.Csv;

namespace Uguu.IO.Csv.Test
{
    [TestFixture]
    public sealed class CsvReaderTest
    {
        [Test]
        public void Test1()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader))
            {
                // 最初の行の前。
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(-1, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                
                // 1 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                try
                {
                    string value = csvReader[3];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 2 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("111", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);
                try
                {
                    string value = csvReader[3];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 最後の行の後。
                Assert.IsFalse(csvReader.Read());
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(2, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
            }
        }
        [Test]
        public void Test2()
        {
            string testData = "111" + Environment.NewLine
                            + "222,222" + Environment.NewLine
                            + "333,333,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader))
            {
                // 最初の行の前。
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(-1, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                
                // 1 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(1, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("111", csvReader[0]);
                try
                {
                    string value = csvReader[1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 2 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(2, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("222", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                try
                {
                    string value = csvReader[2];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 3 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(2, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("333", csvReader[0]);
                Assert.AreEqual("333", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);
                try
                {
                    string value = csvReader[3];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 最後の行の後。
                Assert.IsFalse(csvReader.Read());
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(3, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
            }
        }
        [Test]
        public void Test3()
        {
            string testData = "foo,bar,\"boo" + Environment.NewLine
                            + "111,222\",333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader))
            {
                // 最初の行の前。
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(-1, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                
                // 1 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(4, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo" + Environment.NewLine
                              + "111,222", csvReader[2]);
                Assert.AreEqual("333", csvReader[3]);
                try
                {
                    string value = csvReader[4];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 最後の行の後。
                Assert.IsFalse(csvReader.Read());
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(1, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
            }
        }
        [Test]
        public void Test4()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "" + Environment.NewLine
                            + ",,," + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader))
            {
                // 最初の行の前。
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(-1, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                
                // 1 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                try
                {
                    string value = csvReader[3];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 2 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(0, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 3 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(4, csvReader.FieldCount);
                Assert.AreEqual(2, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("", csvReader[0]);
                Assert.AreEqual("", csvReader[1]);
                Assert.AreEqual("", csvReader[2]);
                Assert.AreEqual("", csvReader[3]);
                try
                {
                    string value = csvReader[4];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 4 行目。
                Assert.IsTrue(csvReader.Read());
                
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(3, csvReader.RowIndex);
                
                try
                {
                    string value = csvReader[-1];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                Assert.AreEqual("111", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);
                try
                {
                    string value = csvReader[3];
                    Assert.Fail();
                }
                catch (IndexOutOfRangeException)
                {
                }
                
                // 最後の行の後。
                Assert.IsFalse(csvReader.Read());
                try
                {
                    int fieldCount = csvReader.FieldCount;
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
                Assert.AreEqual(4, csvReader.RowIndex);
                try
                {
                    string value = csvReader[0];
                    Assert.Fail();
                }
                catch (InvalidOperationException)
                {
                }
            }
        }
        [Test]
        public void TestDispose()
        {
            StringReader reader = new StringReader("foo,bar,boo");
            CsvReader csvReader = new CsvReader(reader);
            csvReader.Read();
            csvReader.Dispose();
            
            try
            {
                int fieldCount = csvReader.FieldCount;
                Assert.Fail();
            }
            catch (ObjectDisposedException)
            {
            }
            Assert.AreEqual(0, csvReader.RowIndex);
            try
            {
                csvReader.Read();
                Assert.Fail();
            }
            catch (ObjectDisposedException)
            {
            }
            csvReader.Dispose();
            csvReader.Close();
        }
        [Test]
        public void TestClose()
        {
            StringReader reader = new StringReader("foo,bar,boo");
            CsvReader csvReader = new CsvReader(reader);
            csvReader.Read();
            csvReader.Close();
            
            try
            {
                int fieldCount = csvReader.FieldCount;
                Assert.Fail();
            }
            catch (ObjectDisposedException)
            {
            }
            Assert.AreEqual(0, csvReader.RowIndex);
            try
            {
                csvReader.Read();
                Assert.Fail();
            }
            catch (ObjectDisposedException)
            {
            }
            csvReader.Dispose();
            csvReader.Close();
        }
        [Test]
        public void TestTabDelimiter()
        {
            string testData = "foo\tbar\tboo" + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader, Environment.NewLine, "\t", true))
            {
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(1, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                Assert.AreEqual("111,222,333", csvReader[0]);
                
                Assert.IsFalse(csvReader.Read());
            }
        }
        [Test]
        public void TestNewLine()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "111,222,333\n\n\n"
                            + "aaa,bbb,ccc\n\n\n";
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader, "\n\n\n", ",", true))
            {
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(5, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo" + Environment.NewLine + "111", csvReader[2]);
                Assert.AreEqual("222", csvReader[3]);
                Assert.AreEqual("333", csvReader[4]);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                Assert.AreEqual("aaa", csvReader[0]);
                Assert.AreEqual("bbb", csvReader[1]);
                Assert.AreEqual("ccc", csvReader[2]);
                
                Assert.IsFalse(csvReader.Read());
            }
        }
        [Test]
        public void TestIsClosed()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            CsvReader csvReader = new CsvReader(reader);
            try
            {
                Assert.IsFalse(csvReader.IsClosed);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                
                Assert.IsFalse(csvReader.IsClosed);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                Assert.AreEqual("111", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);

                Assert.IsFalse(csvReader.IsClosed);
            }
            finally
            {
                csvReader.Close();
            }
            Assert.IsTrue(csvReader.IsClosed);
        }
        [Test]
        public void TestIgnoreEofLine1()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader, Environment.NewLine, ",", true))
            {
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                Assert.AreEqual("111", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);
                
                Assert.IsFalse(csvReader.Read());
            }
        }
        [Test]
        public void TestIgnoreEofLine2()
        {
            string testData = "foo,bar,boo" + Environment.NewLine
                            + "111,222,333" + Environment.NewLine;
            StringReader reader = new StringReader(testData);
            using (CsvReader csvReader = new CsvReader(reader, Environment.NewLine, ",", false))
            {
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(0, csvReader.RowIndex);
                Assert.AreEqual("foo", csvReader[0]);
                Assert.AreEqual("bar", csvReader[1]);
                Assert.AreEqual("boo", csvReader[2]);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(3, csvReader.FieldCount);
                Assert.AreEqual(1, csvReader.RowIndex);
                Assert.AreEqual("111", csvReader[0]);
                Assert.AreEqual("222", csvReader[1]);
                Assert.AreEqual("333", csvReader[2]);
                
                Assert.IsTrue(csvReader.Read());
                Assert.AreEqual(0, csvReader.FieldCount);
                Assert.AreEqual(2, csvReader.RowIndex);
                
                Assert.IsFalse(csvReader.Read());
                Assert.AreEqual(2, csvReader.RowIndex);
            }
        }
    }
}
