﻿/* ------------------------------------------------------------------------- */
//
// Copyright (c) 2010 CubeSoft, Inc.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published
// by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
/* ------------------------------------------------------------------------- */
using Cube.FileSystem;
using Cube.FileSystem.TestService;
using NUnit.Framework;
using System.Linq;

namespace Cube.Pdf.Tests.Editor.ViewModels
{
    /* --------------------------------------------------------------------- */
    ///
    /// OpenTest
    ///
    /// <summary>
    /// Tests for Open commands.
    /// </summary>
    ///
    /* --------------------------------------------------------------------- */
    [TestFixture]
    class OpenTest : ViewModelFixture
    {
        #region Tests

        /* ----------------------------------------------------------------- */
        ///
        /// Open
        ///
        /// <summary>
        /// Executes the test for opening a PDF document and creating
        /// images.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf",       "",         2)]
        [TestCase("SampleAes128.pdf", "password", 2)]
        public void Open(string filename, string password, int n) =>
            Create(filename, password, n, vm =>
        {
            var dummy = vm.Data.Images.Preferences.Dummy;
            var dest  = vm.Data.Images.First();
            Execute(vm, vm.Ribbon.Refresh);
            Assert.That(Wait.For(() => dest.Image != dummy), "Timeout");
        });

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink
        ///
        /// <summary>
        /// Executes the test of the OpenLink command.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf")]
        public void OpenLink(string filename)
        {
            Link(GetResultsWith(filename), GetExamplesWith(filename));
            Create(vm =>
            {
                var src = IO.Get(GetResultsWith($"{filename}.lnk"));
                Assert.That(src.Exists, Is.True);
                vm.Recent.Open.Execute(src);
                Assert.That(Wait.For(() => vm.Data.Count.Value == 2), "Timeout");
            });
        }

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink_NotFound
        ///
        /// <summary>
        /// Confirms the behavior when the specified link is invalid.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [TestCase("Sample.pdf")]
        public void OpenLink_NotFound(string filename)
        {
            var tmp = GetResultsWith(filename);
            var src = $"{tmp}.lnk";

            IO.Copy(GetExamplesWith(filename), tmp);
            Link(src, tmp);
            IO.TryDelete(tmp);
            Assert.That(IO.Exists(src), Is.True,  src);
            Assert.That(IO.Exists(tmp), Is.False, tmp);

            Create(vm =>
            {
                vm.Recent.Open.Execute(IO.Get(src));
                Assert.That(Wait.For(() => !IO.Exists(src)), src);
                Assert.That(vm.Data.IsOpen(), Is.False);
            });
        }

        /* ----------------------------------------------------------------- */
        ///
        /// OpenLink_Null
        ///
        /// <summary>
        /// Confirms the behavior when the specified link is null.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        [Test]
        public void OpenLink_Null() =>Create(vm =>
        {
            vm.Recent.Open.Execute(null);
            Assert.That(Wait.For(() => !vm.Data.Busy.Value), "Timeout");
            Assert.That(vm.Data.IsOpen(), Is.False);
        });

        #endregion

        #region Others

        /* ----------------------------------------------------------------- */
        ///
        /// Link
        ///
        /// <summary>
        /// Creates a link with the specified path.
        /// </summary>
        ///
        /* ----------------------------------------------------------------- */
        private void Link(string link, string src) => new Shortcut(IO)
        {
            FullName     = link,
            Target       = src,
            IconLocation = src,
        }.Create();

        #endregion
    }
}
