/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipo.com/
 *
 * Copyright(C) 2012 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.schedule;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALCellDateTimeField;
import com.aimluck.eip.cayenne.om.portlet.EipTScheduleMap;
import com.aimluck.eip.common.ALAbstractFormWindow;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.schedule.util.ScheduleUtils;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 個別色変更を実施する
 */
public class ScheduleChangeIndividualColorFormData extends ALAbstractFormWindow {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService.getLogger(ScheduleChangeIndividualColorFormData.class.getName());

  /** 重要フラグ更新成功メッセージ（JSONで正常メッセージとして返すため、Javascript側でも判定が必要） */
  public static final String CHANGE_COLOR_SUCCESS_MSG = "ChangeColor=";

  /** 色コード */
  private String colorCode;

  /** ログインユーザー */
  private ALEipUser loginUser;

  /** 対象ユーザーID */
  private int userId;

  /** スケジュールID */
  private String scheduleId = "";

  /** 携帯電話・スマートフォンからのアクセスか否か */
  private boolean isMobile = false;

  /** 携帯・スマートフォン用予定表示日 */
  private ALCellDateTimeField cellViewDate = null;

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#doViewForm(ALAction,
   * RunData, Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context) throws ALPageNotFoundException, ALDBErrorException {
    // 初期表示時・更新時等で共通の初期化処理を実行する

    // 基幹処理でエンティティIDの決定を実施
    super.init(action, rundata, context);

    // ログインユーザー情報を取得
    loginUser = ALEipUtils.getALEipUser(rundata);

    // スケジュールIDを取得
    scheduleId = ALEipUtils.getTemp(rundata, context, ALEipConstants.ENTITY_ID);

    // 対象ユーザーIDを取得
    if (rundata.getParameters().containsKey("userid")) {
      userId = rundata.getParameters().getInt("userid");
    } else {
      // 得られない場合はログインユーザーを対象ユーザーとする
      userId = (int) loginUser.getUserId().getValue();
    }

    if (isMobile) {
      // 携帯電話・スマートフォンからのアクセス時限定の処理

      // 予定表示日の初期化
      cellViewDate = new ALCellDateTimeField("yyyy-MM-dd");
      cellViewDate.setFieldName("指定日時");

      // 表示日の取得
      String viewDate = rundata.getParameters().getString("view_date");
      // 予定表示日の形式チェック
      ALCellDateTimeField validationCellDateTimeField = new ALCellDateTimeField("yyyy-MM-dd");
      validationCellDateTimeField.setValue(viewDate);
      if (!validationCellDateTimeField.validate(new ArrayList<String>())) {
        // 形式誤りの場合は、未設定扱いにする
        viewDate = "";
      }
      // 携帯・スマートフォン用予定表示日の設定
      if (null == viewDate || "".equals(viewDate)) {
        cellViewDate.setValue(new Date());
      } else {
        cellViewDate.setValue(viewDate);
      }
    }

  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#loadFormData(RunData
   * rundata, Context context, List<String> msgList)
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context, List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    // 画面初期表示時の初期値を設定する

    // 戻り値
    boolean result = true;
    // 処理対象のスケジュールマップ
    EipTScheduleMap scheduleMap = null;

    try {

      // スケジュールマップを取得
      scheduleMap = ScheduleUtils.getEipTScheduleMapByLogicKey(userId, new Integer(scheduleId));
      if (scheduleMap == null) {
        // スケジュールマップが見つからない場合
        logger.error("更新対象のスケジュールが見つかりません。ログインユーザー:" + loginUser.getName() + "/スケジュールID:" + scheduleId + "/対象ユーザー:" + userId);
        colorCode = "";
        return false;
      }

      // 個別色
      colorCode = scheduleMap.getIndividualColor();
      if (null == colorCode) {
        colorCode = "";
      }

    } catch (Exception e) {
      logger.error("画面の初期化に失敗しました。ログインユーザー:" + loginUser.getName() + "/スケジュールID:" + scheduleId + "/対象ユーザー:" + userId);
      colorCode = "";
      result = false;
    }

    return result;
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#setFormData(RunData,
   * Context, List<String>)
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context, List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    // リクエストパラメータをデータ変数に設定する

    if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
      // 更新時処理

      // 色コードを取得
      if (rundata.getParameters().containsKey("categoryColor")) {
        colorCode = rundata.getParameters().getString("categoryColor");
      } else {
        // 得られない場合は空文字とする
        colorCode = "";
      }
    }

    return true;
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#updateFormData(RunData
   * rundata, Context context, List<String> msgList)
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context, List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    // 更新対象スケジュールマップ
    EipTScheduleMap scheduleMap = null;
    try {

      // 更新対象スケジュールマップ取得
      scheduleMap = ScheduleUtils.getEipTScheduleMapByLogicKey(userId, new Integer(scheduleId));
      if (scheduleMap == null) {
        logger.error("更新対象のスケジュールが見つかりません。");
        throw new Exception();
      }

      // 色コードを設定
      scheduleMap.setIndividualColor(colorCode);

      Database.commit();

    } catch (Exception e) {
      // 例外発生時
      Database.rollback();
      logger.error("スケジュールマップの個別色更新に失敗しました。ログインユーザー:" + loginUser.getName() + "/スケジュールID:" + scheduleId + "/対象ユーザー:" + userId, e);
      msgList.add("個別色更新に失敗しました。");
      return false;
    }
    return true;
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#initField()
   */
  @Override
  public void initField() {
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() throws ALPageNotFoundException, ALDBErrorException {
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(List<String>
   * msgList)
   */
  @Override
  protected boolean validate(List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    return true;
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#deleteFormData(RunData
   * rundata, Context context, List<String> msgList)
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context, List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    return false;
  }

  /*
   * @see com.aimluck.eip.common.ALAbstractFormData#insertFormData(RunData
   * rundata, Context context, List<String> msgList)
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context, List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {
    return false;
  }

  /**
   * 正常終了メッセージを返します。
   * 
   * @return 正常終了メッセージ（変数名=色コード）
   */
  public String getSuccessMsg() {
    return CHANGE_COLOR_SUCCESS_MSG + colorCode;
  }

  /**
   * 携帯電話・スマートフォンからのアクセスか否かを設定します。
   * 
   * @param value
   *          携帯電話・スマートフォンからのアクセスの場合true
   */
  public void setMobile(boolean value) {
    isMobile = value;
  }

  /**
   * 携帯電話・スマートフォンからのアクセスか否かを設定します。
   * 
   * @return 携帯電話・スマートフォンからのアクセスの場合true
   */
  public boolean isMobile() {
    return isMobile;
  }

  /**
   * 更新対象のスケジュールIDを返します（分類選択画面用）。
   * 
   * @return スケジュールID
   */
  public String getScheduleId() {
    return scheduleId;
  }

  /**
   * 更新対象ユーザーIDを返します（分類選択画面用）。
   * 
   * @return ユーザーID
   */
  public String getUserId() {
    return Integer.toString(userId);
  }

  /**
   * 現在設定中の個別色（色コード）を返します（分類選択画面用）。
   * 
   * @return 色コード
   */
  public String getColorCode() {
    return colorCode;
  }

  /**
   * 分類名リストを返します（分類選択画面用）。
   * 
   * @return 分類名リスト
   */
  public List<String> getScheduleCategoryNameList() {
    return ScheduleUtils.SCHEDULE_CATEGORY_NAME_LIST;
  }

  /**
   * 色コードリストを返します（分類選択画面用）。
   * 
   * @return 色コードリスト
   */
  public List<String> getScheduleCategoryColorList() {
    return ScheduleUtils.SCHEDULE_CATEGORY_COLOR_LIST;
  }

  /**
   * 携帯電話・スマートフォン用予定表示日を返します（分類選択画面用）。
   * 
   * @return 携帯電話・スマートフォン用予定表示日
   */
  public ALCellDateTimeField getCellViewDate() {
    return cellViewDate;
  }

}
