//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 2/25/2012.
//
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QFileDialog>
#include <QDir>
#include <QDebug>
#include <QWebFrame>
#include <QLocalSocket>

#include "TKLog.h"
#include "DNGlobal.h"
#include "DNAlert.h"
#include "DNEngine.h"
#include "TKConsole.h"
#include "dnwebinterface.h"
#include "DNSettings.h"
#include "versioninfo.h"

//static
QColor MainWindow::NORMAL_COLOR(0,0,0);
//static
QColor MainWindow::WARN_COLOR(0xff, 0x95, 0x2b);
//static
QColor MainWindow::ERROR_COLOR(0xff, 0,0);
//static
QWidget *MainWindow::instance = NULL;

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    mEngine(NULL), mWebInterface(NULL), mControlledByCreator(false), mCreatorIPCServer(NULL)
{
    QStringList arguments = qApp->arguments();
    for (int i = 0; i < arguments.length(); i++)
    {
        if (!mControlledByCreator && arguments.at(i) == "-creatorControlled")
        {
            i++;
            if (i < arguments.length() && arguments.at(i).indexOf("denncoCreator_") == 0)
            {
                mCreatorIPCServer = new QLocalServer(this);
                if (mCreatorIPCServer->listen(arguments.at(i)))
                {
                    mControlledByCreator = true;
                }
            }
        }
    }

    instance = this;
    ui->setupUi(this);

    QString windowTitle = "dennco engine - ";
    windowTitle.append(ENGINE_VERSION);

    this->setWindowTitle(windowTitle);

    if (!mControlledByCreator)
    {
        QString defaultPath(QDir::homePath());
        QString contentPath = QString::fromStdString(DNSettings::getValue(DNSettings::CONTEXT_PATH, defaultPath.toStdString()));

        ui->filePath->setText(contentPath);

        ui->console->document()->setMaximumBlockCount(500);
        QFont monofont("Courier");
        monofont.setStyleHint(QFont::Monospace);
        ui->console->setFont(monofont);
        TKLog::setDestination(this);
        connect(this, SIGNAL(consoleUpdated()), this, SLOT(update()), Qt::AutoConnection);

        std::string savedWindowStateStr = DNSettings::getValue(DNSettings::WINDOWLAYOUT,"");
        QByteArray savedWindowStateQArray(savedWindowStateStr.c_str());
        restoreState(QByteArray::fromHex(savedWindowStateQArray));

        std::string savedWindowGeometoryStr = DNSettings::getValue(DNSettings::WINDOWGEOMETORY,"");
        QByteArray savedWindowGeometoryQArray(savedWindowGeometoryStr.c_str());
        restoreGeometry(QByteArray::fromHex(savedWindowGeometoryQArray));
    }
    else
    {
        ui->filePath->setText("");
        ui->filePath->setEnabled(false);
        ui->chooseDirButton->setVisible(false);

        ui->console->document()->setMaximumBlockCount(500);
        QFont monofont("Courier");
        monofont.setStyleHint(QFont::Monospace);
        ui->console->setFont(monofont);
        TKLog::setDestination(this);
        connect(this, SIGNAL(consoleUpdated()), this, SLOT(update()), Qt::AutoConnection);

        std::string savedWindowStateStr = DNSettings::getValue(DNSettings::CREATORCONTROLLED_WINDOWLAYOUT,"");
        QByteArray savedWindowStateQArray(savedWindowStateStr.c_str());
        restoreState(QByteArray::fromHex(savedWindowStateQArray));

        std::string savedWindowGeometoryStr = DNSettings::getValue(DNSettings::CREATORCONTROLLED_WINDOWGEOMETORY,"");
        QByteArray savedWindowGeometoryQArray(savedWindowGeometoryStr.c_str());
        restoreGeometry(QByteArray::fromHex(savedWindowGeometoryQArray));

        connect(mCreatorIPCServer, SIGNAL(newConnection()), this, SLOT(onNewRequestFromCreator()));
    }
}

MainWindow::~MainWindow()
{
    if (mCreatorIPCServer)
        delete mCreatorIPCServer;

    instance = 0;
    delete ui;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (!mControlledByCreator)
    {
        QByteArray savingWindowStateQArray = saveState().toHex();
        DNSettings::setValue(DNSettings::WINDOWLAYOUT, savingWindowStateQArray.constData());

        QByteArray savingWindowGeometoryQArray = saveGeometry().toHex();
        DNSettings::setValue(DNSettings::WINDOWGEOMETORY, savingWindowGeometoryQArray.constData());
    }
    else
    {
        QByteArray savingWindowStateQArray = saveState().toHex();
        DNSettings::setValue(DNSettings::CREATORCONTROLLED_WINDOWLAYOUT, savingWindowStateQArray.constData());

        QByteArray savingWindowGeometoryQArray = saveGeometry().toHex();
        DNSettings::setValue(DNSettings::CREATORCONTROLLED_WINDOWGEOMETORY, savingWindowGeometoryQArray.constData());
    }

    stopEngine();
    deleteCurrentEngine();
    QMainWindow::closeEvent(event);
}


void MainWindow::on_chooseDirButton_clicked()
{
    if (QString::compare(ui->startButton->text(), "Stop", Qt::CaseInsensitive) == 0)
    {
        DNAlert::show("Please stop engine", "Dennco engine is running.\nPlease stop engine before switching directory.");
        return;
    }
    QString dirPath = QFileDialog::getExistingDirectory(this,"Set dennco setting file",ui->filePath->text());
    qDebug()<< dirPath;
    if (dirPath.length() > 0)
    {
        ui->filePath->setText(dirPath);

        DNSettings::setValue(DNSettings::CONTEXT_PATH, dirPath.toStdString());
    }
}


void MainWindow::on_startButton_clicked()
{
    qDebug()<< "start button clicked";
    if (QString::compare(ui->startButton->text(), "Start", Qt::CaseInsensitive) == 0)
    {
        deleteCurrentEngine();

        if (startEngine())
        {
            ui->startButton->setText("Stop");
        }
    }
    else
    {
        stopEngine();
        ui->startButton->setText("Start");
    }
}

bool MainWindow::startEngine()
{
    bool initializationFailed = false;

    mEngine = new DNEngine(ui->filePath->text().toLocal8Bit().data());
    if (!dnGlobal()->isErrorStatusNormal())
    {
        DNAlert::show(dnGlobal()->getMessage1(), dnGlobal()->getMessage2());
        dnGlobal()->resetErrorStatus();
        initializationFailed = true;
    }
    else if (!mEngine->isValid())
    {
        DNAlert::show("DNEngine initialize error", "ERROR. DNEngine not configured properly");
        initializationFailed = true;
    }

    if (!initializationFailed)
    {
        mEngine->startEngine();
        loadUI();
    }
    return !initializationFailed;
}

bool MainWindow::stopEngine()
{
    if (mEngine)
    {
        mEngine->stopEngine();
    }
    return true;
}

void MainWindow::loadUI()
{
    QString path = "file:///";
    path.append(ui->filePath->text().append(QString::fromStdString(mEngine->getUIPath())));
    QUrl pathUrl(path);
    ui->webView->load(pathUrl);
    mWebInterface = new DNWebInterface(mEngine);
    attachWebInterface();
}

void MainWindow::attachWebInterface()
{
    if (!mWebInterface)
        return;

    QWebPage *page = ui->webView->page();
    if (page)
    {
        QWebFrame *frame = page->mainFrame();
        if (frame)
        {
            frame->addToJavaScriptWindowObject("engine", mWebInterface);
            connect(frame, SIGNAL(javaScriptWindowObjectCleared()), this, SLOT(attachWebInterface()));
            connect(frame, SIGNAL(initialLayoutCompleted()), this, SLOT(adjustWindowSize()));
        }
    }
}

void MainWindow::adjustWindowSize()
{
    int hadjust = ui->webView->page()->mainFrame()->scrollBarMaximum(Qt::Horizontal);
    int vadjust = ui->webView->page()->mainFrame()->scrollBarMaximum(Qt::Vertical);
    hadjust = hadjust > 0 ? hadjust : 0;
    vadjust = vadjust > 0 ? vadjust : 0;
    QSize currentSize = size();
    resize(currentSize.width() + hadjust, currentSize.height() + vadjust);
}

void MainWindow::deleteCurrentEngine()
{
    if (mEngine)
    {
        delete mEngine;
        mEngine = NULL;
    }
    if (mWebInterface)
    {
        delete mWebInterface;
        mWebInterface = NULL;
    }
}

void MainWindow::paintEvent( QPaintEvent * event )
{
    mConsoleLock.lock();
    while (!mPendingConsoleMessages.isEmpty())
    {
        PendingConsoleMessage aMessage = mPendingConsoleMessages.dequeue();
        switch(aMessage.messageType)
        {
        case TKLog::NORMAL :
            ui->console->setTextColor(NORMAL_COLOR);
            break;
        case TKLog::WARNING :
            ui->console->setTextColor(WARN_COLOR);
            break;
        case TKLog::ERROR :
            ui->console->setTextColor(ERROR_COLOR);
            break;
        }
        ui->console->append(aMessage.text);
    }
    mConsoleLock.unlock();

    QMainWindow::paintEvent(event);
}


void MainWindow::vprintf(TKLog::MessageType type, const char *fmt, va_list ap)
{
    QString msg = QString().vsprintf(fmt,ap);
#ifdef DEBUG
    qDebug() << msg << endl;
#endif
    mConsoleLock.lock();
    PendingConsoleMessage newMessage;
    newMessage.messageType = type;
    newMessage.text = msg;
    mPendingConsoleMessages.enqueue(newMessage);
    mConsoleLock.unlock();
    emit consoleUpdated();
}

void MainWindow::vDebugPrintf(const char *fmt, va_list ap)
{
#ifdef DEBUG
    QString msg = QString().vsprintf(fmt,ap);
    qDebug() << "DEBUG:" << msg << endl;
#else
    (void)fmt; (void)ap;
#endif
}

void MainWindow::onNewRequestFromCreator()
{
    QLocalSocket *clientConnection = mCreatorIPCServer->nextPendingConnection();
    connect(clientConnection, SIGNAL(disconnected()), clientConnection, SLOT(deleteLater()));

    while(clientConnection->state() == QLocalSocket::ConnectedState || clientConnection->state() == QLocalSocket::ConnectingState)
    {
        clientConnection->waitForReadyRead();
        QString readString = clientConnection->readAll();
        QStringList requestData = readString.split(",");

        if (requestData.length() < 1)
            continue;

        if (requestData.at(0) == "load" && requestData.length() == 2)
        {
    #ifdef DEBUG
            qDebug() << "DEBUG: recieved load  request from denncoCreator. New contant path:" << requestData.at(1) << endl;
    #endif
            ui->filePath->setText(requestData.at(1));

            clientConnection->disconnectFromServer();

            if (QString::compare(ui->startButton->text(), "Start", Qt::CaseInsensitive) != 0)
            {
                stopEngine();
            }

            deleteCurrentEngine();

            if (startEngine())
            {
                ui->startButton->setText("Stop");
            }
            break;
        }
        else if (requestData.at(0) == "close")
        {
            close();
            break;
        }
    }
}

