//  Copyright (c) 2012 Dennco Project
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

//
//  Created by tkawata on 2/22/2012.
//
#include "dnqscellcode.h"

#include "DNUtils.h"
#include "TKLog.h"
#include "TKContainer.h"
#include "dnqscontainer.h"
#include "dnqscellbase.h"
#include "dnqscellcodeinstance.h"
#include "DNGlobal.h"

#include <QStringList>

DNQSCellCode::DNQSCellCode(std::string name, std::string cellapi, DNQSContainer *container, std::string code):
    TKCellCode(name,cellapi), mCellContainer(container)
{
    std::string jname = getJSEscapeString(name.c_str());
    QString stmt = QString::fromStdString(parseCellCodeForScriptEngine(jname, code));

    QScriptEngine *scriptEngine = mCellContainer->getScriptEngine();
    QScriptValue  scriptGlobalObject = mCellContainer->getScriptGlobalObject();
    scriptEngine->evaluate(QScriptProgram(stmt));
    if (scriptEngine->hasUncaughtException())
    {
        QScriptValue error = scriptEngine->uncaughtException();
        int lineNumber = scriptEngine->uncaughtExceptionLineNumber();
        QStringList list = stmt.split("\n", QString::KeepEmptyParts);
        QString errorStatement = "";
        if (lineNumber <= list.length())
        {
            int i = lineNumber < 2 ? 0 : lineNumber - 2;
            int j = i + 2 <= list.length() ? i + 2 : list.length() - 1;
            while (i <= j)
            {
                errorStatement += list.at(i);
                errorStatement += "\n";
                i++;
            }
        }
        QString errorString = error.toString();
        scriptEngine->clearExceptions();

        std::string message = "Failed to construct Cell code : ";
        message += getFQNName();
        message += "\n";
        if (errorStatement.length()>0)
        {
            message += ("Statement:\n...\n");
            message += errorStatement.toStdString();
            message += "...\n";
        }
        message += "Error Message:";
        message += errorString.toStdString();
        dnNotifyError("Initialization failed", message);
        return;
    }

    mCellCodeConstructor = scriptGlobalObject.property(QString::fromStdString(jname), QScriptValue::ResolveLocal);
    if (mCellCodeConstructor.isNull() || mCellCodeConstructor.isUndefined() || ! mCellCodeConstructor.isValid())
    {
        std::string message = "Failed to construct Cell code : ";
        message.append(getFQNName()).append("\n");
        message.append("The constructor is invalid.");
        dnNotifyError("Initialization failed", message);
        return;
    }
}

DNQSCellCode::~DNQSCellCode()
{

}
TKCellCodeInstance* DNQSCellCode::createCellCodeInstance(TKCell *owner, const void *data)
{
    if (!mCellContainer)
    {
        return NULL;
    }
    if (mCellCodeConstructor.isNull() || mCellCodeConstructor.isUndefined() || ! mCellCodeConstructor.isValid())
    {
        std::string message = "Failed to construct Cell code '";
        message.append(owner->getName());
        message.append("'  cellcode:");
        message.append(getFQNName()).append("\n");
        message.append("Cell code invalid");
        dnNotifyError("Initialization failed", message);
        return NULL;
    }

    TKCellCodeInstance *result = NULL;
    QScriptEngine *scriptEngine = mCellContainer->getScriptEngine();

    QScriptValueList args;
    args << ((DNQSCellBase*)owner)->getCellAPI();
    QScriptValue instance = mCellCodeConstructor.construct(args);
    if (scriptEngine->hasUncaughtException())
    {
        QScriptValue error = scriptEngine->uncaughtException();
        QString errorString = error.toString();
        scriptEngine->clearExceptions();

        std::string message = "Failed to construct Cell code '";
        message.append(owner->getName()).append("'\n");
        message.append("cellcode:");
        message.append(getFQNName()).append("\n");
        message.append("Error Message:").append(errorString.toStdString());
        dnNotifyError("Initialization failed", message);
        return NULL;
    }

    if (instance.isValid())
    {
        //
        QScriptContext *context = scriptEngine->pushContext();
        context->setThisObject(instance);
        context->setActivationObject(instance);
        QString customScriptStr((const char*)data);
        QScriptProgram customScript(customScriptStr);
        scriptEngine->evaluate(customScript);
        if (scriptEngine->hasUncaughtException())
        {
            QScriptValue error = scriptEngine->uncaughtException();
            int lineNumber = scriptEngine->uncaughtExceptionLineNumber();
            QStringList list = customScriptStr.split("\n", QString::KeepEmptyParts);
            QString errorStatement = "";
            if (lineNumber <= list.length())
            {
                int i = lineNumber < 2 ? 0 : lineNumber - 2;
                int j = i + 2 < list.length() ? i + 2 : list.length() - 1;
                while (i <= j)
                {
                    errorStatement += list.at(i);
                    errorStatement += "\n";
                    i++;
                }
            }
            QString errorString = error.toString();
            scriptEngine->clearExceptions();

            std::string message = "Failed to construct Cell '";
            message += owner->getName();
            message += "'\n";
            message += "cellcode:";
            message += getFQNName();
            message += "\n";
            if (errorStatement.length()>0)
            {
                message += ("Statement:\n...\n");
                message += errorStatement.toStdString();
                message += "...\n";
            }
            message += "Error Message:";
            message += errorString.toStdString();
            dnNotifyError("Initialization failed", message);
        }

        scriptEngine->popContext();

        result = new DNQSCellCodeInstance(owner, instance);

    }

    return result;
}
