unit u_dzDebugVisualizerHandler;

interface

uses
  Menus,
  Contnrs,
  u_dzDebugVisualizerEventHooking,
  w_dzDebugVisualizerMultilineString;

function UnquoteDebuggerString(const _s: string): string;
function DebuggerStringToDateTime(const _s: string): string;
function DebuggerStringToColor(const _s: string): string;
function DebuggerStringToWebColor(const _s: string): string;

type
  TVisualizerHandler = class
  private
    FNode: PVirtualNode;
    FMenuItem: TMenuItem;
    FExpression: UnicodeString;
  public
    procedure ConvertValue(var _Text: UnicodeString); virtual;
    constructor Create(const _Node: PVirtualNode; _mi: TMenuItem; const _Expression: UnicodeString);
    procedure CheckMenuIfActive; virtual;
  end;

  TGetAsDateTimeHandler = class(TVisualizerHandler)
  public
    procedure ConvertValue(var _Text: UnicodeString); override;
  end;

  TShowMultlineHandler = class(TVisualizerHandler)
  private
    FForm: Tf_dzDebugVisualizerMultilineString;
  public
    procedure ConvertValue(var _Text: UnicodeString); override;
    constructor Create(const _Node: PVirtualNode; _mi: TMenuItem; const _Expression: string);
    destructor Destroy; override;
    procedure CheckMenuIfActive; override;
  end;

type
  TVisualizerHandlerList = class
  private
    FList: TObjectList;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(_Handler: TVisualizerHandler): Integer;
    function Count: Integer;
    procedure Delete(_Idx: Integer); overload;
    function Delete(_Node: PVirtualNode): Boolean; overload;
    function TryGetHandler(_Node: PVirtualNode; out _Idx: Integer): Boolean; overload;
    function TryGetHandler(_Node: PVirtualNode; out _Handler: TVisualizerHandler): Boolean; overload;
  end;

implementation

uses
  SysUtils,
  Classes,
  GraphUtil,
  Graphics;

function UnquoteDebuggerString(const _s: string): string;
var
  sl: TStringList;
  Len: Integer;
  s: string;
  i: Integer;
begin
  Result := '';
  s := StringReplace(_s, '#$D#$A', sLineBreak, [rfreplaceAll]);
  if s = '' then
    Exit; //==>

  sl := TStringList.Create;
  try
    sl.Text := s;
    for i := 0 to sl.Count - 1 do begin
      s := sl[i];
      // There is a bug in the Delphi Debugger that prevent us from using AnsiDequotedStr here:
      // It does not escape single quotes within the string but apparently only adds a single quote
      // to the front and end of the string.
      Len := Length(s);
      if (s <> '') and (s[1] = #39) and (s[Len] = #39) then begin
        sl[i] := Copy(s, 2, Len - 2);
      end;
    end;
    Result := sl.Text;
  finally
    sl.Free;
  end;
end;

function DebuggerStringToDateTime(const _s: string): string;
var
  dt: TDateTime;
begin
  Result := _s;
  if not TryStrToFloat(_s, Double(dt)) then
    Exit;
  try
    Result := DateTimeToStr(dt);
  except
    // ignore
  end;
end;

function DebuggerStringToColor(const _s: string): string;
var
  IntValue: Integer;
  RGBValue: Integer;
  R: Integer;
  G: Integer;
  B: Integer;
begin
  Result := _s;
  if not TryStrToInt(_s, IntValue) then
    Exit;
  RGBValue := ColorToRGB(IntValue);

  R := RGBValue and $FF;
  G := (RGBValue shr 8) and $FF;
  B := (RGBValue shr 16) and $FF;
  try
    Result := ColorToString(IntValue) + ' ' + ColorToWebColorName(IntValue) + ' '
      + Format('RGB(%d, %d, %d)', [R, G, B]);
  except
    // ignore
  end;
end;

function DebuggerStringToWebColor(const _s: string): string;
var
  IntValue: Integer;
begin
  Result := _s;
  if not TryStrToInt(_s, IntValue) then
    Exit;
  try
    Result := ColorToWebColorName(IntValue);
  except
    // ignore
  end;
end;

{ TVisualizerHandler }

constructor TVisualizerHandler.Create(const _Node: PVirtualNode; _mi: TMenuItem; const _Expression: UnicodeString);
begin
  inherited Create;
  FNode := _Node;
  FMenuItem := _mi;
  FExpression := _Expression;
end;

procedure TVisualizerHandler.CheckMenuIfActive;
begin
  FMenuItem.Checked := True;
end;

procedure TVisualizerHandler.ConvertValue(var _Text: UnicodeString);
begin
  // don't change the text
end;

{ TGetAsDateTimeHandler }

procedure TGetAsDateTimeHandler.ConvertValue(var _Text: UnicodeString);
begin
  _Text := DebuggerStringToDateTime(_Text);
end;

{ TShowMultlineHandler }

constructor TShowMultlineHandler.Create(const _Node: PVirtualNode; _mi: TMenuItem; const _Expression: string);
begin
  inherited Create(_Node, _mi, _Expression);
  FForm := Tf_dzDebugVisualizerMultilineString.ShowNew(nil, _Expression, '');
end;

destructor TShowMultlineHandler.Destroy;
begin
  FreeAndNil(FForm);
  inherited;
end;

procedure TShowMultlineHandler.CheckMenuIfActive;
begin
  FMenuItem.Checked := FForm.Visible;
end;

procedure TShowMultlineHandler.ConvertValue(var _Text: UnicodeString);
var
  WatchValue: string;
begin
  inherited;

  WatchValue := UnquoteDebuggerString(_Text);
  FForm.m_Output.Lines.Text := WatchValue;
end;

{ TVisualizerHandlerList }

constructor TVisualizerHandlerList.Create;
begin
  inherited Create;
  FList := TObjectList.Create;
end;

destructor TVisualizerHandlerList.Destroy;
begin
  FreeAndNil(FList);
  inherited;
end;

function TVisualizerHandlerList.Add(_Handler: TVisualizerHandler): Integer;
begin
  Result := FList.Add(_Handler);
end;

function TVisualizerHandlerList.Count: Integer;
begin
  Result := FList.Count;
end;

procedure TVisualizerHandlerList.Delete(_Idx: Integer);
begin
  FList.Delete(_Idx);
end;

function TVisualizerHandlerList.Delete(_Node: PVirtualNode): Boolean;
var
  Idx: Integer;
begin
  Result := TryGetHandler(_Node, Idx);
  if Result then
    Delete(Idx);
end;

function TVisualizerHandlerList.TryGetHandler(_Node: PVirtualNode; out _Idx: Integer): Boolean;
var
  i: Integer;
  Handler: TVisualizerHandler;
begin
  Result := True;
  if Assigned(FList) then begin
    for i := 0 to FList.Count - 1 do begin
      Handler := FList[i] as TVisualizerHandler;
      if Handler.FNode = _Node then begin
        _Idx := i;
        Exit; //==>
      end;
    end;
  end;
  Result := False;
end;

function TVisualizerHandlerList.TryGetHandler(_Node: PVirtualNode;
  out _Handler: TVisualizerHandler): Boolean;
var
  i: Integer;
begin
  Result := True;
  if Assigned(FList) then begin
    for i := 0 to FList.Count - 1 do begin
      _Handler := FList[i] as TVisualizerHandler;
      if _Handler.FNode = _Node then
        Exit; //==>
    end;
  end;
  Result := False;
end;

end.
