unit u_dzDebugVisualizerWatchFormHandler;

interface

uses
  Windows,
  SysUtils,
  Classes,
  Controls,
  Forms,
  Menus,
  u_dzDebugVisualizerEventHooking,
  u_dzDebugVisualizerHandler;

type
  TDebugVisualizerWatchFormHandler = class(TComponent)
  private
    FGetTextHook: TVSTGetTextEventHook;
    FVisualizerList: TVisualizerHandlerList;
    FWatchWindow: TForm;
    FWatchTree: TControl;
    FLastVSTNode: PVirtualNode;
    mi_CopyWatchValue: TMenuItem;
    mi_ShowMultilineString: TMenuItem;
    mi_ShowAsDateTime: TMenuItem;
    FLastValuelText: UnicodeString;
    FLastExpression: UnicodeString;
    FOnDestroyed: TNotifyEvent;
    procedure OnGetTextHook(_Sender: TBaseVirtualTree; _Node: PVirtualNode; _Column: TColumnIndex;
      _TextType: TVSTTextType; var _CellText: UnicodeString);
    procedure InstallTreeViewHooks;
    procedure UninstallTreeViewHooks;
    procedure mi_ShowMultilineStringClick(Sender: TObject);
    procedure mi_ShowAsDateTimeClick(Sender: TObject);
    procedure FocusChanging(_Sender: TBaseVirtualTree; _OldNode, _NewNode: PVirtualNode;
      _OldColumn, _NewColumn: TColumnIndex; var Allowed: Boolean);
    procedure RemoveMenuItems;
  public
    constructor Create(_Owner: TComponent); override;
    destructor Destroy; override;
    property OnDestroyed: TNotifyEvent read FOnDestroyed write FOnDestroyed;
  end;

implementation

uses
  TypInfo;

{ TDebugVisualizerWatchFormHandler }

constructor TDebugVisualizerWatchFormHandler.Create(_Owner: TComponent);
var
  i: Integer;
  pm: TPopupMenu;
  Item: TMenuItem;
  ctrl: TControl;
begin
  inherited Create(_Owner);

  FVisualizerList := TVisualizerHandlerList.Create;

  FWatchWindow := _Owner as TForm;

  // First create a menu item to insert in the Watch Window's context menu
  mi_ShowMultilineString := TMenuItem.Create(nil);
  mi_ShowMultilineString.OnClick := mi_ShowMultilineStringClick;
  mi_ShowMultilineString.Caption := 'Show Multiline String';

  mi_ShowAsDateTime := TMenuItem.Create(nil);
  mi_ShowAsDateTime.OnClick := mi_ShowAsDateTimeClick;
  mi_ShowAsDateTime.Caption := 'Show as TDateTime';

  for i := 0 to FWatchWindow.ControlCount - 1 do begin
    ctrl := FWatchWindow.Controls[i];
    if ctrl.ClassNameIs('TVirtualStringTree') and SameText(ctrl.Name, 'WatchTree') then begin
      FWatchTree := ctrl;
      InstallTreeViewHooks;
      Break;
    end;
  end;

  //  Scan the Watch Window's context menu to find the existing "Copy watch value" entry
  //  and insert our menu items after it
  pm := FWatchWindow.PopupMenu;
  for i := 0 to pm.Items.Count - 1 do begin
    Item := pm.Items[i];
    if CompareText('Copy Watch &Value', Item.Caption) = 0 then begin
      mi_CopyWatchValue := Item;
      pm.Items.Insert(i + 1, mi_ShowMultilineString);
      pm.Items.Insert(i + 1, mi_ShowAsDateTime);
      Break;
    end;
  end;
end;

destructor TDebugVisualizerWatchFormHandler.Destroy;
begin
  UninstallTreeViewHooks;
  RemoveMenuItems;
  FreeAndNil(FVisualizerList);
  if Assigned(FOnDestroyed) then
    FOnDestroyed(Self);
  inherited;
end;

procedure TDebugVisualizerWatchFormHandler.RemoveMenuItems;
var
  Idx: Integer;
  PopupMenu: TMenuItem;
begin
  try
    if Assigned(mi_CopyWatchValue) then begin
      PopupMenu := mi_CopyWatchValue.Parent;
      if Assigned(PopupMenu) then begin
        // If PopupMenu is already NIL, it has already freed our additional menu items.
        // If not, it may be that the plugin is being unloaded so we must remove the menu items
        // from the popup menu before freeing them.
        Idx := PopupMenu.IndexOf(mi_ShowMultilineString);
        if Idx <> -1 then
          PopupMenu.Delete(Idx);
        FreeAndNil(mi_ShowMultilineString);
        Idx := PopupMenu.IndexOf(mi_ShowAsDateTime);
        if Idx <> -1 then
          PopupMenu.Delete(Idx);
        FreeAndNil(mi_ShowAsDateTime);
      end;
    end;
  except
    on e: Exception do begin
      asm nop end;
      // ignore, we don't want any exceptions to prevent the IDE from closing
    end;
  end;
end;

type
  TVTFocusChangingEvent = procedure(Sender: TBaseVirtualTree; OldNode, NewNode: PVirtualNode; OldColumn,
    NewColumn: TColumnIndex; var Allowed: Boolean) of object;

procedure TDebugVisualizerWatchFormHandler.InstallTreeViewHooks;
var
  OrigEvent: TMethod;
  NewEvent: TVSTGetTextEvent;
  Method: TVTFocusChangingEvent;
begin
  try
    OrigEvent := GetMethodProp(FWatchTree, 'OnGetText');
    FGetTextHook := TVSTGetTextEventHook.Create(TVSTGetTextEvent(OrigEvent), OnGetTextHook);
    NewEvent := FGetTextHook.HandleEvent;
    SetMethodProp(FWatchTree, 'OnGetText', TMethod(NewEvent));
    Method := FocusChanging;
    SetMethodProp(FWatchTree, 'OnFocusChanging', TMethod(Method));
  except
    on e: Exception do begin
      asm nop end;
      // ignore
	end;
  end;
end;

procedure TDebugVisualizerWatchFormHandler.UninstallTreeViewHooks;
var
  Ptr: TMethod;
  PrevHook: TVSTGetTextEventHook;
  Method: TMethod;
begin
  if not Assigned(FWatchTree) then
    Exit; //==>

  Method.Code := nil;
  Method.Data := nil;
  SetMethodProp(FWatchTree, 'OnFocusChanging', TMethod(Method));

  if not Assigned(FGetTextHook) then
    Exit; //==>

  Ptr := GetMethodProp(FWatchTree, 'OnGetText');

  if not Assigned(Ptr.Data) and not Assigned(Ptr.Code) then begin
    // Somebody has assigned NIL to the event.
    // It's probably safe to assume that there will be no reference
    // to our hook left, so we just free the object and be done.
    FGetTextHook.Free;
    Exit;
  end;

  if Ptr.Data = FGetTextHook then begin
    // We are lucky, nobody has tampered with the event,
    // we can just assign the original event,
    // free the hook object and be done with it.
    SetMethodProp(FWatchTree, 'OnGetText', TMethod(FGetTextHook.OrigEvent));
    FGetTextHook.Free;
    Exit;
  end;

  if TObject(Ptr.Data).ClassNameIs(FGetTextHook.ClassName) then begin
    // Somebody else, who knows about this standard, has hooked the event.
    // (Remember: Do not change the class name or the class
    // structure. Otherwise this check will fail!)
    // Let's check whether we can find our own hook in the chain.
    PrevHook := Ptr.Data;
    Ptr := TMethod(PrevHook.OrigEvent);
    while Assigned(Ptr.Data) and TObject(Ptr.Data).ClassNameIs(FGetTextHook.ClassName) do begin
      if Ptr.Data = FGetTextHook then begin
        // We found our own hook. Remove it from the chain and be done.
        PrevHook.OrigEvent := FGetTextHook.OrigEvent;
        FGetTextHook.Free;
        Exit;
      end;
      // check the next hook in the chain
      PrevHook := Ptr.Data;
      Ptr := TMethod(TVSTGetTextEventHook(Ptr.Data).OrigEvent);
    end;
  end;

  // If we get here, somebody, who does not adhere to this standard,
  // has changed the event. The best thing we can do, is assign NIL
  // to the HookEvent so it no longer gets called.
  // We cannot free the hook because somebody might still have a
  // reference to FEventHook.HandleEvent. So there will be a small
  // memory leak.
  TMethod(FGetTextHook.HookEvent).Code := nil;
  TMethod(FGetTextHook.HookEvent).Data := nil;
end;

procedure TDebugVisualizerWatchFormHandler.OnGetTextHook(_Sender: TBaseVirtualTree;
  _Node: PVirtualNode; _Column: TColumnIndex; _TextType: TVSTTextType;
  var _CellText: UnicodeString);
var
  Handler: TVisualizerHandler;
begin
  if _Column = 0 then begin
    FLastExpression := _CellText;
  end else if _Column = 1 then begin
    FLastValuelText := _CellText;
    if FVisualizerList.TryGetHandler(_Node, Handler) then
      Handler.ConvertValue(_CellText);
  end;
end;

procedure TDebugVisualizerWatchFormHandler.FocusChanging(_Sender: TBaseVirtualTree; _OldNode,
  _NewNode: PVirtualNode; _OldColumn, _NewColumn: TColumnIndex; var Allowed: Boolean);
var
  Handler: TVisualizerHandler;
begin
  if not Assigned(_NewNode) then
    Exit;

  FLastVSTNode := _NewNode;
  mi_ShowAsDateTime.Checked := False;
  mi_ShowMultilineString.Checked := False;
  if FVisualizerList.TryGetHandler(_NewNode, Handler) then begin
    Handler.CheckMenuIfActive;
  end;
end;

procedure TDebugVisualizerWatchFormHandler.mi_ShowAsDateTimeClick(Sender: TObject);
begin
  if not Assigned(FVisualizerList) or not Assigned(FLastVSTNode) then
    Exit; //==>

  FVisualizerList.Delete(FLastVSTNode);

  if not mi_ShowAsDateTime.Checked then
    FVisualizerList.Add(TGetAsDateTimeHandler.Create(FLastVSTNode, mi_ShowAsDateTime, FLastExpression));
  FWatchTree.Invalidate;
end;

procedure TDebugVisualizerWatchFormHandler.mi_ShowMultilineStringClick(Sender: TObject);
begin
  if not Assigned(FVisualizerList) or not Assigned(FLastVSTNode) then
    Exit; //==>

  FVisualizerList.Delete(FLastVSTNode);

  FVisualizerList.Add(TShowMultlineHandler.Create(FLastVSTNode, mi_ShowMultilineString, FLastExpression));
  FWatchTree.Invalidate;
end;

end.
