unit u_dzDialogUtils;

{$INCLUDE 'dzlib.inc'}

interface

uses
  Classes,
  Controls,
  u_dzTranslator;

type
  TFileFilter = record
  private
    FDescription: string;
    FMask: string;
  public
    procedure Init(const _Description, _Mask: string; _AddMaskToDesc: Boolean);
    property Description: string read FDescription;
    property Mask: string read FMask;
  end;

type
  TFileFilterArr = array of TFileFilter;

  PFileFilterBuilder = ^TFileFilterBuilder;
  TFileFilterBuilder = record
  private
  private
    FFilters: TFileFilterArr;
    FIncludeAllFiles: Boolean;
    FAllSupported: string;
  public
    function Init(_IncludeAllFiles: Boolean = True; const _AllSupported: string = ''): PFileFilterBuilder;
    ///<summary> Adds a new filter, the first one added is the default.
    ///          @param Description is the file type description e.g. 'Text file'. This descrition
    ///                             should be localized.
    ///          @param Mask is the file mask, e.g. '*.txt', it should not be localized.
    ///          @param AddMaskToDesc determines whether the mask is appended to the descripition as
    ///                               ' (*.bla)', defaults to true, e.g. 'Text file (*.txt)
    ///          @returns the interface itself, so chaining is possible like this:
    ///                   od.Filter := FileFilterHelper.AddFilter('bla', '*.bla').AddFilter('blub', '*.blu').Value;
    function Add(const _Description: string; const _Mask: string;
      _AddMaskToDesc: Boolean = True): PFileFilterBuilder;
    ///<summary>
    /// Adds an entry <descriptionfmt> (<maskfmt>)|<maskfmt>
    /// where both *fmts are passed through Format(*fmt, Values) </summary>
    function AddFmt(const _DescriptionFmt: string; const _MaskFmt: string;
      const _Values: array of const): PFileFilterBuilder;
    function AddAvi: PFileFilterBuilder;
    function AddBmp: PFileFilterBuilder;
    function AddCsv: PFileFilterBuilder;
    function AddDbf: PFileFilterBuilder;
    function AddEmf: PFileFilterBuilder;
    function AddExe: PFileFilterBuilder;
    function AddGif: PFileFilterBuilder;
    function AddHtml: PFileFilterBuilder;
    function AddIni: PFileFilterBuilder;
    function AddJpg: PFileFilterBuilder;
    function AddLog: PFileFilterBuilder;
    function AddMdb: PFileFilterBuilder;
    function AddOdt: PFileFilterBuilder;
    function AddOds: PFileFilterBuilder;
    function AddPdf: PFileFilterBuilder;
    function AddPicture: PFileFilterBuilder;
    function AddRtf: PFileFilterBuilder;
    function AddTiff: PFileFilterBuilder;
    function AddTxt: PFileFilterBuilder;
    function AddXml: PFileFilterBuilder;
    function AddXls: PFileFilterBuilder;
    function Value: string;
    function Filter: string; deprecated; // use Value instead
  end;

function FileFilterBuilder(_IncludeAllFiles: Boolean = True; const _AllSupported: string = ''): TFileFilterBuilder;

function TOpenDialog_Execute(_Owner: TWinControl; const _Title: string; const _Filter: string;
  var _fn: string): Boolean;

function TSaveDialog_Execute(_Owner: TWinControl; const _Title: string; const _Filter: string;
  var _fn: string): Boolean;

implementation

uses
  Dialogs,
  SysUtils;

function TOpenDialog_Execute(_Owner: TWinControl; const _Title: string; const _Filter: string;
  var _fn: string): Boolean;
var
  od: TOpenDialog;
begin
  od := TOpenDialog.Create(_Owner);
  try
    od.Name := '';
    od.Title := _Title;
    od.Filter := _Filter;
    od.Options := [ofHideReadOnly, ofPathMustExist, ofFileMustExist, ofEnableSizing];
    od.FileName := ExtractFileName(_fn);
    od.InitialDir := ExtractFileDir(_fn);
    od.DefaultExt := '*';
    Result := od.Execute({$IFDEF OPENDIALOG_EXCUTE_HAS_HANDLE}_Owner.Handle{$ENDIF});
    if not Result then
      Exit;
    _fn := od.FileName;
  finally
    FreeAndNil(od);
  end;
end;

function TSaveDialog_Execute(_Owner: TWinControl; const _Title: string; const _Filter: string;
  var _fn: string): Boolean;
var
  sd: TSaveDialog;
begin
  sd := TSaveDialog.Create(_Owner);
  try
    sd.Name := '';
    sd.Options := [ofOverwritePrompt, ofHideReadOnly, ofPathMustExist, ofEnableSizing];
    sd.Title := _Title;
    sd.Filter := _Filter;
    sd.FileName := ExtractFileName(_fn);
    sd.InitialDir := ExtractFileDir(_fn);
    sd.DefaultExt := '*';
    Result := sd.Execute({$IFDEF OPENDIALOG_EXCUTE_HAS_HANDLE}_Owner.Handle{$ENDIF});
    if not Result then
      Exit;
    _fn := sd.FileName;
  finally
    FreeAndNil(sd);
  end;
end;

function FileFilterBuilder(_IncludeAllFiles: Boolean = True; const _AllSupported: string = ''): TFileFilterBuilder;
begin
  Result.Init(_IncludeAllFiles, _AllSupported);
end;

{ TFileFilterBuilder }

function TFileFilterBuilder.Init(_IncludeAllFiles: Boolean;
  const _AllSupported: string): PFileFilterBuilder;
begin
  FIncludeAllFiles := _IncludeAllFiles;
  FAllSupported := _AllSupported;
  Result := @Self;
end;

function TFileFilterBuilder.Add(const _Description, _Mask: string;
  _AddMaskToDesc: Boolean): PFileFilterBuilder;
var
  Idx: Integer;
begin
  Idx := Length(FFilters);
  SetLength(FFilters, Idx + 1);
  FFilters[Idx].Init(_Description, _Mask, _AddMaskToDesc);
  Result := @Self;
end;

function TFileFilterBuilder.AddFmt(const _DescriptionFmt, _MaskFmt: string;
  const _Values: array of const): PFileFilterBuilder;
begin
  Result := Add(Format(_DescriptionFmt, _Values), Format(_MaskFmt, _Values));
end;

function TFileFilterBuilder.AddAvi: PFileFilterBuilder;
begin
  Result := Add(_('AVI files'), '*.AVI');
end;

function TFileFilterBuilder.AddBmp: PFileFilterBuilder;
begin
  Result := Add(_('Bitmap Files'), '*.BMP');
end;

function TFileFilterBuilder.AddCsv: PFileFilterBuilder;
begin
  Result := Add(_('Comma-separated values'), '*.CSV');
end;

function TFileFilterBuilder.AddDbf: PFileFilterBuilder;
begin
  Result := Add(_('DBase tables'), '*.DBF');
end;

function TFileFilterBuilder.AddEmf: PFileFilterBuilder;
begin
  Result := Add(_('Windows Enhanced Metafile'), '*.EMF');
end;

function TFileFilterBuilder.AddExe: PFileFilterBuilder;
begin
  Result := Add(_('Executable Files'), '*.EXE');
end;

function TFileFilterBuilder.AddGif: PFileFilterBuilder;
begin
  Result := Add(_('GIF Image'), '*.GIF');
end;

function TFileFilterBuilder.AddHtml: PFileFilterBuilder;
begin
  Result := Add(_('Hypertext Markup Language'), '*.html');
end;

function TFileFilterBuilder.AddIni: PFileFilterBuilder;
begin
  Result := Add(_('INI files'), '*.INI');
end;

function TFileFilterBuilder.AddJpg: PFileFilterBuilder;
begin
  Result := Add(_('JPEG Files'), '*.jpg;*.jpeg');
end;

function TFileFilterBuilder.AddLog: PFileFilterBuilder;
begin
  Result := Add(_('Log files'), '*.LOG');
end;

function TFileFilterBuilder.AddMdb: PFileFilterBuilder;
begin
  Result := Add(_('Microsoft Access Databases'), '*.mdb');
end;

function TFileFilterBuilder.AddOds: PFileFilterBuilder;
begin
  Result := Add(_('Open Document Spreadsheet'), '*.ODS');
end;

function TFileFilterBuilder.AddOdt: PFileFilterBuilder;
begin
  Result := Add(_('Open Document Text'), '*.ODT');
end;

function TFileFilterBuilder.AddPdf: PFileFilterBuilder;
begin
  Result := Add(_('Portable Document Format'), '*.PDF');
end;

function TFileFilterBuilder.AddPicture: PFileFilterBuilder;
begin
  Result := Add(_('Picture files'), '*.bmp;*.jpg;*.jpeg').AddBmp.AddJpg;
end;

function TFileFilterBuilder.AddRtf: PFileFilterBuilder;
begin
  Result := Add(_('Rich Text Format'), '*.RTF');
end;

function TFileFilterBuilder.AddTiff: PFileFilterBuilder;
begin
  Result := Add(_('Tagged Image File Format'), '*.TIF;*.TIFF');
end;

function TFileFilterBuilder.AddTxt: PFileFilterBuilder;
begin
  Result := Add(_('Text files'), '*.TXT');
end;

function TFileFilterBuilder.AddXls: PFileFilterBuilder;
begin
  Result := Add(_('Microsoft Excel File Format'), '*.XLS');
end;

function TFileFilterBuilder.AddXml: PFileFilterBuilder;
begin
  Result := Add(_('Extensible Markup Language'), '*.XML');
end;

function TFileFilterBuilder.Filter: string;
begin
  Result := Value;
end;

function TFileFilterBuilder.Value: string;

  procedure AddToResult(const _Description, _Mask: string); overload;
  begin
    if Result <> '' then
      Result := Result + '|';
    Result := Result + _Description + '|' + _Mask;
  end;

  procedure AddToResult(_Filter: TFileFilter); overload;
  begin
    AddToResult(_Filter.Description, _Filter.Mask);
  end;

var
  i: Integer;
  s: string;
begin
  Result := '';

  if FAllSupported <> '' then begin
    s := '';
    for i := Low(FFilters) to High(FFilters) do begin
      if s <> '' then
        s := s + ';';
      s := s + FFilters[i].Mask;
    end;
    AddToResult(FAllSupported, s);
  end;

  for i := Low(FFilters) to High(FFilters) do begin
    AddToResult(FFilters[i]);
  end;

  if FIncludeAllFiles then
    AddToResult(_('all files') + ' (*.*)', '*.*');
end;

{ TFileFilter }

procedure TFileFilter.Init(const _Description, _Mask: string; _AddMaskToDesc: Boolean);
begin
  FDescription := _Description;
  FMask := _Mask;
  if _AddMaskToDesc then
    FDescription := FDescription + ' (' + _Mask + ')';
end;

end.

