unit u_dzWmMessageToString;

{$INCLUDE 'dzlib.inc'}

interface

uses
  SysUtils,
  Classes,
  Messages,
  u_dzTranslator;

type
  TWmMessageToString = class
  private
    FList: TObject;
    procedure InitList;
    procedure Reg(_WmMsg: Cardinal; const _WmStr: string);
  public
    constructor Create;
    destructor Destroy; override;
    function MsgToString(const _WmMsg: Cardinal): string; overload;
    function MsgToString(const _Msg: TMessage): string; overload;
  end;

implementation

uses
  Controls,
  u_dzConvertUtils,
  u_dzQuickSort;

type
  TMessageStringItem = class
  private
    FWmMsg: Cardinal;
    FWmStrings: TStringList;
  public
    constructor Create(_WmMsg: Cardinal; const _WmStr: string);
    destructor Destroy; override;
    procedure Add(const _WmStr: string);
    function FirstWmStr: string;
    function AllWmStr: string;
    property WmMsg: Cardinal read FWmMsg;
    property WmStrings: TStringList read FWmStrings;
  end;

{ TMessageStringItem }

constructor TMessageStringItem.Create(_WmMsg: Cardinal; const _WmStr: string);
begin
  inherited Create;
  FWmMsg := _WmMsg;
  FWmStrings := TStringList.Create;
  FWmStrings.Delimiter := ' ';
  FWmStrings.Add(_WmStr);
end;

destructor TMessageStringItem.Destroy;
begin
  FreeAndNil(FWmStrings);
  inherited;
end;

function TMessageStringItem.FirstWmStr: string;
begin
  Result := FWmStrings[0];
end;

function TMessageStringItem.AllWmStr: string;
begin
  Result := FWmStrings.DelimitedText;
end;

procedure TMessageStringItem.Add(const _WmStr: string);
begin
  if FWmStrings.IndexOf(_WmStr) = -1 then
    FWmStrings.Add(_WmStr);
end;

{$DEFINE __DZ_SORTED_OBJECT_LIST_TEMPLATE__}
type
  _LIST_ANCESTOR_ = TObject;
  _ITEM_TYPE_ = TMessageStringItem;
  _KEY_TYPE_ = Cardinal;
{$INCLUDE 't_dzSortedObjectListTemplate.tpl'}

type
  TMessageStringItemList = class(_DZ_SORTED_OBJECT_LIST_TEMPLATE_)
  protected
    procedure DuplicateError(const _Item: TMessageStringItem); override;
    function KeyOf(const _Item: TMessageStringItem): Cardinal; override;
    function Compare(const _Key1, _Key2: Cardinal): Integer; override;
  end;

{$INCLUDE 't_dzSortedObjectListTemplate.tpl'}

procedure TMessageStringItemList.DuplicateError(const _Item: TMessageStringItem);
begin
  raise Exception.CreateFmt(_('[%s] does not allow duplicates when trying to insert "%s"'),
    [ClassName, _Item.FirstWmStr]);
end;

function TMessageStringItemList.KeyOf(const _Item: TMessageStringItem): Cardinal;
begin
  Result := _Item.WmMsg;
end;

function TMessageStringItemList.Compare(const _Key1, _Key2: Cardinal): Integer;
begin
  Result := Int64(_Key1) - Int64(_Key2);
end;

{ TWmMessageToString }

constructor TWmMessageToString.Create;
var
  List: TMessageStringItemList;
begin
  inherited;
  List := TMessageStringItemList.Create;
  FList := List;
  InitList;
end;

destructor TWmMessageToString.Destroy;
begin
  FreeAndNil(FList);
  inherited;
end;

function TWmMessageToString.MsgToString(const _WmMsg: Cardinal): string;
var
  Item: TMessageStringItem;
begin
  if (FList as TMessageStringItemList).Find(_WmMsg, Item) then
    Result := Item.AllWmStr
  else
    Result := '$' + Long2Hex(_WmMsg) + ' (' + IntToStr(_WmMsg) + ')';
end;

function TWmMessageToString.MsgToString(const _Msg: TMessage): string;
begin
  Result := Format('%0:s (WParam: %1:x (%1:d), LParam: %2:x (%2:d))',
    [MsgToString(_Msg.Msg), _Msg.wParam, _Msg.LParam]);
end;

procedure TWmMessageToString.Reg(_WmMsg: Cardinal; const _WmStr: string);
var
  List: TMessageStringItemList;
  Item: TMessageStringItem;
begin
  List := (FList as TMessageStringItemList);
  if List.Find(_WmMsg, Item) then
    Item.Add(_WmStr)
  else
    List.Add(TMessageStringItem.Create(_WmMsg, _WmStr));
end;

procedure TWmMessageToString.InitList;
begin
  // From Messages.pas
  Reg(WM_NULL, 'WM_NULL');
  Reg(WM_CREATE, 'WM_CREATE');
  Reg(WM_DESTROY, 'WM_DESTROY');
  Reg(WM_MOVE, 'WM_MOVE');
  Reg(WM_SIZE, 'WM_SIZE');
  Reg(WM_ACTIVATE, 'WM_ACTIVATE');
  Reg(WM_SETFOCUS, 'WM_SETFOCUS');
  Reg(WM_KILLFOCUS, 'WM_KILLFOCUS');
  Reg(WM_ENABLE, 'WM_ENABLE');
  Reg(WM_SETREDRAW, 'WM_SETREDRAW');
  Reg(WM_SETTEXT, 'WM_SETTEXT');
  Reg(WM_GETTEXT, 'WM_GETTEXT');
  Reg(WM_GETTEXTLENGTH, 'WM_GETTEXTLENGTH');
  Reg(WM_PAINT, 'WM_PAINT');
  Reg(WM_CLOSE, 'WM_CLOSE');
  Reg(WM_QUERYENDSESSION, 'WM_QUERYENDSESSION');
  Reg(WM_QUIT, 'WM_QUIT');
  Reg(WM_QUERYOPEN, 'WM_QUERYOPEN');
  Reg(WM_ERASEBKGND, 'WM_ERASEBKGND');
  Reg(WM_SYSCOLORCHANGE, 'WM_SYSCOLORCHANGE');
  Reg(WM_ENDSESSION, 'WM_ENDSESSION');
  Reg(WM_SYSTEMERROR, 'WM_SYSTEMERROR');
  Reg(WM_SHOWWINDOW, 'WM_SHOWWINDOW');
  Reg(WM_CTLCOLOR, 'WM_CTLCOLOR');
  Reg(WM_WININICHANGE, 'WM_WININICHANGE');
  Reg(WM_SETTINGCHANGE, 'WM_SETTINGCHANGE');
  Reg(WM_DEVMODECHANGE, 'WM_DEVMODECHANGE');
  Reg(WM_ACTIVATEAPP, 'WM_ACTIVATEAPP');
  Reg(WM_FONTCHANGE, 'WM_FONTCHANGE');
  Reg(WM_TIMECHANGE, 'WM_TIMECHANGE');
  Reg(WM_CANCELMODE, 'WM_CANCELMODE');
  Reg(WM_SETCURSOR, 'WM_SETCURSOR');
  Reg(WM_MOUSEACTIVATE, 'WM_MOUSEACTIVATE');
  Reg(WM_CHILDACTIVATE, 'WM_CHILDACTIVATE');
  Reg(WM_QUEUESYNC, 'WM_QUEUESYNC');
  Reg(WM_GETMINMAXINFO, 'WM_GETMINMAXINFO');
  Reg(WM_PAINTICON, 'WM_PAINTICON');
  Reg(WM_ICONERASEBKGND, 'WM_ICONERASEBKGND');
  Reg(WM_NEXTDLGCTL, 'WM_NEXTDLGCTL');
  Reg(WM_SPOOLERSTATUS, 'WM_SPOOLERSTATUS');
  Reg(WM_DRAWITEM, 'WM_DRAWITEM');
  Reg(WM_MEASUREITEM, 'WM_MEASUREITEM');
  Reg(WM_DELETEITEM, 'WM_DELETEITEM');
  Reg(WM_VKEYTOITEM, 'WM_VKEYTOITEM');
  Reg(WM_CHARTOITEM, 'WM_CHARTOITEM');
  Reg(WM_SETFONT, 'WM_SETFONT');
  Reg(WM_GETFONT, 'WM_GETFONT');
  Reg(WM_SETHOTKEY, 'WM_SETHOTKEY');
  Reg(WM_GETHOTKEY, 'WM_GETHOTKEY');
  Reg(WM_QUERYDRAGICON, 'WM_QUERYDRAGICON');
  Reg(WM_COMPAREITEM, 'WM_COMPAREITEM');
  Reg(WM_GETOBJECT, 'WM_GETOBJECT');
  Reg(WM_COMPACTING, 'WM_COMPACTING');
  Reg(WM_COMMNOTIFY, 'WM_COMMNOTIFY');
  Reg(WM_WINDOWPOSCHANGING, 'WM_WINDOWPOSCHANGING');
  Reg(WM_WINDOWPOSCHANGED, 'WM_WINDOWPOSCHANGED');
  Reg(WM_POWER, 'WM_POWER');
  Reg(WM_COPYDATA, 'WM_COPYDATA');
  Reg(WM_CANCELJOURNAL, 'WM_CANCELJOURNAL');
  Reg(WM_NOTIFY, 'WM_NOTIFY');
  Reg(WM_INPUTLANGCHANGEREQUEST, 'WM_INPUTLANGCHANGEREQUEST');
  Reg(WM_INPUTLANGCHANGE, 'WM_INPUTLANGCHANGE');
  Reg(WM_TCARD, 'WM_TCARD');
  Reg(WM_HELP, 'WM_HELP');
  Reg(WM_USERCHANGED, 'WM_USERCHANGED');
  Reg(WM_NOTIFYFORMAT, 'WM_NOTIFYFORMAT');
  Reg(WM_CONTEXTMENU, 'WM_CONTEXTMENU');
  Reg(WM_STYLECHANGING, 'WM_STYLECHANGING');
  Reg(WM_STYLECHANGED, 'WM_STYLECHANGED');
  Reg(WM_DISPLAYCHANGE, 'WM_DISPLAYCHANGE');
  Reg(WM_GETICON, 'WM_GETICON');
  Reg(WM_SETICON, 'WM_SETICON');
  Reg(WM_NCCREATE, 'WM_NCCREATE');
  Reg(WM_NCDESTROY, 'WM_NCDESTROY');
  Reg(WM_NCCALCSIZE, 'WM_NCCALCSIZE');
  Reg(WM_NCHITTEST, 'WM_NCHITTEST');
  Reg(WM_NCPAINT, 'WM_NCPAINT');
  Reg(WM_NCACTIVATE, 'WM_NCACTIVATE');
  Reg(WM_GETDLGCODE, 'WM_GETDLGCODE');
  Reg(WM_NCMOUSEMOVE, 'WM_NCMOUSEMOVE');
  Reg(WM_NCLBUTTONDOWN, 'WM_NCLBUTTONDOWN');
  Reg(WM_NCLBUTTONUP, 'WM_NCLBUTTONUP');
  Reg(WM_NCLBUTTONDBLCLK, 'WM_NCLBUTTONDBLCLK');
  Reg(WM_NCRBUTTONDOWN, 'WM_NCRBUTTONDOWN');
  Reg(WM_NCRBUTTONUP, 'WM_NCRBUTTONUP');
  Reg(WM_NCRBUTTONDBLCLK, 'WM_NCRBUTTONDBLCLK');
  Reg(WM_NCMBUTTONDOWN, 'WM_NCMBUTTONDOWN');
  Reg(WM_NCMBUTTONUP, 'WM_NCMBUTTONUP');
  Reg(WM_NCMBUTTONDBLCLK, 'WM_NCMBUTTONDBLCLK');
{$IFDEF RTL150_UP}
  Reg(WM_NCXBUTTONDOWN, 'WM_NCXBUTTONDOWN');
  Reg(WM_NCXBUTTONUP, 'WM_NCXBUTTONUP');
  Reg(WM_NCXBUTTONDBLCLK, 'WM_NCXBUTTONDBLCLK');
{$ENDIF}
{$IFDEF RTL210_UP}
  Reg(WM_INPUT_DEVICE_CHANGE, 'WM_INPUT_DEVICE_CHANGE');
{$ELSE}
  Reg($00FE, 'WM_INPUT_DEVICE_CHANGE');
{$ENDIF}
{$IFDEF RTL150_UP}
  Reg(WM_INPUT, 'WM_INPUT');
{$ENDIF}
  Reg(WM_KEYFIRST, 'WM_KEYFIRST');
  Reg(WM_KEYDOWN, 'WM_KEYDOWN');
  Reg(WM_KEYUP, 'WM_KEYUP');
  Reg(WM_CHAR, 'WM_CHAR');
  Reg(WM_DEADCHAR, 'WM_DEADCHAR');
  Reg(WM_SYSKEYDOWN, 'WM_SYSKEYDOWN');
  Reg(WM_SYSKEYUP, 'WM_SYSKEYUP');
  Reg(WM_SYSCHAR, 'WM_SYSCHAR');
  Reg(WM_SYSDEADCHAR, 'WM_SYSDEADCHAR');
{$IFDEF RTL160_UP}
  Reg(WM_UNICHAR, 'WM_UNICHAR');
{$ENDIF}
  Reg(WM_KEYLAST, 'WM_KEYLAST');
  Reg(WM_INITDIALOG, 'WM_INITDIALOG');
  Reg(WM_COMMAND, 'WM_COMMAND');
  Reg(WM_SYSCOMMAND, 'WM_SYSCOMMAND');
  Reg(WM_TIMER, 'WM_TIMER');
  Reg(WM_HSCROLL, 'WM_HSCROLL');
  Reg(WM_VSCROLL, 'WM_VSCROLL');
  Reg(WM_INITMENU, 'WM_INITMENU');
  Reg(WM_INITMENUPOPUP, 'WM_INITMENUPOPUP');
{$IFDEF RTL210_UP}
  Reg(WM_GESTURE, 'WM_GESTURE');
  Reg(WM_GESTURENOTIFY, 'WM_GESTURENOTIFY');
{$ENDIF}
  Reg(WM_MENUSELECT, 'WM_MENUSELECT');
  Reg(WM_MENUCHAR, 'WM_MENUCHAR');
  Reg(WM_ENTERIDLE, 'WM_ENTERIDLE');
  Reg(WM_MENURBUTTONUP, 'WM_MENURBUTTONUP');
  Reg(WM_MENUDRAG, 'WM_MENUDRAG');
  Reg(WM_MENUGETOBJECT, 'WM_MENUGETOBJECT');
  Reg(WM_UNINITMENUPOPUP, 'WM_UNINITMENUPOPUP');
  Reg(WM_MENUCOMMAND, 'WM_MENUCOMMAND');
  Reg(WM_CHANGEUISTATE, 'WM_CHANGEUISTATE');
  Reg(WM_UPDATEUISTATE, 'WM_UPDATEUISTATE');
  Reg(WM_QUERYUISTATE, 'WM_QUERYUISTATE');
  Reg(WM_CTLCOLORMSGBOX, 'WM_CTLCOLORMSGBOX');
  Reg(WM_CTLCOLOREDIT, 'WM_CTLCOLOREDIT');
  Reg(WM_CTLCOLORLISTBOX, 'WM_CTLCOLORLISTBOX');
  Reg(WM_CTLCOLORBTN, 'WM_CTLCOLORBTN');
  Reg(WM_CTLCOLORDLG, 'WM_CTLCOLORDLG');
  Reg(WM_CTLCOLORSCROLLBAR, 'WM_CTLCOLORSCROLLBAR');
  Reg(WM_CTLCOLORSTATIC, 'WM_CTLCOLORSTATIC');
  Reg(WM_MOUSEFIRST, 'WM_MOUSEFIRST');
  Reg(WM_MOUSEMOVE, 'WM_MOUSEMOVE');
  Reg(WM_LBUTTONDOWN, 'WM_LBUTTONDOWN');
  Reg(WM_LBUTTONUP, 'WM_LBUTTONUP');
  Reg(WM_LBUTTONDBLCLK, 'WM_LBUTTONDBLCLK');
  Reg(WM_RBUTTONDOWN, 'WM_RBUTTONDOWN');
  Reg(WM_RBUTTONUP, 'WM_RBUTTONUP');
  Reg(WM_RBUTTONDBLCLK, 'WM_RBUTTONDBLCLK');
  Reg(WM_MBUTTONDOWN, 'WM_MBUTTONDOWN');
  Reg(WM_MBUTTONUP, 'WM_MBUTTONUP');
  Reg(WM_MBUTTONDBLCLK, 'WM_MBUTTONDBLCLK');
  Reg(WM_MOUSEWHEEL, 'WM_MOUSEWHEEL');
{$IFDEF RTL210_UP}
  Reg(WM_XBUTTONDOWN, 'WM_XBUTTONDOWN');
  Reg(WM_XBUTTONUP, 'WM_XBUTTONUP');
  Reg(WM_XBUTTONDBLCLK, 'WM_XBUTTONDBLCLK');
  Reg(WM_MOUSEHWHEEL, 'WM_MOUSEHWHEEL');
{$ENDIF}
  Reg(WM_MOUSELAST, 'WM_MOUSELAST');
  Reg(WM_PARENTNOTIFY, 'WM_PARENTNOTIFY');
  Reg(WM_ENTERMENULOOP, 'WM_ENTERMENULOOP');
  Reg(WM_EXITMENULOOP, 'WM_EXITMENULOOP');
  Reg(WM_NEXTMENU, 'WM_NEXTMENU');
  Reg(WM_SIZING, 'WM_SIZING');
  Reg(WM_CAPTURECHANGED, 'WM_CAPTURECHANGED');
  Reg(WM_MOVING, 'WM_MOVING');
  Reg(WM_POWERBROADCAST, 'WM_POWERBROADCAST');
  Reg(WM_DEVICECHANGE, 'WM_DEVICECHANGE');
  Reg(WM_IME_STARTCOMPOSITION, 'WM_IME_STARTCOMPOSITION');
  Reg(WM_IME_ENDCOMPOSITION, 'WM_IME_ENDCOMPOSITION');
  Reg(WM_IME_COMPOSITION, 'WM_IME_COMPOSITION');
  Reg(WM_IME_KEYLAST, 'WM_IME_KEYLAST');
  Reg(WM_IME_SETCONTEXT, 'WM_IME_SETCONTEXT');
  Reg(WM_IME_NOTIFY, 'WM_IME_NOTIFY');
  Reg(WM_IME_CONTROL, 'WM_IME_CONTROL');
  Reg(WM_IME_COMPOSITIONFULL, 'WM_IME_COMPOSITIONFULL');
  Reg(WM_IME_SELECT, 'WM_IME_SELECT');
  Reg(WM_IME_CHAR, 'WM_IME_CHAR');
  Reg(WM_IME_REQUEST, 'WM_IME_REQUEST');
  Reg(WM_IME_KEYDOWN, 'WM_IME_KEYDOWN');
  Reg(WM_IME_KEYUP, 'WM_IME_KEYUP');
  Reg(WM_MDICREATE, 'WM_MDICREATE');
  Reg(WM_MDIDESTROY, 'WM_MDIDESTROY');
  Reg(WM_MDIACTIVATE, 'WM_MDIACTIVATE');
  Reg(WM_MDIRESTORE, 'WM_MDIRESTORE');
  Reg(WM_MDINEXT, 'WM_MDINEXT');
  Reg(WM_MDIMAXIMIZE, 'WM_MDIMAXIMIZE');
  Reg(WM_MDITILE, 'WM_MDITILE');
  Reg(WM_MDICASCADE, 'WM_MDICASCADE');
  Reg(WM_MDIICONARRANGE, 'WM_MDIICONARRANGE');
  Reg(WM_MDIGETACTIVE, 'WM_MDIGETACTIVE');
  Reg(WM_MDISETMENU, 'WM_MDISETMENU');
  Reg(WM_ENTERSIZEMOVE, 'WM_ENTERSIZEMOVE');
  Reg(WM_EXITSIZEMOVE, 'WM_EXITSIZEMOVE');
  Reg(WM_DROPFILES, 'WM_DROPFILES');
  Reg(WM_MDIREFRESHMENU, 'WM_MDIREFRESHMENU');
{$IFDEF RTL210_UP}
  Reg(WM_TOUCH, 'WM_TOUCH');
{$ENDIF}
  Reg(WM_MOUSEHOVER, 'WM_MOUSEHOVER');
  Reg(WM_MOUSELEAVE, 'WM_MOUSELEAVE');
{$IFDEF RTL150_UP}
  Reg(WM_NCMOUSEHOVER, 'WM_NCMOUSEHOVER');
  Reg(WM_NCMOUSELEAVE, 'WM_NCMOUSELEAVE');
  Reg(WM_WTSSESSION_CHANGE, 'WM_WTSSESSION_CHANGE');
  Reg(WM_TABLET_FIRST, 'WM_TABLET_FIRST');
  Reg(WM_TABLET_LAST, 'WM_TABLET_LAST');
{$ENDIF}
  Reg(WM_CUT, 'WM_CUT');
  Reg(WM_COPY, 'WM_COPY');
  Reg(WM_PASTE, 'WM_PASTE');
  Reg(WM_CLEAR, 'WM_CLEAR');
  Reg(WM_UNDO, 'WM_UNDO');
  Reg(WM_RENDERFORMAT, 'WM_RENDERFORMAT');
  Reg(WM_RENDERALLFORMATS, 'WM_RENDERALLFORMATS');
  Reg(WM_DESTROYCLIPBOARD, 'WM_DESTROYCLIPBOARD');
  Reg(WM_DRAWCLIPBOARD, 'WM_DRAWCLIPBOARD');
  Reg(WM_PAINTCLIPBOARD, 'WM_PAINTCLIPBOARD');
  Reg(WM_VSCROLLCLIPBOARD, 'WM_VSCROLLCLIPBOARD');
  Reg(WM_SIZECLIPBOARD, 'WM_SIZECLIPBOARD');
  Reg(WM_ASKCBFORMATNAME, 'WM_ASKCBFORMATNAME');
  Reg(WM_CHANGECBCHAIN, 'WM_CHANGECBCHAIN');
  Reg(WM_HSCROLLCLIPBOARD, 'WM_HSCROLLCLIPBOARD');
  Reg(WM_QUERYNEWPALETTE, 'WM_QUERYNEWPALETTE');
  Reg(WM_PALETTEISCHANGING, 'WM_PALETTEISCHANGING');
  Reg(WM_PALETTECHANGED, 'WM_PALETTECHANGED');
  Reg(WM_HOTKEY, 'WM_HOTKEY');
  Reg(WM_PRINT, 'WM_PRINT');
  Reg(WM_PRINTCLIENT, 'WM_PRINTCLIENT');
{$IFDEF RTL150_UP}
  Reg(WM_APPCOMMAND, 'WM_APPCOMMAND');
  Reg(WM_THEMECHANGED, 'WM_THEMECHANGED');
{$ENDIF}
{$IFDEF RTL210_UP}
  Reg(WM_CLIPBOARDUPDATE, 'WM_CLIPBOARDUPDATE');
{$ENDIF}
  Reg(WM_HANDHELDFIRST, 'WM_HANDHELDFIRST');
  Reg(WM_HANDHELDLAST, 'WM_HANDHELDLAST');
  Reg(WM_PENWINFIRST, 'WM_PENWINFIRST');
  Reg(WM_PENWINLAST, 'WM_PENWINLAST');
  Reg(WM_COALESCE_FIRST, 'WM_COALESCE_FIRST');
  Reg(WM_COALESCE_LAST, 'WM_COALESCE_LAST');
  Reg(WM_DDE_FIRST, 'WM_DDE_FIRST');
  Reg(WM_DDE_INITIATE, 'WM_DDE_INITIATE');
  Reg(WM_DDE_TERMINATE, 'WM_DDE_TERMINATE');
  Reg(WM_DDE_ADVISE, 'WM_DDE_ADVISE');
  Reg(WM_DDE_UNADVISE, 'WM_DDE_UNADVISE');
  Reg(WM_DDE_ACK, 'WM_DDE_ACK');
  Reg(WM_DDE_DATA, 'WM_DDE_DATA');
  Reg(WM_DDE_REQUEST, 'WM_DDE_REQUEST');
  Reg(WM_DDE_POKE, 'WM_DDE_POKE');
  Reg(WM_DDE_EXECUTE, 'WM_DDE_EXECUTE');
  Reg(WM_DDE_LAST, 'WM_DDE_LAST');
{$IFDEF RTL190_UP}
  Reg(WM_DWMCOMPOSITIONCHANGED, 'WM_DWMCOMPOSITIONCHANGED');
  Reg(WM_DWMNCRENDERINGCHANGED, 'WM_DWMNCRENDERINGCHANGED');
  Reg(WM_DWMCOLORIZATIONCOLORCHANGED, 'WM_DWMCOLORIZATIONCOLORCHANGED');
  Reg(WM_DWMWINDOWMAXIMIZEDCHANGE, 'WM_DWMWINDOWMAXIMIZEDCHANGE');
{$ENDIF}
{$IFDEF RTL210_UP}
  Reg(WM_DWMSENDICONICTHUMBNAIL, 'WM_DWMSENDICONICTHUMBNAIL');
  Reg(WM_DWMSENDICONICLIVEPREVIEWBITMAP, 'WM_DWMSENDICONICLIVEPREVIEWBITMAP');
  Reg(WM_GETTITLEBARINFOEX, 'WM_GETTITLEBARINFOEX');
  Reg(WM_TABLET_DEFBASE, 'WM_TABLET_DEFBASE');
  Reg(WM_TABLET_MAXOFFSET, 'WM_TABLET_MAXOFFSET');
  Reg(WM_TABLET_ADDED, 'WM_TABLET_ADDED');
  Reg(WM_TABLET_DELETED, 'WM_TABLET_DELETED');
  Reg(WM_TABLET_FLICK, 'WM_TABLET_FLICK');
  Reg(WM_TABLET_QUERYSYSTEMGESTURESTATUS, 'WM_TABLET_QUERYSYSTEMGESTURESTATUS');
{$ENDIF}
  Reg(BM_GETCHECK, 'BM_GETCHECK');
  Reg(BM_SETCHECK, 'BM_SETCHECK');
  Reg(BM_GETSTATE, 'BM_GETSTATE');
  Reg(BM_SETSTATE, 'BM_SETSTATE');
  Reg(BM_SETSTYLE, 'BM_SETSTYLE');
  Reg(BM_CLICK, 'BM_CLICK');
  Reg(BM_GETIMAGE, 'BM_GETIMAGE');
  Reg(BM_SETIMAGE, 'BM_SETIMAGE');
{$IFDEF RTL210_UP}
  Reg(BM_SETDONTCLICK, 'BM_SETDONTCLICK');
{$ENDIF}
//  Reg(LBN_ERRSPACE, 'LBN_ERRSPACE');
  Reg(LBN_SELCHANGE, 'LBN_SELCHANGE');
  Reg(LBN_DBLCLK, 'LBN_DBLCLK');
  Reg(LBN_SELCANCEL, 'LBN_SELCANCEL');
  Reg(LBN_SETFOCUS, 'LBN_SETFOCUS');
  Reg(LBN_KILLFOCUS, 'LBN_KILLFOCUS');
  Reg(LB_ADDSTRING, 'LB_ADDSTRING');
  Reg(LB_INSERTSTRING, 'LB_INSERTSTRING');
  Reg(LB_DELETESTRING, 'LB_DELETESTRING');
  Reg(LB_SELITEMRANGEEX, 'LB_SELITEMRANGEEX');
  Reg(LB_RESETCONTENT, 'LB_RESETCONTENT');
  Reg(LB_SETSEL, 'LB_SETSEL');
  Reg(LB_SETCURSEL, 'LB_SETCURSEL');
  Reg(LB_GETSEL, 'LB_GETSEL');
  Reg(LB_GETCURSEL, 'LB_GETCURSEL');
  Reg(LB_GETTEXT, 'LB_GETTEXT');
  Reg(LB_GETTEXTLEN, 'LB_GETTEXTLEN');
  Reg(LB_GETCOUNT, 'LB_GETCOUNT');
  Reg(LB_SELECTSTRING, 'LB_SELECTSTRING');
  Reg(LB_DIR, 'LB_DIR');
  Reg(LB_GETTOPINDEX, 'LB_GETTOPINDEX');
  Reg(LB_FINDSTRING, 'LB_FINDSTRING');
  Reg(LB_GETSELCOUNT, 'LB_GETSELCOUNT');
  Reg(LB_GETSELITEMS, 'LB_GETSELITEMS');
  Reg(LB_SETTABSTOPS, 'LB_SETTABSTOPS');
  Reg(LB_GETHORIZONTALEXTENT, 'LB_GETHORIZONTALEXTENT');
  Reg(LB_SETHORIZONTALEXTENT, 'LB_SETHORIZONTALEXTENT');
  Reg(LB_SETCOLUMNWIDTH, 'LB_SETCOLUMNWIDTH');
  Reg(LB_ADDFILE, 'LB_ADDFILE');
  Reg(LB_SETTOPINDEX, 'LB_SETTOPINDEX');
  Reg(LB_GETITEMRECT, 'LB_GETITEMRECT');
  Reg(LB_GETITEMDATA, 'LB_GETITEMDATA');
  Reg(LB_SETITEMDATA, 'LB_SETITEMDATA');
  Reg(LB_SELITEMRANGE, 'LB_SELITEMRANGE');
  Reg(LB_SETANCHORINDEX, 'LB_SETANCHORINDEX');
  Reg(LB_GETANCHORINDEX, 'LB_GETANCHORINDEX');
  Reg(LB_SETCARETINDEX, 'LB_SETCARETINDEX');
  Reg(LB_GETCARETINDEX, 'LB_GETCARETINDEX');
  Reg(LB_SETITEMHEIGHT, 'LB_SETITEMHEIGHT');
  Reg(LB_GETITEMHEIGHT, 'LB_GETITEMHEIGHT');
  Reg(LB_FINDSTRINGEXACT, 'LB_FINDSTRINGEXACT');
  Reg(LB_SETLOCALE, 'LB_SETLOCALE');
  Reg(LB_GETLOCALE, 'LB_GETLOCALE');
  Reg(LB_SETCOUNT, 'LB_SETCOUNT');
  Reg(LB_INITSTORAGE, 'LB_INITSTORAGE');
  Reg(LB_ITEMFROMPOINT, 'LB_ITEMFROMPOINT');
  Reg(LB_MSGMAX, 'LB_MSGMAX');
//  Reg(CBN_ERRSPACE, 'CBN_ERRSPACE');
  Reg(CBN_SELCHANGE, 'CBN_SELCHANGE');
  Reg(CBN_DBLCLK, 'CBN_DBLCLK');
  Reg(CBN_SETFOCUS, 'CBN_SETFOCUS');
  Reg(CBN_KILLFOCUS, 'CBN_KILLFOCUS');
  Reg(CBN_EDITCHANGE, 'CBN_EDITCHANGE');
  Reg(CBN_EDITUPDATE, 'CBN_EDITUPDATE');
  Reg(CBN_DROPDOWN, 'CBN_DROPDOWN');
  Reg(CBN_CLOSEUP, 'CBN_CLOSEUP');
  Reg(CBN_SELENDOK, 'CBN_SELENDOK');
  Reg(CBN_SELENDCANCEL, 'CBN_SELENDCANCEL');
  Reg(CB_GETEDITSEL, 'CB_GETEDITSEL');
  Reg(CB_LIMITTEXT, 'CB_LIMITTEXT');
  Reg(CB_SETEDITSEL, 'CB_SETEDITSEL');
  Reg(CB_ADDSTRING, 'CB_ADDSTRING');
  Reg(CB_DELETESTRING, 'CB_DELETESTRING');
  Reg(CB_DIR, 'CB_DIR');
  Reg(CB_GETCOUNT, 'CB_GETCOUNT');
  Reg(CB_GETCURSEL, 'CB_GETCURSEL');
  Reg(CB_GETLBTEXT, 'CB_GETLBTEXT');
  Reg(CB_GETLBTEXTLEN, 'CB_GETLBTEXTLEN');
  Reg(CB_INSERTSTRING, 'CB_INSERTSTRING');
  Reg(CB_RESETCONTENT, 'CB_RESETCONTENT');
  Reg(CB_FINDSTRING, 'CB_FINDSTRING');
  Reg(CB_SELECTSTRING, 'CB_SELECTSTRING');
  Reg(CB_SETCURSEL, 'CB_SETCURSEL');
  Reg(CB_SHOWDROPDOWN, 'CB_SHOWDROPDOWN');
  Reg(CB_GETITEMDATA, 'CB_GETITEMDATA');
  Reg(CB_SETITEMDATA, 'CB_SETITEMDATA');
  Reg(CB_GETDROPPEDCONTROLRECT, 'CB_GETDROPPEDCONTROLRECT');
  Reg(CB_SETITEMHEIGHT, 'CB_SETITEMHEIGHT');
  Reg(CB_GETITEMHEIGHT, 'CB_GETITEMHEIGHT');
  Reg(CB_SETEXTENDEDUI, 'CB_SETEXTENDEDUI');
  Reg(CB_GETEXTENDEDUI, 'CB_GETEXTENDEDUI');
  Reg(CB_GETDROPPEDSTATE, 'CB_GETDROPPEDSTATE');
  Reg(CB_FINDSTRINGEXACT, 'CB_FINDSTRINGEXACT');
  Reg(CB_SETLOCALE, 'CB_SETLOCALE');
  Reg(CB_GETLOCALE, 'CB_GETLOCALE');
  Reg(CB_GETTOPINDEX, 'CB_GETTOPINDEX');
  Reg(CB_SETTOPINDEX, 'CB_SETTOPINDEX');
  Reg(CB_GETHORIZONTALEXTENT, 'CB_GETHORIZONTALEXTENT');
  Reg(CB_SETHORIZONTALEXTENT, 'CB_SETHORIZONTALEXTENT');
  Reg(CB_GETDROPPEDWIDTH, 'CB_GETDROPPEDWIDTH');
  Reg(CB_SETDROPPEDWIDTH, 'CB_SETDROPPEDWIDTH');
  Reg(CB_INITSTORAGE, 'CB_INITSTORAGE');
  Reg(CB_MSGMAX, 'CB_MSGMAX');
  Reg(EN_SETFOCUS, 'EN_SETFOCUS');
  Reg(EN_KILLFOCUS, 'EN_KILLFOCUS');
  Reg(EN_CHANGE, 'EN_CHANGE');
  Reg(EN_UPDATE, 'EN_UPDATE');
  Reg(EN_ERRSPACE, 'EN_ERRSPACE');
  Reg(EN_MAXTEXT, 'EN_MAXTEXT');
  Reg(EN_HSCROLL, 'EN_HSCROLL');
  Reg(EN_VSCROLL, 'EN_VSCROLL');
  Reg(EM_GETSEL, 'EM_GETSEL');
  Reg(EM_SETSEL, 'EM_SETSEL');
  Reg(EM_GETRECT, 'EM_GETRECT');
  Reg(EM_SETRECT, 'EM_SETRECT');
  Reg(EM_SETRECTNP, 'EM_SETRECTNP');
  Reg(EM_SCROLL, 'EM_SCROLL');
  Reg(EM_LINESCROLL, 'EM_LINESCROLL');
  Reg(EM_SCROLLCARET, 'EM_SCROLLCARET');
  Reg(EM_GETMODIFY, 'EM_GETMODIFY');
  Reg(EM_SETMODIFY, 'EM_SETMODIFY');
  Reg(EM_GETLINECOUNT, 'EM_GETLINECOUNT');
  Reg(EM_LINEINDEX, 'EM_LINEINDEX');
  Reg(EM_SETHANDLE, 'EM_SETHANDLE');
  Reg(EM_GETHANDLE, 'EM_GETHANDLE');
  Reg(EM_GETTHUMB, 'EM_GETTHUMB');
  Reg(EM_LINELENGTH, 'EM_LINELENGTH');
  Reg(EM_REPLACESEL, 'EM_REPLACESEL');
  Reg(EM_GETLINE, 'EM_GETLINE');
  Reg(EM_LIMITTEXT, 'EM_LIMITTEXT');
  Reg(EM_CANUNDO, 'EM_CANUNDO');
  Reg(EM_UNDO, 'EM_UNDO');
  Reg(EM_FMTLINES, 'EM_FMTLINES');
  Reg(EM_LINEFROMCHAR, 'EM_LINEFROMCHAR');
  Reg(EM_SETTABSTOPS, 'EM_SETTABSTOPS');
  Reg(EM_SETPASSWORDCHAR, 'EM_SETPASSWORDCHAR');
  Reg(EM_EMPTYUNDOBUFFER, 'EM_EMPTYUNDOBUFFER');
  Reg(EM_GETFIRSTVISIBLELINE, 'EM_GETFIRSTVISIBLELINE');
  Reg(EM_SETREADONLY, 'EM_SETREADONLY');
  Reg(EM_SETWORDBREAKPROC, 'EM_SETWORDBREAKPROC');
  Reg(EM_GETWORDBREAKPROC, 'EM_GETWORDBREAKPROC');
  Reg(EM_GETPASSWORDCHAR, 'EM_GETPASSWORDCHAR');
  Reg(EM_SETMARGINS, 'EM_SETMARGINS');
  Reg(EM_GETMARGINS, 'EM_GETMARGINS');
  Reg(EM_SETLIMITTEXT, 'EM_SETLIMITTEXT');
  Reg(EM_GETLIMITTEXT, 'EM_GETLIMITTEXT');
  Reg(EM_POSFROMCHAR, 'EM_POSFROMCHAR');
  Reg(EM_CHARFROMPOS, 'EM_CHARFROMPOS');
  Reg(EM_SETIMESTATUS, 'EM_SETIMESTATUS');
  Reg(EM_GETIMESTATUS, 'EM_GETIMESTATUS');
  Reg(SBM_SETPOS, 'SBM_SETPOS');
  Reg(SBM_GETPOS, 'SBM_GETPOS');
  Reg(SBM_SETRANGE, 'SBM_SETRANGE');
  Reg(SBM_SETRANGEREDRAW, 'SBM_SETRANGEREDRAW');
  Reg(SBM_GETRANGE, 'SBM_GETRANGE');
  Reg(SBM_ENABLE_ARROWS, 'SBM_ENABLE_ARROWS');
  Reg(SBM_SETSCROLLINFO, 'SBM_SETSCROLLINFO');
  Reg(SBM_GETSCROLLINFO, 'SBM_GETSCROLLINFO');
{$IFDEF RTL200_UP}
  Reg(SBM_GETSCROLLBARINFO, 'SBM_GETSCROLLBARINFO');
{$ENDIF}
  Reg(DM_GETDEFID, 'DM_GETDEFID');
  Reg(DM_SETDEFID, 'DM_SETDEFID');
  Reg(DM_REPOSITION, 'DM_REPOSITION');
  Reg(PSM_PAGEINFO, 'PSM_PAGEINFO');
  Reg(PSM_SHEETINFO, 'PSM_SHEETINFO');

  // from controls.pas
  Reg(CM_BASE, 'CM_BASE');
{$IFDEF RTL200_UP}
  Reg(CM_CLROFFSET, 'CM_CLROFFSET');
{$ENDIF}
  Reg(CM_ACTIVATE, 'CM_ACTIVATE');
  Reg(CM_DEACTIVATE, 'CM_DEACTIVATE');
  Reg(CM_GOTFOCUS, 'CM_GOTFOCUS');
  Reg(CM_LOSTFOCUS, 'CM_LOSTFOCUS');
  Reg(CM_CANCELMODE, 'CM_CANCELMODE');
  Reg(CM_DIALOGKEY, 'CM_DIALOGKEY');
  Reg(CM_DIALOGCHAR, 'CM_DIALOGCHAR');
  Reg(CM_FOCUSCHANGED, 'CM_FOCUSCHANGED');
  Reg(CM_PARENTFONTCHANGED, 'CM_PARENTFONTCHANGED');
  Reg(CM_PARENTCOLORCHANGED, 'CM_PARENTCOLORCHANGED');
  Reg(CM_HITTEST, 'CM_HITTEST');
  Reg(CM_VISIBLECHANGED, 'CM_VISIBLECHANGED');
  Reg(CM_ENABLEDCHANGED, 'CM_ENABLEDCHANGED');
  Reg(CM_COLORCHANGED, 'CM_COLORCHANGED');
  Reg(CM_FONTCHANGED, 'CM_FONTCHANGED');
  Reg(CM_CURSORCHANGED, 'CM_CURSORCHANGED');
  Reg(CM_CTL3DCHANGED, 'CM_CTL3DCHANGED');
  Reg(CM_PARENTCTL3DCHANGED, 'CM_PARENTCTL3DCHANGED');
  Reg(CM_TEXTCHANGED, 'CM_TEXTCHANGED');
  Reg(CM_MOUSEENTER, 'CM_MOUSEENTER');
  Reg(CM_MOUSELEAVE, 'CM_MOUSELEAVE');
  Reg(CM_MENUCHANGED, 'CM_MENUCHANGED');
  Reg(CM_APPKEYDOWN, 'CM_APPKEYDOWN');
  Reg(CM_APPSYSCOMMAND, 'CM_APPSYSCOMMAND');
  Reg(CM_BUTTONPRESSED, 'CM_BUTTONPRESSED');
  Reg(CM_SHOWINGCHANGED, 'CM_SHOWINGCHANGED');
  Reg(CM_ENTER, 'CM_ENTER');
  Reg(CM_EXIT, 'CM_EXIT');
  Reg(CM_DESIGNHITTEST, 'CM_DESIGNHITTEST');
  Reg(CM_ICONCHANGED, 'CM_ICONCHANGED');
  Reg(CM_WANTSPECIALKEY, 'CM_WANTSPECIALKEY');
  Reg(CM_INVOKEHELP, 'CM_INVOKEHELP');
  Reg(CM_WINDOWHOOK, 'CM_WINDOWHOOK');
  Reg(CM_RELEASE, 'CM_RELEASE');
  Reg(CM_SHOWHINTCHANGED, 'CM_SHOWHINTCHANGED');
  Reg(CM_PARENTSHOWHINTCHANGED, 'CM_PARENTSHOWHINTCHANGED');
  Reg(CM_SYSCOLORCHANGE, 'CM_SYSCOLORCHANGE');
  Reg(CM_WININICHANGE, 'CM_WININICHANGE');
  Reg(CM_FONTCHANGE, 'CM_FONTCHANGE');
  Reg(CM_TIMECHANGE, 'CM_TIMECHANGE');
  Reg(CM_TABSTOPCHANGED, 'CM_TABSTOPCHANGED');
  Reg(CM_UIACTIVATE, 'CM_UIACTIVATE');
  Reg(CM_UIDEACTIVATE, 'CM_UIDEACTIVATE');
  Reg(CM_DOCWINDOWACTIVATE, 'CM_DOCWINDOWACTIVATE');
  Reg(CM_CONTROLLISTCHANGE, 'CM_CONTROLLISTCHANGE');
  Reg(CM_GETDATALINK, 'CM_GETDATALINK');
  Reg(CM_CHILDKEY, 'CM_CHILDKEY');
  Reg(CM_DRAG, 'CM_DRAG');
  Reg(CM_HINTSHOW, 'CM_HINTSHOW');
  Reg(CM_DIALOGHANDLE, 'CM_DIALOGHANDLE');
  Reg(CM_ISTOOLCONTROL, 'CM_ISTOOLCONTROL');
  Reg(CM_RECREATEWND, 'CM_RECREATEWND');
  Reg(CM_INVALIDATE, 'CM_INVALIDATE');
  Reg(CM_SYSFONTCHANGED, 'CM_SYSFONTCHANGED');
  Reg(CM_CONTROLCHANGE, 'CM_CONTROLCHANGE');
  Reg(CM_CHANGED, 'CM_CHANGED');
  Reg(CM_DOCKCLIENT, 'CM_DOCKCLIENT');
  Reg(CM_UNDOCKCLIENT, 'CM_UNDOCKCLIENT');
  Reg(CM_FLOAT, 'CM_FLOAT');
  Reg(CM_BORDERCHANGED, 'CM_BORDERCHANGED');
  Reg(CM_BIDIMODECHANGED, 'CM_BIDIMODECHANGED');
  Reg(CM_PARENTBIDIMODECHANGED, 'CM_PARENTBIDIMODECHANGED');
  Reg(CM_ALLCHILDRENFLIPPED, 'CM_ALLCHILDRENFLIPPED');
  Reg(CM_ACTIONUPDATE, 'CM_ACTIONUPDATE');
  Reg(CM_ACTIONEXECUTE, 'CM_ACTIONEXECUTE');
  Reg(CM_HINTSHOWPAUSE, 'CM_HINTSHOWPAUSE');
  Reg(CM_DOCKNOTIFICATION, 'CM_DOCKNOTIFICATION');
  Reg(CM_MOUSEWHEEL, 'CM_MOUSEWHEEL');
  Reg(CM_ISSHORTCUT, 'CM_ISSHORTCUT');
{$IFDEF RTL200_UP}
  Reg(CM_UPDATEACTIONS, 'CM_UPDATEACTIONS');
{$ENDIF}
//  Reg(CM_RAWX11EVENT,'CM_RAWX11EVENT');
{$IFDEF RTL160_UP}
  Reg(CM_INVALIDATEDOCKHOST, 'CM_INVALIDATEDOCKHOST');
  Reg(CM_SETACTIVECONTROL, 'CM_SETACTIVECONTROL');
  Reg(CM_POPUPHWNDDESTROY, 'CM_POPUPHWNDDESTROY');
  Reg(CM_CREATEPOPUP, 'CM_CREATEPOPUP');
  Reg(CM_DESTROYHANDLE, 'CM_DESTROYHANDLE');
  Reg(CM_MOUSEACTIVATE, 'CM_MOUSEACTIVATE');
{$ENDIF}
{$IFDEF RTL180_UP}
  Reg(CM_CONTROLLISTCHANGING, 'CM_CONTROLLISTCHANGING');
{$ENDIF}
{$IFDEF RTL190_UP}
  Reg(CM_BUFFEREDPRINTCLIENT, 'CM_BUFFEREDPRINTCLIENT');
  Reg(CM_UNTHEMECONTROL, 'CM_UNTHEMECONTROL');
{$ENDIF}
{$IFDEF RTL200_UP}
  Reg(CM_DOUBLEBUFFEREDCHANGED, 'CM_DOUBLEBUFFEREDCHANGED');
  Reg(CM_PARENTDOUBLEBUFFEREDCHANGED, 'CM_PARENTDOUBLEBUFFEREDCHANGED');
{$ENDIF}
{$IFDEF RTL230_UP}
  Reg(CM_STYLECHANGED, 'CM_STYLECHANGED');
{$ELSE}
{$IFDEF RTL200_UP}
  Reg(CM_THEMECHANGED, 'CM_THEMECHANGED');
{$ENDIF}
{$ENDIF}
{$IFDEF RTL210_UP}
  Reg(CM_GESTURE, 'CM_GESTURE');
  Reg(CM_CUSTOMGESTURESCHANGED, 'CM_CUSTOMGESTURESCHANGED');
  Reg(CM_GESTUREMANAGERCHANGED, 'CM_GESTUREMANAGERCHANGED');
  Reg(CM_STANDARDGESTURESCHANGED, 'CM_STANDARDGESTURESCHANGED');
  Reg(CM_INPUTLANGCHANGE, 'CM_INPUTLANGCHANGE');
  Reg(CM_TABLETOPTIONSCHANGED, 'CM_TABLETOPTIONSCHANGED');
  Reg(CM_PARENTTABLETOPTIONSCHANGED, 'CM_PARENTTABLETOPTIONSCHANGED');
{$ENDIF}
  Reg(CN_BASE, 'CN_BASE');
  Reg(CN_CHARTOITEM, 'CN_CHARTOITEM');
  Reg(CN_COMMAND, 'CN_COMMAND');
  Reg(CN_COMPAREITEM, 'CN_COMPAREITEM');
  Reg(CN_CTLCOLORBTN, 'CN_CTLCOLORBTN');
  Reg(CN_CTLCOLORDLG, 'CN_CTLCOLORDLG');
  Reg(CN_CTLCOLOREDIT, 'CN_CTLCOLOREDIT');
  Reg(CN_CTLCOLORLISTBOX, 'CN_CTLCOLORLISTBOX');
  Reg(CN_CTLCOLORMSGBOX, 'CN_CTLCOLORMSGBOX');
  Reg(CN_CTLCOLORSCROLLBAR, 'CN_CTLCOLORSCROLLBAR');
  Reg(CN_CTLCOLORSTATIC, 'CN_CTLCOLORSTATIC');
  Reg(CN_DELETEITEM, 'CN_DELETEITEM');
  Reg(CN_DRAWITEM, 'CN_DRAWITEM');
  Reg(CN_HSCROLL, 'CN_HSCROLL');
  Reg(CN_MEASUREITEM, 'CN_MEASUREITEM');
  Reg(CN_PARENTNOTIFY, 'CN_PARENTNOTIFY');
  Reg(CN_VKEYTOITEM, 'CN_VKEYTOITEM');
  Reg(CN_VSCROLL, 'CN_VSCROLL');
  Reg(CN_KEYDOWN, 'CN_KEYDOWN');
  Reg(CN_KEYUP, 'CN_KEYUP');
  Reg(CN_CHAR, 'CN_CHAR');
  Reg(CN_SYSKEYDOWN, 'CN_SYSKEYDOWN');
  Reg(CN_SYSCHAR, 'CN_SYSCHAR');
  Reg(CN_NOTIFY, 'CN_NOTIFY');

  // from http://wiki.winehq.org/List_Of_Windows_Messages
  Reg(0, 'WM_NULL');
  Reg(1, 'WM_CREATE');
  Reg(2, 'WM_DESTROY');
  Reg(3, 'WM_MOVE');
  Reg(5, 'WM_SIZE');
  Reg(6, 'WM_ACTIVATE');
  Reg(7, 'WM_SETFOCUS');
  Reg(8, 'WM_KILLFOCUS');
  Reg(10, 'WM_ENABLE');
  Reg(11, 'WM_SETREDRAW');
  Reg(12, 'WM_SETTEXT');
  Reg(13, 'WM_GETTEXT');
  Reg(14, 'WM_GETTEXTLENGTH');
  Reg(15, 'WM_PAINT');
  Reg(16, 'WM_CLOSE');
  Reg(17, 'WM_QUERYENDSESSION');
  Reg(18, 'WM_QUIT');
  Reg(19, 'WM_QUERYOPEN');
  Reg(20, 'WM_ERASEBKGND');
  Reg(21, 'WM_SYSCOLORCHANGE');
  Reg(22, 'WM_ENDSESSION');
  Reg(24, 'WM_SHOWWINDOW');
  Reg(25, 'WM_CTLCOLOR');
  Reg(26, 'WM_WININICHANGE');
  Reg(27, 'WM_DEVMODECHANGE');
  Reg(28, 'WM_ACTIVATEAPP');
  Reg(29, 'WM_FONTCHANGE');
  Reg(30, 'WM_TIMECHANGE');
  Reg(31, 'WM_CANCELMODE');
  Reg(32, 'WM_SETCURSOR');
  Reg(33, 'WM_MOUSEACTIVATE');
  Reg(34, 'WM_CHILDACTIVATE');
  Reg(35, 'WM_QUEUESYNC');
  Reg(36, 'WM_GETMINMAXINFO');
  Reg(38, 'WM_PAINTICON');
  Reg(39, 'WM_ICONERASEBKGND');
  Reg(40, 'WM_NEXTDLGCTL');
  Reg(42, 'WM_SPOOLERSTATUS');
  Reg(43, 'WM_DRAWITEM');
  Reg(44, 'WM_MEASUREITEM');
  Reg(45, 'WM_DELETEITEM');
  Reg(46, 'WM_VKEYTOITEM');
  Reg(47, 'WM_CHARTOITEM');
  Reg(48, 'WM_SETFONT');
  Reg(49, 'WM_GETFONT');
  Reg(50, 'WM_SETHOTKEY');
  Reg(51, 'WM_GETHOTKEY');
  Reg(55, 'WM_QUERYDRAGICON');
  Reg(57, 'WM_COMPAREITEM');
  Reg(61, 'WM_GETOBJECT');
  Reg(65, 'WM_COMPACTING');
  Reg(68, 'WM_COMMNOTIFY');
  Reg(70, 'WM_WINDOWPOSCHANGING');
  Reg(71, 'WM_WINDOWPOSCHANGED');
  Reg(72, 'WM_POWER');
  Reg(73, 'WM_COPYGLOBALDATA');
  Reg(74, 'WM_COPYDATA');
  Reg(75, 'WM_CANCELJOURNAL');
  Reg(78, 'WM_NOTIFY');
  Reg(80, 'WM_INPUTLANGCHANGEREQUEST');
  Reg(81, 'WM_INPUTLANGCHANGE');
  Reg(82, 'WM_TCARD');
  Reg(83, 'WM_HELP');
  Reg(84, 'WM_USERCHANGED');
  Reg(85, 'WM_NOTIFYFORMAT');
  Reg(123, 'WM_CONTEXTMENU');
  Reg(124, 'WM_STYLECHANGING');
  Reg(125, 'WM_STYLECHANGED');
  Reg(126, 'WM_DISPLAYCHANGE');
  Reg(127, 'WM_GETICON');
  Reg(128, 'WM_SETICON');
  Reg(129, 'WM_NCCREATE');
  Reg(130, 'WM_NCDESTROY');
  Reg(131, 'WM_NCCALCSIZE');
  Reg(132, 'WM_NCHITTEST');
  Reg(133, 'WM_NCPAINT');
  Reg(134, 'WM_NCACTIVATE');
  Reg(135, 'WM_GETDLGCODE');
  Reg(136, 'WM_SYNCPAINT');
  Reg(160, 'WM_NCMOUSEMOVE');
  Reg(161, 'WM_NCLBUTTONDOWN');
  Reg(162, 'WM_NCLBUTTONUP');
  Reg(163, 'WM_NCLBUTTONDBLCLK');
  Reg(164, 'WM_NCRBUTTONDOWN');
  Reg(165, 'WM_NCRBUTTONUP');
  Reg(166, 'WM_NCRBUTTONDBLCLK');
  Reg(167, 'WM_NCMBUTTONDOWN');
  Reg(168, 'WM_NCMBUTTONUP');
  Reg(169, 'WM_NCMBUTTONDBLCLK');
  Reg(171, 'WM_NCXBUTTONDOWN');
  Reg(172, 'WM_NCXBUTTONUP');
  Reg(173, 'WM_NCXBUTTONDBLCLK');
  Reg(224, 'SBM_SETPOS');
  Reg(225, 'SBM_GETPOS');
  Reg(226, 'SBM_SETRANGE');
  Reg(227, 'SBM_GETRANGE');
  Reg(228, 'SBM_ENABLE_ARROWS');
  Reg(230, 'SBM_SETRANGEREDRAW');
  Reg(233, 'SBM_SETSCROLLINFO');
  Reg(234, 'SBM_GETSCROLLINFO');
  Reg(235, 'SBM_GETSCROLLBARINFO');
  Reg(255, 'WM_INPUT');
  Reg(256, 'WM_KEYDOWN');
  Reg(256, 'WM_KEYFIRST');
  Reg(257, 'WM_KEYUP');
  Reg(258, 'WM_CHAR');
  Reg(259, 'WM_DEADCHAR');
  Reg(260, 'WM_SYSKEYDOWN');
  Reg(261, 'WM_SYSKEYUP');
  Reg(262, 'WM_SYSCHAR');
  Reg(263, 'WM_SYSDEADCHAR');
  Reg(264, 'WM_KEYLAST');
  Reg(265, 'WM_WNT_CONVERTREQUESTEX');
  Reg(266, 'WM_CONVERTREQUEST');
  Reg(267, 'WM_CONVERTRESULT');
  Reg(268, 'WM_INTERIM');
  Reg(269, 'WM_IME_STARTCOMPOSITION');
  Reg(270, 'WM_IME_ENDCOMPOSITION');
  Reg(271, 'WM_IME_COMPOSITION');
  Reg(271, 'WM_IME_KEYLAST');
  Reg(272, 'WM_INITDIALOG');
  Reg(273, 'WM_COMMAND');
  Reg(274, 'WM_SYSCOMMAND');
  Reg(275, 'WM_TIMER');
  Reg(276, 'WM_HSCROLL');
  Reg(277, 'WM_VSCROLL');
  Reg(278, 'WM_INITMENU');
  Reg(279, 'WM_INITMENUPOPUP');
  Reg(280, 'WM_SYSTIMER');
  Reg(287, 'WM_MENUSELECT');
  Reg(288, 'WM_MENUCHAR');
  Reg(289, 'WM_ENTERIDLE');
  Reg(290, 'WM_MENURBUTTONUP');
  Reg(291, 'WM_MENUDRAG');
  Reg(292, 'WM_MENUGETOBJECT');
  Reg(293, 'WM_UNINITMENUPOPUP');
  Reg(294, 'WM_MENUCOMMAND');
  Reg(295, 'WM_CHANGEUISTATE');
  Reg(296, 'WM_UPDATEUISTATE');
  Reg(297, 'WM_QUERYUISTATE');
  Reg(306, 'WM_CTLCOLORMSGBOX');
  Reg(307, 'WM_CTLCOLOREDIT');
  Reg(308, 'WM_CTLCOLORLISTBOX');
  Reg(309, 'WM_CTLCOLORBTN');
  Reg(310, 'WM_CTLCOLORDLG');
  Reg(311, 'WM_CTLCOLORSCROLLBAR');
  Reg(312, 'WM_CTLCOLORSTATIC');
  Reg(512, 'WM_MOUSEFIRST');
  Reg(512, 'WM_MOUSEMOVE');
  Reg(513, 'WM_LBUTTONDOWN');
  Reg(514, 'WM_LBUTTONUP');
  Reg(515, 'WM_LBUTTONDBLCLK');
  Reg(516, 'WM_RBUTTONDOWN');
  Reg(517, 'WM_RBUTTONUP');
  Reg(518, 'WM_RBUTTONDBLCLK');
  Reg(519, 'WM_MBUTTONDOWN');
  Reg(520, 'WM_MBUTTONUP');
  Reg(521, 'WM_MBUTTONDBLCLK');
  Reg(521, 'WM_MOUSELAST');
  Reg(522, 'WM_MOUSEWHEEL');
  Reg(523, 'WM_XBUTTONDOWN');
  Reg(524, 'WM_XBUTTONUP');
  Reg(525, 'WM_XBUTTONDBLCLK');
  Reg(528, 'WM_PARENTNOTIFY');
  Reg(529, 'WM_ENTERMENULOOP');
  Reg(530, 'WM_EXITMENULOOP');
  Reg(531, 'WM_NEXTMENU');
  Reg(532, 'WM_SIZING');
  Reg(533, 'WM_CAPTURECHANGED');
  Reg(534, 'WM_MOVING');
  Reg(536, 'WM_POWERBROADCAST');
  Reg(537, 'WM_DEVICECHANGE');
  Reg(544, 'WM_MDICREATE');
  Reg(545, 'WM_MDIDESTROY');
  Reg(546, 'WM_MDIACTIVATE');
  Reg(547, 'WM_MDIRESTORE');
  Reg(548, 'WM_MDINEXT');
  Reg(549, 'WM_MDIMAXIMIZE');
  Reg(550, 'WM_MDITILE');
  Reg(551, 'WM_MDICASCADE');
  Reg(552, 'WM_MDIICONARRANGE');
  Reg(553, 'WM_MDIGETACTIVE');
  Reg(560, 'WM_MDISETMENU');
  Reg(561, 'WM_ENTERSIZEMOVE');
  Reg(562, 'WM_EXITSIZEMOVE');
  Reg(563, 'WM_DROPFILES');
  Reg(564, 'WM_MDIREFRESHMENU');
  Reg(640, 'WM_IME_REPORT');
  Reg(641, 'WM_IME_SETCONTEXT');
  Reg(642, 'WM_IME_NOTIFY');
  Reg(643, 'WM_IME_CONTROL');
  Reg(644, 'WM_IME_COMPOSITIONFULL');
  Reg(645, 'WM_IME_SELECT');
  Reg(646, 'WM_IME_CHAR');
  Reg(648, 'WM_IME_REQUEST');
  Reg(656, 'WM_IMEKEYDOWN');
  Reg(656, 'WM_IME_KEYDOWN');
  Reg(657, 'WM_IMEKEYUP');
  Reg(657, 'WM_IME_KEYUP');
  Reg(672, 'WM_NCMOUSEHOVER');
  Reg(673, 'WM_MOUSEHOVER');
  Reg(674, 'WM_NCMOUSELEAVE');
  Reg(675, 'WM_MOUSELEAVE');
  Reg(768, 'WM_CUT');
  Reg(769, 'WM_COPY');
  Reg(770, 'WM_PASTE');
  Reg(771, 'WM_CLEAR');
  Reg(772, 'WM_UNDO');
  Reg(773, 'WM_RENDERFORMAT');
  Reg(774, 'WM_RENDERALLFORMATS');
  Reg(775, 'WM_DESTROYCLIPBOARD');
  Reg(776, 'WM_DRAWCLIPBOARD');
  Reg(777, 'WM_PAINTCLIPBOARD');
  Reg(778, 'WM_VSCROLLCLIPBOARD');
  Reg(779, 'WM_SIZECLIPBOARD');
  Reg(780, 'WM_ASKCBFORMATNAME');
  Reg(781, 'WM_CHANGECBCHAIN');
  Reg(782, 'WM_HSCROLLCLIPBOARD');
  Reg(783, 'WM_QUERYNEWPALETTE');
  Reg(784, 'WM_PALETTEISCHANGING');
  Reg(785, 'WM_PALETTECHANGED');
  Reg(786, 'WM_HOTKEY');
  Reg(791, 'WM_PRINT');
  Reg(792, 'WM_PRINTCLIENT');
  Reg(793, 'WM_APPCOMMAND');
  Reg(856, 'WM_HANDHELDFIRST');
  Reg(863, 'WM_HANDHELDLAST');
  Reg(864, 'WM_AFXFIRST');
  Reg(895, 'WM_AFXLAST');
  Reg(896, 'WM_PENWINFIRST');
  Reg(897, 'WM_RCRESULT');
  Reg(898, 'WM_HOOKRCRESULT');
  Reg(899, 'WM_GLOBALRCCHANGE');
  Reg(899, 'WM_PENMISCINFO');
  Reg(900, 'WM_SKB');
  Reg(901, 'WM_HEDITCTL');
  Reg(901, 'WM_PENCTL');
  Reg(902, 'WM_PENMISC');
  Reg(903, 'WM_CTLINIT');
  Reg(904, 'WM_PENEVENT');
  Reg(911, 'WM_PENWINLAST');
  Reg(1024, 'DDM_SETFMT');
  Reg(1024, 'DM_GETDEFID');
  Reg(1024, 'NIN_SELECT');
  Reg(1024, 'TBM_GETPOS');
  Reg(1024, 'WM_PSD_PAGESETUPDLG');
  Reg(1024, 'WM_USER');
  Reg(1025, 'CBEM_INSERTITEMA');
  Reg(1025, 'DDM_DRAW');
  Reg(1025, 'DM_SETDEFID');
  Reg(1025, 'HKM_SETHOTKEY');
  Reg(1025, 'PBM_SETRANGE');
  Reg(1025, 'RB_INSERTBANDA');
  Reg(1025, 'SB_SETTEXTA');
  Reg(1025, 'TB_ENABLEBUTTON');
  Reg(1025, 'TBM_GETRANGEMIN');
  Reg(1025, 'TTM_ACTIVATE');
  Reg(1025, 'WM_CHOOSEFONT_GETLOGFONT');
  Reg(1025, 'WM_PSD_FULLPAGERECT');
  Reg(1026, 'CBEM_SETIMAGELIST');
  Reg(1026, 'DDM_CLOSE');
  Reg(1026, 'DM_REPOSITION');
  Reg(1026, 'HKM_GETHOTKEY');
  Reg(1026, 'PBM_SETPOS');
  Reg(1026, 'RB_DELETEBAND');
  Reg(1026, 'SB_GETTEXTA');
  Reg(1026, 'TB_CHECKBUTTON');
  Reg(1026, 'TBM_GETRANGEMAX');
  Reg(1026, 'WM_PSD_MINMARGINRECT');
  Reg(1027, 'CBEM_GETIMAGELIST');
  Reg(1027, 'DDM_BEGIN');
  Reg(1027, 'HKM_SETRULES');
  Reg(1027, 'PBM_DELTAPOS');
  Reg(1027, 'RB_GETBARINFO');
  Reg(1027, 'SB_GETTEXTLENGTHA');
  Reg(1027, 'TBM_GETTIC');
  Reg(1027, 'TB_PRESSBUTTON');
  Reg(1027, 'TTM_SETDELAYTIME');
  Reg(1027, 'WM_PSD_MARGINRECT');
  Reg(1028, 'CBEM_GETITEMA');
  Reg(1028, 'DDM_END');
  Reg(1028, 'PBM_SETSTEP');
  Reg(1028, 'RB_SETBARINFO');
  Reg(1028, 'SB_SETPARTS');
  Reg(1028, 'TB_HIDEBUTTON');
  Reg(1028, 'TBM_SETTIC');
  Reg(1028, 'TTM_ADDTOOLA');
  Reg(1028, 'WM_PSD_GREEKTEXTRECT');
  Reg(1029, 'CBEM_SETITEMA');
  Reg(1029, 'PBM_STEPIT');
  Reg(1029, 'TB_INDETERMINATE');
  Reg(1029, 'TBM_SETPOS');
  Reg(1029, 'TTM_DELTOOLA');
  Reg(1029, 'WM_PSD_ENVSTAMPRECT');
  Reg(1030, 'CBEM_GETCOMBOCONTROL');
  Reg(1030, 'PBM_SETRANGE32');
  Reg(1030, 'RB_SETBANDINFOA');
  Reg(1030, 'SB_GETPARTS');
  Reg(1030, 'TB_MARKBUTTON');
  Reg(1030, 'TBM_SETRANGE');
  Reg(1030, 'TTM_NEWTOOLRECTA');
  Reg(1030, 'WM_PSD_YAFULLPAGERECT');
  Reg(1031, 'CBEM_GETEDITCONTROL');
  Reg(1031, 'PBM_GETRANGE');
  Reg(1031, 'RB_SETPARENT');
  Reg(1031, 'SB_GETBORDERS');
  Reg(1031, 'TBM_SETRANGEMIN');
  Reg(1031, 'TTM_RELAYEVENT');
  Reg(1032, 'CBEM_SETEXSTYLE');
  Reg(1032, 'PBM_GETPOS');
  Reg(1032, 'RB_HITTEST');
  Reg(1032, 'SB_SETMINHEIGHT');
  Reg(1032, 'TBM_SETRANGEMAX');
  Reg(1032, 'TTM_GETTOOLINFOA');
  Reg(1033, 'CBEM_GETEXSTYLE');
  Reg(1033, 'CBEM_GETEXTENDEDSTYLE');
  Reg(1033, 'PBM_SETBARCOLOR');
  Reg(1033, 'RB_GETRECT');
  Reg(1033, 'SB_SIMPLE');
  Reg(1033, 'TB_ISBUTTONENABLED');
  Reg(1033, 'TBM_CLEARTICS');
  Reg(1033, 'TTM_SETTOOLINFOA');
  Reg(1034, 'CBEM_HASEDITCHANGED');
  Reg(1034, 'RB_INSERTBANDW');
  Reg(1034, 'SB_GETRECT');
  Reg(1034, 'TB_ISBUTTONCHECKED');
  Reg(1034, 'TBM_SETSEL');
  Reg(1034, 'TTM_HITTESTA');
  Reg(1034, 'WIZ_QUERYNUMPAGES');
  Reg(1035, 'CBEM_INSERTITEMW');
  Reg(1035, 'RB_SETBANDINFOW');
  Reg(1035, 'SB_SETTEXTW');
  Reg(1035, 'TB_ISBUTTONPRESSED');
  Reg(1035, 'TBM_SETSELSTART');
  Reg(1035, 'TTM_GETTEXTA');
  Reg(1035, 'WIZ_NEXT');
  Reg(1036, 'CBEM_SETITEMW');
  Reg(1036, 'RB_GETBANDCOUNT');
  Reg(1036, 'SB_GETTEXTLENGTHW');
  Reg(1036, 'TB_ISBUTTONHIDDEN');
  Reg(1036, 'TBM_SETSELEND');
  Reg(1036, 'TTM_UPDATETIPTEXTA');
  Reg(1036, 'WIZ_PREV');
  Reg(1037, 'CBEM_GETITEMW');
  Reg(1037, 'RB_GETROWCOUNT');
  Reg(1037, 'SB_GETTEXTW');
  Reg(1037, 'TB_ISBUTTONINDETERMINATE');
  Reg(1037, 'TTM_GETTOOLCOUNT');
  Reg(1038, 'CBEM_SETEXTENDEDSTYLE');
  Reg(1038, 'RB_GETROWHEIGHT');
  Reg(1038, 'SB_ISSIMPLE');
  Reg(1038, 'TB_ISBUTTONHIGHLIGHTED');
  Reg(1038, 'TBM_GETPTICS');
  Reg(1038, 'TTM_ENUMTOOLSA');
  Reg(1039, 'SB_SETICON');
  Reg(1039, 'TBM_GETTICPOS');
  Reg(1039, 'TTM_GETCURRENTTOOLA');
  Reg(1040, 'RB_IDTOINDEX');
  Reg(1040, 'SB_SETTIPTEXTA');
  Reg(1040, 'TBM_GETNUMTICS');
  Reg(1040, 'TTM_WINDOWFROMPOINT');
  Reg(1041, 'RB_GETTOOLTIPS');
  Reg(1041, 'SB_SETTIPTEXTW');
  Reg(1041, 'TBM_GETSELSTART');
  Reg(1041, 'TB_SETSTATE');
  Reg(1041, 'TTM_TRACKACTIVATE');
  Reg(1042, 'RB_SETTOOLTIPS');
  Reg(1042, 'SB_GETTIPTEXTA');
  Reg(1042, 'TB_GETSTATE');
  Reg(1042, 'TBM_GETSELEND');
  Reg(1042, 'TTM_TRACKPOSITION');
  Reg(1043, 'RB_SETBKCOLOR');
  Reg(1043, 'SB_GETTIPTEXTW');
  Reg(1043, 'TB_ADDBITMAP');
  Reg(1043, 'TBM_CLEARSEL');
  Reg(1043, 'TTM_SETTIPBKCOLOR');
  Reg(1044, 'RB_GETBKCOLOR');
  Reg(1044, 'SB_GETICON');
  Reg(1044, 'TB_ADDBUTTONSA');
  Reg(1044, 'TBM_SETTICFREQ');
  Reg(1044, 'TTM_SETTIPTEXTCOLOR');
  Reg(1045, 'RB_SETTEXTCOLOR');
  Reg(1045, 'TB_INSERTBUTTONA');
  Reg(1045, 'TBM_SETPAGESIZE');
  Reg(1045, 'TTM_GETDELAYTIME');
  Reg(1046, 'RB_GETTEXTCOLOR');
  Reg(1046, 'TB_DELETEBUTTON');
  Reg(1046, 'TBM_GETPAGESIZE');
  Reg(1046, 'TTM_GETTIPBKCOLOR');
  Reg(1047, 'RB_SIZETORECT');
  Reg(1047, 'TB_GETBUTTON');
  Reg(1047, 'TBM_SETLINESIZE');
  Reg(1047, 'TTM_GETTIPTEXTCOLOR');
  Reg(1048, 'RB_BEGINDRAG');
  Reg(1048, 'TB_BUTTONCOUNT');
  Reg(1048, 'TBM_GETLINESIZE');
  Reg(1048, 'TTM_SETMAXTIPWIDTH');
  Reg(1049, 'RB_ENDDRAG');
  Reg(1049, 'TB_COMMANDTOINDEX');
  Reg(1049, 'TBM_GETTHUMBRECT');
  Reg(1049, 'TTM_GETMAXTIPWIDTH');
  Reg(1050, 'RB_DRAGMOVE');
  Reg(1050, 'TBM_GETCHANNELRECT');
  Reg(1050, 'TB_SAVERESTOREA');
  Reg(1050, 'TTM_SETMARGIN');
  Reg(1051, 'RB_GETBARHEIGHT');
  Reg(1051, 'TB_CUSTOMIZE');
  Reg(1051, 'TBM_SETTHUMBLENGTH');
  Reg(1051, 'TTM_GETMARGIN');
  Reg(1052, 'RB_GETBANDINFOW');
  Reg(1052, 'TB_ADDSTRINGA');
  Reg(1052, 'TBM_GETTHUMBLENGTH');
  Reg(1052, 'TTM_POP');
  Reg(1053, 'RB_GETBANDINFOA');
  Reg(1053, 'TB_GETITEMRECT');
  Reg(1053, 'TBM_SETTOOLTIPS');
  Reg(1053, 'TTM_UPDATE');
  Reg(1054, 'RB_MINIMIZEBAND');
  Reg(1054, 'TB_BUTTONSTRUCTSIZE');
  Reg(1054, 'TBM_GETTOOLTIPS');
  Reg(1054, 'TTM_GETBUBBLESIZE');
  Reg(1055, 'RB_MAXIMIZEBAND');
  Reg(1055, 'TBM_SETTIPSIDE');
  Reg(1055, 'TB_SETBUTTONSIZE');
  Reg(1055, 'TTM_ADJUSTRECT');
  Reg(1056, 'TBM_SETBUDDY');
  Reg(1056, 'TB_SETBITMAPSIZE');
  Reg(1056, 'TTM_SETTITLEA');
  Reg(1057, 'MSG_FTS_JUMP_VA');
  Reg(1057, 'TB_AUTOSIZE');
  Reg(1057, 'TBM_GETBUDDY');
  Reg(1057, 'TTM_SETTITLEW');
  Reg(1058, 'RB_GETBANDBORDERS');
  Reg(1059, 'MSG_FTS_JUMP_QWORD');
  Reg(1059, 'RB_SHOWBAND');
  Reg(1059, 'TB_GETTOOLTIPS');
  Reg(1060, 'MSG_REINDEX_REQUEST');
  Reg(1060, 'TB_SETTOOLTIPS');
  Reg(1061, 'MSG_FTS_WHERE_IS_IT');
  Reg(1061, 'RB_SETPALETTE');
  Reg(1061, 'TB_SETPARENT');
  Reg(1062, 'RB_GETPALETTE');
  Reg(1063, 'RB_MOVEBAND');
  Reg(1063, 'TB_SETROWS');
  Reg(1064, 'TB_GETROWS');
  Reg(1065, 'TB_GETBITMAPFLAGS');
  Reg(1066, 'TB_SETCMDID');
  Reg(1067, 'RB_PUSHCHEVRON');
  Reg(1067, 'TB_CHANGEBITMAP');
  Reg(1068, 'TB_GETBITMAP');
  Reg(1069, 'MSG_GET_DEFFONT');
  Reg(1069, 'TB_GETBUTTONTEXTA');
  Reg(1070, 'TB_REPLACEBITMAP');
  Reg(1071, 'TB_SETINDENT');
  Reg(1072, 'TB_SETIMAGELIST');
  Reg(1073, 'TB_GETIMAGELIST');
  Reg(1074, 'TB_LOADIMAGES');
  Reg(1074, 'TTM_ADDTOOLW');
  Reg(1075, 'TB_GETRECT');
  Reg(1075, 'TTM_DELTOOLW');
  Reg(1076, 'TB_SETHOTIMAGELIST');
  Reg(1076, 'TTM_NEWTOOLRECTW');
  Reg(1077, 'TB_GETHOTIMAGELIST');
  Reg(1077, 'TTM_GETTOOLINFOW');
  Reg(1078, 'TB_SETDISABLEDIMAGELIST');
  Reg(1078, 'TTM_SETTOOLINFOW');
  Reg(1079, 'TB_GETDISABLEDIMAGELIST');
  Reg(1079, 'TTM_HITTESTW');
  Reg(1080, 'TB_SETSTYLE');
  Reg(1080, 'TTM_GETTEXTW');
  Reg(1081, 'TB_GETSTYLE');
  Reg(1081, 'TTM_UPDATETIPTEXTW');
  Reg(1082, 'TB_GETBUTTONSIZE');
  Reg(1082, 'TTM_ENUMTOOLSW');
  Reg(1083, 'TB_SETBUTTONWIDTH');
  Reg(1083, 'TTM_GETCURRENTTOOLW');
  Reg(1084, 'TB_SETMAXTEXTROWS');
  Reg(1085, 'TB_GETTEXTROWS');
  Reg(1086, 'TB_GETOBJECT');
  Reg(1087, 'TB_GETBUTTONINFOW');
  Reg(1088, 'TB_SETBUTTONINFOW');
  Reg(1089, 'TB_GETBUTTONINFOA');
  Reg(1090, 'TB_SETBUTTONINFOA');
  Reg(1091, 'TB_INSERTBUTTONW');
  Reg(1092, 'TB_ADDBUTTONSW');
  Reg(1093, 'TB_HITTEST');
  Reg(1094, 'TB_SETDRAWTEXTFLAGS');
  Reg(1095, 'TB_GETHOTITEM');
  Reg(1096, 'TB_SETHOTITEM');
  Reg(1097, 'TB_SETANCHORHIGHLIGHT');
  Reg(1098, 'TB_GETANCHORHIGHLIGHT');
  Reg(1099, 'TB_GETBUTTONTEXTW');
  Reg(1100, 'TB_SAVERESTOREW');
  Reg(1101, 'TB_ADDSTRINGW');
  Reg(1102, 'TB_MAPACCELERATORA');
  Reg(1103, 'TB_GETINSERTMARK');
  Reg(1104, 'TB_SETINSERTMARK');
  Reg(1105, 'TB_INSERTMARKHITTEST');
  Reg(1106, 'TB_MOVEBUTTON');
  Reg(1107, 'TB_GETMAXSIZE');
  Reg(1108, 'TB_SETEXTENDEDSTYLE');
  Reg(1109, 'TB_GETEXTENDEDSTYLE');
  Reg(1110, 'TB_GETPADDING');
  Reg(1111, 'TB_SETPADDING');
  Reg(1112, 'TB_SETINSERTMARKCOLOR');
  Reg(1113, 'TB_GETINSERTMARKCOLOR');
  Reg(1114, 'TB_MAPACCELERATORW');
  Reg(1115, 'TB_GETSTRINGW');
  Reg(1116, 'TB_GETSTRINGA');
  Reg(1123, 'TAPI_REPLY');
  Reg(1124, 'ACM_OPENA');
  Reg(1124, 'BFFM_SETSTATUSTEXTA');
  Reg(1124, 'CDM_FIRST');
  Reg(1124, 'CDM_GETSPEC');
  Reg(1124, 'IPM_CLEARADDRESS');
  Reg(1124, 'WM_CAP_UNICODE_START');
  Reg(1125, 'ACM_PLAY');
  Reg(1125, 'BFFM_ENABLEOK');
  Reg(1125, 'CDM_GETFILEPATH');
  Reg(1125, 'IPM_SETADDRESS');
  Reg(1125, 'PSM_SETCURSEL');
  Reg(1125, 'UDM_SETRANGE');
  Reg(1125, 'WM_CHOOSEFONT_SETLOGFONT');
  Reg(1126, 'ACM_STOP');
  Reg(1126, 'BFFM_SETSELECTIONA');
  Reg(1126, 'CDM_GETFOLDERPATH');
  Reg(1126, 'IPM_GETADDRESS');
  Reg(1126, 'PSM_REMOVEPAGE');
  Reg(1126, 'UDM_GETRANGE');
  Reg(1126, 'WM_CAP_SET_CALLBACK_ERRORW');
  Reg(1126, 'WM_CHOOSEFONT_SETFLAGS');
  Reg(1127, 'ACM_OPENW');
  Reg(1127, 'BFFM_SETSELECTIONW');
  Reg(1127, 'CDM_GETFOLDERIDLIST');
  Reg(1127, 'IPM_SETRANGE');
  Reg(1127, 'PSM_ADDPAGE');
  Reg(1127, 'UDM_SETPOS');
  Reg(1127, 'WM_CAP_SET_CALLBACK_STATUSW');
  Reg(1128, 'BFFM_SETSTATUSTEXTW');
  Reg(1128, 'CDM_SETCONTROLTEXT');
  Reg(1128, 'IPM_SETFOCUS');
  Reg(1128, 'PSM_CHANGED');
  Reg(1128, 'UDM_GETPOS');
  Reg(1129, 'CDM_HIDECONTROL');
  Reg(1129, 'IPM_ISBLANK');
  Reg(1129, 'PSM_RESTARTWINDOWS');
  Reg(1129, 'UDM_SETBUDDY');
  Reg(1130, 'CDM_SETDEFEXT');
  Reg(1130, 'PSM_REBOOTSYSTEM');
  Reg(1130, 'UDM_GETBUDDY');
  Reg(1131, 'PSM_CANCELTOCLOSE');
  Reg(1131, 'UDM_SETACCEL');
  Reg(1132, 'EM_CONVPOSITION');
  Reg(1132, 'PSM_QUERYSIBLINGS');
  Reg(1132, 'UDM_GETACCEL');
  Reg(1133, 'MCIWNDM_GETZOOM');
  Reg(1133, 'PSM_UNCHANGED');
  Reg(1133, 'UDM_SETBASE');
  Reg(1134, 'PSM_APPLY');
  Reg(1134, 'UDM_GETBASE');
  Reg(1135, 'PSM_SETTITLEA');
  Reg(1135, 'UDM_SETRANGE32');
  Reg(1136, 'PSM_SETWIZBUTTONS');
  Reg(1136, 'UDM_GETRANGE32');
  Reg(1136, 'WM_CAP_DRIVER_GET_NAMEW');
  Reg(1137, 'PSM_PRESSBUTTON');
  Reg(1137, 'UDM_SETPOS32');
  Reg(1137, 'WM_CAP_DRIVER_GET_VERSIONW');
  Reg(1138, 'PSM_SETCURSELID');
  Reg(1138, 'UDM_GETPOS32');
  Reg(1139, 'PSM_SETFINISHTEXTA');
  Reg(1140, 'PSM_GETTABCONTROL');
  Reg(1141, 'PSM_ISDIALOGMESSAGE');
  Reg(1142, 'MCIWNDM_REALIZE');
  Reg(1142, 'PSM_GETCURRENTPAGEHWND');
  Reg(1143, 'MCIWNDM_SETTIMEFORMATA');
  Reg(1143, 'PSM_INSERTPAGE');
  Reg(1144, 'MCIWNDM_GETTIMEFORMATA');
  Reg(1144, 'PSM_SETTITLEW');
  Reg(1144, 'WM_CAP_FILE_SET_CAPTURE_FILEW');
  Reg(1145, 'MCIWNDM_VALIDATEMEDIA');
  Reg(1145, 'PSM_SETFINISHTEXTW');
  Reg(1145, 'WM_CAP_FILE_GET_CAPTURE_FILEW');
  Reg(1147, 'MCIWNDM_PLAYTO');
  Reg(1147, 'WM_CAP_FILE_SAVEASW');
  Reg(1148, 'MCIWNDM_GETFILENAMEA');
  Reg(1149, 'MCIWNDM_GETDEVICEA');
  Reg(1149, 'PSM_SETHEADERTITLEA');
  Reg(1149, 'WM_CAP_FILE_SAVEDIBW');
  Reg(1150, 'MCIWNDM_GETPALETTE');
  Reg(1150, 'PSM_SETHEADERTITLEW');
  Reg(1151, 'MCIWNDM_SETPALETTE');
  Reg(1151, 'PSM_SETHEADERSUBTITLEA');
  Reg(1152, 'MCIWNDM_GETERRORA');
  Reg(1152, 'PSM_SETHEADERSUBTITLEW');
  Reg(1153, 'PSM_HWNDTOINDEX');
  Reg(1154, 'PSM_INDEXTOHWND');
  Reg(1155, 'MCIWNDM_SETINACTIVETIMER');
  Reg(1155, 'PSM_PAGETOINDEX');
  Reg(1156, 'PSM_INDEXTOPAGE');
  Reg(1157, 'DL_BEGINDRAG');
  Reg(1157, 'MCIWNDM_GETINACTIVETIMER');
  Reg(1157, 'PSM_IDTOINDEX');
  Reg(1158, 'DL_DRAGGING');
  Reg(1158, 'PSM_INDEXTOID');
  Reg(1159, 'DL_DROPPED');
  Reg(1159, 'PSM_GETRESULT');
  Reg(1160, 'DL_CANCELDRAG');
  Reg(1160, 'PSM_RECALCPAGESIZES');
  Reg(1164, 'MCIWNDM_GET_SOURCE');
  Reg(1165, 'MCIWNDM_PUT_SOURCE');
  Reg(1166, 'MCIWNDM_GET_DEST');
  Reg(1167, 'MCIWNDM_PUT_DEST');
  Reg(1168, 'MCIWNDM_CAN_PLAY');
  Reg(1169, 'MCIWNDM_CAN_WINDOW');
  Reg(1170, 'MCIWNDM_CAN_RECORD');
  Reg(1171, 'MCIWNDM_CAN_SAVE');
  Reg(1172, 'MCIWNDM_CAN_EJECT');
  Reg(1173, 'MCIWNDM_CAN_CONFIG');
  Reg(1174, 'IE_GETINK');
  Reg(1174, 'IE_MSGFIRST');
  Reg(1174, 'MCIWNDM_PALETTEKICK');
  Reg(1175, 'IE_SETINK');
  Reg(1176, 'IE_GETPENTIP');
  Reg(1177, 'IE_SETPENTIP');
  Reg(1178, 'IE_GETERASERTIP');
  Reg(1179, 'IE_SETERASERTIP');
  Reg(1180, 'IE_GETBKGND');
  Reg(1181, 'IE_SETBKGND');
  Reg(1182, 'IE_GETGRIDORIGIN');
  Reg(1183, 'IE_SETGRIDORIGIN');
  Reg(1184, 'IE_GETGRIDPEN');
  Reg(1185, 'IE_SETGRIDPEN');
  Reg(1186, 'IE_GETGRIDSIZE');
  Reg(1187, 'IE_SETGRIDSIZE');
  Reg(1188, 'IE_GETMODE');
  Reg(1189, 'IE_SETMODE');
  Reg(1190, 'IE_GETINKRECT');
  Reg(1190, 'WM_CAP_SET_MCI_DEVICEW');
  Reg(1191, 'WM_CAP_GET_MCI_DEVICEW');
  Reg(1204, 'WM_CAP_PAL_OPENW');
  Reg(1205, 'WM_CAP_PAL_SAVEW');
  Reg(1208, 'IE_GETAPPDATA');
  Reg(1209, 'IE_SETAPPDATA');
  Reg(1210, 'IE_GETDRAWOPTS');
  Reg(1211, 'IE_SETDRAWOPTS');
  Reg(1212, 'IE_GETFORMAT');
  Reg(1213, 'IE_SETFORMAT');
  Reg(1214, 'IE_GETINKINPUT');
  Reg(1215, 'IE_SETINKINPUT');
  Reg(1216, 'IE_GETNOTIFY');
  Reg(1217, 'IE_SETNOTIFY');
  Reg(1218, 'IE_GETRECOG');
  Reg(1219, 'IE_SETRECOG');
  Reg(1220, 'IE_GETSECURITY');
  Reg(1221, 'IE_SETSECURITY');
  Reg(1222, 'IE_GETSEL');
  Reg(1223, 'IE_SETSEL');
  Reg(1224, 'CDM_LAST');
  Reg(1224, 'IE_DOCOMMAND');
  Reg(1224, 'MCIWNDM_NOTIFYMODE');
  Reg(1225, 'IE_GETCOMMAND');
  Reg(1226, 'IE_GETCOUNT');
  Reg(1227, 'IE_GETGESTURE');
  Reg(1227, 'MCIWNDM_NOTIFYMEDIA');
  Reg(1228, 'IE_GETMENU');
  Reg(1229, 'IE_GETPAINTDC');
  Reg(1229, 'MCIWNDM_NOTIFYERROR');
  Reg(1230, 'IE_GETPDEVENT');
  Reg(1231, 'IE_GETSELCOUNT');
  Reg(1232, 'IE_GETSELITEMS');
  Reg(1233, 'IE_GETSTYLE');
  Reg(1243, 'MCIWNDM_SETTIMEFORMATW');
  Reg(1244, 'EM_OUTLINE');
  Reg(1244, 'MCIWNDM_GETTIMEFORMATW');
  Reg(1245, 'EM_GETSCROLLPOS');
  Reg(1246, 'EM_SETSCROLLPOS');
  Reg(1247, 'EM_SETFONTSIZE');
  Reg(1248, 'MCIWNDM_GETFILENAMEW');
  Reg(1249, 'MCIWNDM_GETDEVICEW');
  Reg(1252, 'MCIWNDM_GETERRORW');
  Reg(1536, 'FM_GETFOCUS');
  Reg(1537, 'FM_GETDRIVEINFOA');
  Reg(1538, 'FM_GETSELCOUNT');
  Reg(1539, 'FM_GETSELCOUNTLFN');
  Reg(1540, 'FM_GETFILESELA');
  Reg(1541, 'FM_GETFILESELLFNA');
  Reg(1542, 'FM_REFRESH_WINDOWS');
  Reg(1543, 'FM_RELOAD_EXTENSIONS');
  Reg(1553, 'FM_GETDRIVEINFOW');
  Reg(1556, 'FM_GETFILESELW');
  Reg(1557, 'FM_GETFILESELLFNW');
  Reg(1625, 'WLX_WM_SAS');
  Reg(2024, 'SM_GETSELCOUNT');
  Reg(2024, 'UM_GETSELCOUNT');
  Reg(2024, 'WM_CPL_LAUNCH');
  Reg(2025, 'SM_GETSERVERSELA');
  Reg(2025, 'UM_GETUSERSELA');
  Reg(2025, 'WM_CPL_LAUNCHED');
  Reg(2026, 'SM_GETSERVERSELW');
  Reg(2026, 'UM_GETUSERSELW');
  Reg(2027, 'SM_GETCURFOCUSA');
  Reg(2027, 'UM_GETGROUPSELA');
  Reg(2028, 'SM_GETCURFOCUSW');
  Reg(2028, 'UM_GETGROUPSELW');
  Reg(2029, 'SM_GETOPTIONS');
  Reg(2029, 'UM_GETCURFOCUSA');
  Reg(2030, 'UM_GETCURFOCUSW');
  Reg(2031, 'UM_GETOPTIONS');
  Reg(2032, 'UM_GETOPTIONS2');
  Reg(8192, 'OCMBASE');
  Reg(8217, 'OCM_CTLCOLOR');
  Reg(8235, 'OCM_DRAWITEM');
  Reg(8236, 'OCM_MEASUREITEM');
  Reg(8237, 'OCM_DELETEITEM');
  Reg(8238, 'OCM_VKEYTOITEM');
  Reg(8239, 'OCM_CHARTOITEM');
  Reg(8249, 'OCM_COMPAREITEM');
  Reg(8270, 'OCM_NOTIFY');
  Reg(8465, 'OCM_COMMAND');
  Reg(8468, 'OCM_HSCROLL');
  Reg(8469, 'OCM_VSCROLL');
  Reg(8498, 'OCM_CTLCOLORMSGBOX');
  Reg(8499, 'OCM_CTLCOLOREDIT');
  Reg(8500, 'OCM_CTLCOLORLISTBOX');
  Reg(8501, 'OCM_CTLCOLORBTN');
  Reg(8502, 'OCM_CTLCOLORDLG');
  Reg(8503, 'OCM_CTLCOLORSCROLLBAR');
  Reg(8504, 'OCM_CTLCOLORSTATIC');
  Reg(8720, 'OCM_PARENTNOTIFY');
  Reg(32768, 'WM_APP');
  Reg(52429, 'WM_RASDIALEVENT');

  // from http://stackoverflow.com/questions/23751216/where-to-find-definition-of-window-message-wm-uahdrawmenuitem-0x92
  Reg($0090, 'WM_UAHDESTROYWINDOW');
  Reg($0091, 'WM_UAHDRAWMENU');
  Reg($0092, 'WM_UAHDRAWMENUITEM');
  Reg($0093, 'WM_UAHINITMENU');
  Reg($0094, 'WM_UAHMEASUREMENUITEM');
  Reg($0095, 'WM_UAHNCPAINTMENUPOPUP');

  // from http://microsoft.public.win32.programmer.ui.narkive.com/EVDpEGV5/undocumented-mysterious-window-message-0xae
  Reg($00AE, 'WM_NCUAHDRAWCAPTION');
  Reg($00AF, 'WM_NCUAHDRAWFRAME');
end;

end.

