unit u_dzConnectionString;

interface

uses
  Windows,
  SysUtils,
  Classes;

type
  TAdoServerType = (asUnknown, asJet, asMsSql, asOracle);

type
  TStringArray = array of string;

type
  TConnectionInfoRec = record
  private
    FConnectionString: string;
    procedure SetPart(const _Name, _Value: string);
    function GetPart(const _Name: string): string;
    function GetInitialCatalog: string;
    function GetProvider: string;
    function GetDataSource: string;
    function GetUserId: string;
    procedure SetInitialCatalog(const _Value: string);
    procedure SetProvider(const _Value: string);
    procedure SetDataSource(const _Value: string);
    procedure SetUserId(const _Value: string);
    function GetConnectionString: string;
    procedure SetConnectionString(const _Value: string);
    function GetPassword: string;
    procedure SetPassword(const _Value: string);
  public
    class function DetermineServerType(const _ConnectionString: string): TAdoServerType; static;
    class function EditConnectionString(_ParentHandle: HWND; var _ConnectionString: string): Boolean; overload; static;
    function EditConnectionString(_ParentHandle: HWND): Boolean; overload;
    function GetParts: TStringArray; overload;
    procedure GetParts(_sl: TStrings); overload;
    function ServerType: TAdoServerType;
    property Provider: string read GetProvider write SetProvider;
    property ServerName: string read GetDataSource write SetDataSource;
    property DataSource: string read GetDataSource write SetDataSource;
    property Database: string read GetInitialCatalog write SetInitialCatalog; // same as InitialCatalog
    property InitialCatalog: string read GetInitialCatalog write SetInitialCatalog;
    property Username: string read GetUserId write SetUserId; // same as UserId
    property UserId: string read GetUserId write SetUserId;
    property Password: string read GetPassword write SetPassword;
    property ConnectionString: string read GetConnectionString write SetConnectionString;
  end;

implementation

uses
  StrUtils,
  OleDB,
  ComObj,
  ActiveX,
  u_dzNamedThread;

type
  TMoveWindowThread = class(TNamedThread)
  private
    FParentHandle: HWND;
    FParentCenterX: Integer;
    FParentCenterY: Integer;
    procedure CenterWindow(wHandle: hwnd);
  protected
    procedure Execute; override;
  public
    constructor Create(_ParentHandle: HWND);
  end;

{ TMoveWindowThread }

constructor TMoveWindowThread.Create(_ParentHandle: HWND);
begin
  FreeOnTerminate := True;
  FParentHandle := _ParentHandle;
  inherited Create(False);
end;

procedure TMoveWindowThread.CenterWindow(wHandle: hwnd);
var
  Rect: TRect;
  WindowCenterX: Integer;
  WindowCenterY: Integer;
  MoveByX: Integer;
  MoveByY: Integer;
begin
  GetWindowRect(wHandle, Rect);
  WindowCenterX := Round(Rect.Left / 2 + Rect.Right / 2);
  WindowCenterY := Round(Rect.Top / 2 + Rect.Bottom / 2);
  MoveByX := WindowCenterX - FParentCenterX;
  MoveByY := WindowCenterY - FParentCenterY;
  MoveWindow(wHandle, Rect.Left - MoveByX, Rect.Top - MoveByY,
    Rect.Right - Rect.Left, Rect.Bottom - Rect.Top, False);
end;

procedure TMoveWindowThread.Execute;
var
  Rect: TRect;
  MaxTickCount: DWORD;
  ThreadInfo: TGUIThreadinfo;
begin
  inherited;
  GetWindowRect(FParentHandle, Rect);
  FParentCenterX := Round(Rect.Left / 2 + Rect.Right / 2);
  FParentCenterY := Round(Rect.Top / 2 + Rect.Bottom / 2);

  ThreadInfo.cbSize := SizeOf(ThreadInfo);
  MaxTickCount := GetTickCount + 10000; // 10 Seconds should be plenty
  while MaxTickCount > GetTickCount do begin
    Sleep(50);
    if GetGUIThreadInfo(MainThreadID, ThreadInfo) then begin
      if ThreadInfo.hwndActive <> FParentHandle then begin
        CenterWindow(ThreadInfo.hwndActive);
        Exit;
      end;
    end;
  end;
end;

{ TConnectionInfoRec }

class function TConnectionInfoRec.EditConnectionString(_ParentHandle: HWND;
  var _ConnectionString: string): Boolean;
var
  DataInit: IDataInitialize;
  DBPrompt: IDBPromptInitialize;
  DataSource: IUnknown;
  InitStr: PWideChar;
  s: WideString;
begin
  DataInit := CreateComObject(CLSID_DataLinks) as IDataInitialize;
  if _ConnectionString <> '' then begin
    s := _ConnectionString;
    DataInit.GetDataSource(nil, CLSCTX_INPROC_SERVER,
      PWideChar(s), IUnknown, DataSource);
  end;
  DBPrompt := CreateComObject(CLSID_DataLinks) as IDBPromptInitialize;

  if _ParentHandle <> 0 then begin
    // This is a hack to make the dialog appear centered on the parent window
    // According to https://msdn.microsoft.com/en-us/library/ms725392(v=vs.85).aspx
    // the dialog should automatically be centered on the passed parent handle,
    // but if the parent window is not on the primary monitor this does not work.
    // So, we start a background thread that waits for the dialog to appear and then
    // moves it to the correct position.
    TMoveWindowThread.Create(_ParentHandle);
  end;
  Result := Succeeded(DBPrompt.PromptDataSource(nil, _ParentHandle,
    DBPROMPTOPTIONS_PROPERTYSHEET, 0, nil, nil, IUnknown, DataSource));
  if Result then begin
    InitStr := nil;
    DataInit.GetInitializationString(DataSource, True, InitStr);
    _ConnectionString := InitStr;
  end;
end;

function TConnectionInfoRec.EditConnectionString(_ParentHandle: HWND): Boolean;
var
  s: string;
begin
  s := FConnectionString;
  Result := EditConnectionString(_ParentHandle, s);
  if Result then
    FConnectionString := s;
end;

function TConnectionInfoRec.GetConnectionString: string;
begin
  Result := FConnectionString;
end;

procedure TConnectionInfoRec.SetConnectionString(const _Value: string);
begin
  FConnectionString := _Value;
end;

function TConnectionInfoRec.GetPart(const _Name: string): string;
var
  sl: TStringList;
begin
  sl := TStringList.Create;
  try
    GetParts(sl);
    Result := sl.Values[_Name];
  finally
    FreeAndNil(sl);
  end;
end;

procedure TConnectionInfoRec.SetPart(const _Name, _Value: string);
var
  sl: TStringList;
  i: Integer;
begin
  sl := TStringList.Create;
  try
    GetParts(sl);
    if _Value = '' then begin
      for i := sl.Count - 1 downto 0 do begin
        if SameText(sl.Names[i], _Name) then
          sl.Delete(i);
      end;
    end else
      sl.Values[_Name] := _Value;
    FConnectionString := sl.DelimitedText;
  finally
    FreeAndNil(sl);
  end;
end;

procedure TConnectionInfoRec.GetParts(_sl: TStrings);
begin
  _sl.StrictDelimiter := True;
  _sl.Delimiter := ';';
  _sl.DelimitedText := FConnectionString;
end;

function TConnectionInfoRec.GetParts: TStringArray;
var
  sl: TStringList;
  i: Integer;
begin
  sl := TStringList.Create;
  try
    GetParts(sl);
    SetLength(Result, sl.Count);
    for i := 0 to sl.Count - 1 do
      Result[i] := sl[i];
  finally
    FreeAndNil(sl);
  end;
end;

function TConnectionInfoRec.GetInitialCatalog: string;
begin
  Result := GetPart('Initial Catalog');
end;

procedure TConnectionInfoRec.SetInitialCatalog(const _Value: string);
begin
  SetPart('Initial Catalog', _Value);
end;

function TConnectionInfoRec.GetProvider: string;
begin
  Result := GetPart('Provider');
end;

procedure TConnectionInfoRec.SetProvider(const _Value: string);
begin
  SetPart('Provider', _Value);
end;

function TConnectionInfoRec.GetDataSource: string;
begin
  Result := GetPart('Data Source');
end;

procedure TConnectionInfoRec.SetDataSource(const _Value: string);
begin
  SetPart('Data Source', _Value);
end;

function TConnectionInfoRec.GetUserId: string;
begin
  Result := GetPart('User ID');
end;

procedure TConnectionInfoRec.SetUserId(const _Value: string);
begin
  SetPart('User ID', _Value);
end;

function TConnectionInfoRec.GetPassword: string;
begin
  Result := GetPart('Password');
end;

procedure TConnectionInfoRec.SetPassword(const _Value: string);
begin
  SetPart('Password', _Value);
end;

function TConnectionInfoRec.ServerType: TAdoServerType;
begin
  Result := DetermineServerType(FConnectionString);
end;

class function TConnectionInfoRec.DetermineServerType(const _ConnectionString: string): TAdoServerType;
begin
  if AnsiContainsText(_ConnectionString, 'Provider=OraOLEDB.Oracle') then
    Result := asOracle
  else if AnsiContainsText(_ConnectionString, 'Provider=SQLOLEDB')
    or AnsiContainsText(_ConnectionString, 'Provider=SQLNCLI10') then
    Result := asMsSql
  else if AnsiContainsText(_ConnectionString, 'Provider=Microsoft.Jet.OLEDB') then
    Result := asJet
  else
    Result := asUnknown;
end;

end.

