unit u_dzErrorThread;

{$INCLUDE 'jedi.inc'}

interface

uses
  SysUtils,
  Classes,
  u_dzNamedThread;

type
  ///<summary>
  /// A thread that handles exceptions and provides the error message as well as has a HasFinished
  /// property.
  /// Do not override Execute, override doExecute instead. </summary>
  TErrorThread = class(TNamedThread)
  private
    FExceptionClass: TClass;
    FErrorMessage: string;
    FHasFinished: Boolean;
  protected
    ///<summary>
    /// Calls inherited to set the thread name and then the doExecute method.
    /// Any Exceptions are caught and their  Message stored in ErrorMessage.
    /// After the doExecute method has finished the HasFinished property is set to true. </summary>
    procedure Execute; override;
    ///<summary>
    /// empty method that must be overriden by descendants to do anything </summary>
    procedure doExecute; virtual;
  public
    ///<summary>
    /// Is true, when the thread has finished executing </summary>
    property HasFinished: Boolean read FHasFinished;
    ///<summary>
    /// If an unhandled exception occurred in the callback, its message is stored in ErrorMessage.
    /// Only valid after the thread has finished executing (that is HasFinished = true) </summary>
    property ErrorMessage: string read FErrorMessage;
    ///<summary>
    /// Class of exception whose message was stored in ErrorMessage </summary>
    property ExceptionClass: TClass read FExceptionClass;
  end;

{$DEFINE __DZ_OBJECT_LIST_TEMPLATE__}
type
  _LIST_ANCESTOR_ = TObject;
  _ITEM_TYPE_ = TErrorThread;
{$INCLUDE 't_dzObjectListTemplate.tpl'}

type
  {: List for storing TErrorThread items }
  TErrorThreadList = class(_DZ_OBJECT_LIST_TEMPLATE_)
  public
    function StillRunning: Integer;
    procedure GetErrors(_sl: TStrings);
  end;

implementation

{$INCLUDE 't_dzObjectListTemplate.tpl'}

{ TErrorThread }

procedure TErrorThread.doExecute;
begin
  // does nothing
end;

procedure TErrorThread.Execute;
begin
  try
    try
      inherited;
      doExecute;
    except
      on e: Exception do begin
        FExceptionClass := e.ClassType;
        FErrorMessage := e.Message;
        UniqueString(FErrorMessage);
      end;
    end;
  finally
    FHasFinished := True;
  end;

end;

{ TErrorThreadList }

procedure TErrorThreadList.GetErrors(_sl: TStrings);
var
  i: Integer;
  s: string;
begin
  for i := 0 to Count - 1 do begin
    s := Items[i].ErrorMessage;
    if s <> '' then begin
      UniqueString(s);
      _sl.Add(s);
    end;
  end;
end;

function TErrorThreadList.StillRunning: Integer;
var
  i: Integer;
begin
  Result := 0;
  for i := 0 to Count - 1 do
    if not Items[i].HasFinished then
      Inc(Result);
end;

end.
