unit u_dzGraphicsUtils;

{.$DEFINE dzUseGraphics32}

interface

uses
  Windows,
  Types,
  SysUtils,
  Graphics,
{$IFDEF dzUseGraphics32}
  GR32, // libs\graphics32\src
{$ENDIF}
  u_dzTranslator;

type
  TRgbBrightnessChannelEnum = (rcbAverage, rcbFastLuminance, rcbRed, rcbGreen, rcbBlue, rcbLuminance);

const
  // Constant from GraphUtil (implementation section)
  HLSMAX = 240; // H,L, and S vary over 0-HLSMAX

type
{$IFDEF dzUseGraphics32}
  THlsValueType = Single; // 0..1
{$ELSE}
  THlsValueType = Word; // 0..HLSMAX (240)
{$ENDIF}
  THlsRec = record
    Hue: THlsValueType;
    Luminance: THlsValueType;
    Saturation: THlsValueType;
  end;

type
  TdzRgbTriple = packed record
    // do not change the order of the fields, do not add any fields
    Blue: Byte;
    Green: Byte;
    Red: Byte;
    function GetColor: TColor;
    procedure SetColor(_Color: TColor);
    procedure SetGray(_Value: Byte);
    function GetLuminance: Byte;
    function GetFastLuminance: Byte; overload;
    class function GetFastLuminance(_Red, _Green, _Blue: Byte): Byte; overload; static;
    function GetBrightness(_Channel: TRgbBrightnessChannelEnum): Byte;
    procedure SetBrightness(_Value: Byte);
    procedure GetHls(out _Hls: THlsRec);
    procedure SetHls(const _Hls: THlsRec);
  end;

type
  TdzRgbTripleArray = packed array[0..MaxInt div SizeOf(TdzRgbTriple) - 1] of TdzRgbTriple;
  PdzRgbTripleArray = ^TdzRgbTripleArray;

type
  TdzRgbQuad = packed record
    // do not change the order of the fields, do not add any fields
    Blue: Byte;
    Green: Byte;
    Red: Byte;
    Reserved: Byte;
    function GetColor: TColor;
    procedure SetColor(_Color: TColor);
    procedure SetGray(_Value: Byte);
    function GetLuminance: Word;
    function GetFastLuminance: Word;
    function GetBrightness(_Channel: TRgbBrightnessChannelEnum): Word;
    procedure SetBrightness(_Value: Byte);
    procedure GetHls(out _Hue, _Luminance, _Saturation: Word);
    procedure SetHls(_Hue, _Luminance, _Saturation: Word);
  end;

type
  TdzRgbQuadArray = packed array[0..MaxInt div SizeOf(TdzRgbQuad) - 1] of TdzRgbQuad;
  PdzRgbQuadArray = ^TdzRgbQuadArray;

///<summary> Returns the Rect's width </summary>
function TRect_Width(_Rect: TRect): Integer; inline;

///<summary> Returns the Rect's height </summary>
function TRect_Height(_Rect: TRect): Integer; inline;

///<summary>
/// @returns a TRect generated from Left, Top, Width and Height </summary>
function TRect_FromLTWH(_l, _t, _w, _h: Integer): TRect;

///<summary> returns the center point of the Rect </summary>
function TRect_Center(_Rect: TRect): TPoint; inline;

///<summary>
/// Check whether a TRect contains a TPoint </summary>
function TRect_Contains(_Rect: TRect; _Pnt: TPoint): Boolean; inline; overload;

///<summary>
/// Check whether a TRect contains a point with the given coordinates </summary>
function TRect_Contains(_Rect: TRect; _x, _y: Integer): Boolean; inline; overload;

///<summary> Returns the bounding box of the active clipping region </summary>
function TCanvas_GetClipRect(_Canvas: TCanvas): TRect;
///<summary> Sets a clipping rect, returns true, if the region is not empty, false if it is empty </summary>
function TCanvas_SetClipRect(_Canvas: TCanvas; _Rect: TRect): Boolean;

type
  TDrawTextFlags = (
    dtfLeft, dtfRight, dtfCenter, // horizontal alignment
    dtfWordBreak, // Breaks words. Lines are automatically broken between words if a word would
                  // extend past the edge of the rectangle specified by the lpRect parameter.
                  // A carriage return-line feed sequence also breaks the line.
                  // If this is not specified, output is on one line.
    dtfCalcRect, // Determines the width and height of the rectangle. If there are multiple lines
                 // of text, DrawText uses the width of the rectangle pointed to by the lpRect
                 // parameter and extends the base of the rectangle to bound the last line of text.
                 // If the largest word is wider than the rectangle, the width is expanded.
                 // If the text is less than the width of the rectangle, the width is reduced.
                 // If there is only one line of text, DrawText modifies the right side of the
                 // rectangle so that it bounds the last character in the line. In either case,
                 // DrawText returns the height of the formatted text but does not draw the text.
    dtfNoClip); // draw without clipping (slightly faster)
// not implemented:
//    dtfSingleLine, // only print as single line (ignore line breaks)
//    dtfTopSingle, dtfBottomSingle, dtfVCenterSingle, // vertical alignment, only if dtfSingleLine is given
//    dtfPathEllipsis, // replace characters in the middle of the string with ellipses ('...') so that
                     // the result fits in the specified rectangle. If the string contains backslash
                     // (\) characters, preserves as much as possible of the text after the last backslash.
//    dtfEndEllipsis, // if the end of a string does not fit in the rectangle, it is truncated and
                    // ellipses ('...') are added. If a word that is not at the end of the string
                    // goes beyond the limits of the rectangle, it is truncated without ellipses.
                    // (Unless dtfWordEllipsis is also specified.)
//    dtfWordEllipsis, // Truncates any word that does not fit in the rectangle and adds ellipses ('...').
//    dtfModifyStringEllipsis, // if given, together with one of the dtfXxxEllipsis flags, the
                             // string is modified to matcht the output.
//    dtfEditControl,
//    dtfExpandTabs, dtfExternalLeading, dtfHidePrefix, dtfInternal,
//    dtfNoFullWidthCharBreak, dtfNoPrefix,
//    dtfPrefixOnly, dtRtlReading, dtfTabStop,
  TDrawTextFlagSet = set of TDrawTextFlags;

///<summary>
/// Calculates the Rect necessary for drawing the text.
/// @returns the calculated height </summary>
function TCanvas_DrawText(_Canvas: TCanvas; const _Text: string; var _Rect: TRect; _Flags: TDrawTextFlagSet): Integer;

///<summary> calls Windows.SaveDC and returns an interface which will automatically call
///          Windows.RestoreDC when destroyed </summary>
function TCanvas_SaveDC(_Canvas: TCanvas): IInterface;

procedure TCanvas_DrawArrow(_Canvas: TCanvas; _From, _To: TPoint; _ArrowHeadLength: Integer = 15);

///<summary>
/// Draws an isoceles right triangle. The tip is either on the top or bottom, depending
/// on the sign of the Height parameter.
/// @param Canvas is the canvas to draw on
/// @param Tip is the coordinates of the vertex point
/// @param Height is the height of the triangle, if negative, the triangle is painted upside down </summary>
procedure TCanvas_DrawTriangle(_Canvas: TCanvas; _Tip: TPoint; _Height: Integer);

///<summary> abbreviation for StretchBlt that takes TRect </summary>
function dzStretchBlt(_DestHandle: Hdc; _DestRect: TRect;
  _SrcHandle: Hdc; _SrcRect: TRect; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for StretchBlt that takes TCanvas and TRect </summary>
function dzStretchBlt(_DestCnv: TCanvas; _DestRect: TRect;
  _SrcHandle: Hdc; _SrcRect: TRect; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for StretchBlt that takes TRect and TBitmap </summary>
function dzStretchBlt(_DestHandle: Hdc; _DestRect: TRect;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for StretchBlt that takes TCanvas, TRect and TBitmap </summary>
function dzStretchBlt(_DestCnv: TCanvas; _DestRect: TRect;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary>
/// Abbreviation for StretchBlt that takes two TBitmap, resizes and keeps the spect ratio,
/// using stretchmode HALFTONE (which usually gives the best quality but is a bit slower).
/// The original stretchmode and the brush origin are preserved.
/// https://msdn.microsoft.com/en-us/library/windows/desktop/dd145089(v=vs.85).aspx </summary>
function dzStretchBlt(_DestBmp, _SrcBmp: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; {inline; }
overload;

///<summary> abbreviation for BitBlt that takes TPoint and TBitmap </summary>
function dzBitBlt(_DestHandle: Hdc; _DestPos: TPoint;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for BitBlt that takes TCanvas, TPoint and TBitmap </summary>
function dzBitBlt(_DestCvn: TCanvas; _DestPos: TPoint;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for BitBlt that takes TRect and TBitmap </summary>
function dzBitBlt(_DestHandle: Hdc; _DestRect: TRect;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> abbreviation for BitBlt that takes TCanvas, TRect and TBitmap </summary>
function dzBitBlt(_DestCnv: TCanvas; _DestRect: TRect;
  _Src: TBitmap; _Rop: DWORD = SRCCOPY): LongBool; inline; overload;

///<summary> load a jpeg file and assign it to the bitmap </summary>
procedure TBitmap_LoadJpg(_bmp: TBitmap; const _JpgFn: string);

///<summary> save a bitmap as a jpeg file </summary>
procedure TBitmap_SaveJpg(_bmp: TBitmap; const _JpgFn: string);

///<summary>
/// Assign a buffer containg a bitmap in BGR 8 format to the TBitmap </summary>
procedure TBitmap_AssignBgr8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);

///<summary>
/// Assign a buffer containg a bitmap in RGB 8 format to the TBitmap
/// @NOTE: This is much slower than TBitmap_AssignBgr8, so if you have got the choice,
///        go with BGR 8 format. </summary>
procedure TBitmap_AssignRgb8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);

///<summary>
/// Assign a buffer containg a bitmap in Mono 8 format to the TBitmap with 24 bit colors </summary>
procedure TBitmap_AssignMono824(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);

///<summary>
/// Assign a buffer containg a bitmap in Mono 8 format to a 8 bit gray scale TBitmap </summary>
procedure TBitmap_AssignMono8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);

type
  TNumColors = 1..256;

function MakeGrayPalette(_NumColors: TNumColors = 256): HPALETTE;

///<summary>
// Calculates the (perceived) brightness of an RGB color value (luminance) </summary>
function ColorBrightness(_Red, _Green, _Blue: Byte): Byte; overload;
///<summary>
// Calculates the (perceived) brightness of a TColor value (luminance) </summary>
function ColorBrightness(_Color: TColor): Byte; overload;

///<summary>
/// @returns clWhite or clBlack depending on the brightness (luminance) of the color </summary>
function BestForegroundForColor(_Red, _Green, _Blue: Byte): TColor; overload;
///<summary>
/// @returns clWhite or clBlack depending on the brightness (luminance) of the color </summary>
function BestForegroundForColor(_Color: TColor): TColor; overload;

function TryStr2Color(const _s: string; out _Color: TColor): Boolean;

implementation

uses
  Math,
  GraphUtil,
  jpeg,
  u_dzConvertUtils;

function _(const _s: string): string; inline;
begin
  Result := dzDGetText(_s, 'dzlib');
end;

function TRect_Width(_Rect: TRect): Integer; inline;
begin
  Result := _Rect.Right - _Rect.Left;
end;

function TRect_Height(_Rect: TRect): Integer; inline;
begin
  Result := _Rect.Bottom - _Rect.Top;
end;

function TRect_FromLTWH(_l, _t, _w, _h: Integer): TRect;
begin
  Result := Rect(_l, _t, _l + _w, _t + _h);
end;

function TRect_Center(_Rect: TRect): TPoint; inline;
begin
  Result.X := (_Rect.Left + _Rect.Right) div 2;
  Result.Y := (_Rect.Top + _Rect.Bottom) div 2;
end;

function TRect_Contains(_Rect: TRect; _Pnt: TPoint): Boolean; inline;
begin
  Result := TRect_Contains(_Rect, _Pnt.X, _Pnt.Y);
end;

function TRect_Contains(_Rect: TRect; _x, _y: Integer): Boolean; inline; overload;
begin
  Result := (_Rect.Left <= _x) and (_Rect.Right >= _x)
    and (_Rect.Top <= _y) and (_Rect.Bottom >= _y);
end;

function dzStretchBlt(_DestHandle: Hdc; _DestRect: TRect; _SrcHandle: Hdc; _SrcRect: TRect; _Rop: DWORD): LongBool;
begin
  Result := StretchBlt(_DestHandle, _DestRect.Left, _DestRect.Top, TRect_Width(_DestRect), TRect_Height(_DestRect),
    _SrcHandle, _SrcRect.Left, _SrcRect.Top, TRect_Width(_SrcRect), TRect_Height(_SrcRect), _Rop);
end;

function dzStretchBlt(_DestCnv: TCanvas; _DestRect: TRect; _SrcHandle: Hdc; _SrcRect: TRect; _Rop: DWORD): LongBool;
begin
  Result := dzStretchBlt(_DestCnv.Handle, _DestRect, _SrcHandle, _SrcRect, _Rop);
end;

function dzStretchBlt(_DestHandle: Hdc; _DestRect: TRect; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := StretchBlt(_DestHandle, _DestRect.Left, _DestRect.Top, TRect_Width(_DestRect), TRect_Height(_DestRect),
    _Src.Canvas.Handle, 0, 0, _Src.Width, _Src.Height, _Rop);
end;

function dzStretchBlt(_DestCnv: TCanvas; _DestRect: TRect; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := dzStretchBlt(_DestCnv.Handle, _DestRect, _Src, _Rop);
end;

function dzStretchBlt(_DestBmp, _SrcBmp: TBitmap; _Rop: DWORD = SRCCOPY): LongBool;
var
  DstHandle: Hdc;
  OrigBltMode: Integer;
  OrigBrushOrigin: TPoint;
  wSrc: Integer;
  hSrc: Integer;
  X: Int64;
  Y: Integer;
  wDst: Integer;
  hDst: Integer;
begin
  DstHandle := _DestBmp.Canvas.Handle;
  OrigBltMode := GetStretchBltMode(DstHandle);
  try
    SetBrushOrgEx(DstHandle, 0, 0, @OrigBrushOrigin);
    SetStretchBltMode(DstHandle, HALFTONE);
    wDst := _DestBmp.Width;
    hDst := _DestBmp.Height;
    wSrc := _SrcBmp.Width;
    hSrc := _SrcBmp.Height;
    if (hSrc = 0) or (wSrc = 0) then begin
      // SrcBmp is empty, nothing to do
      // todo: Should this clear DestBmp?
      Result := False;
    end else begin
      if hSrc > wSrc then begin
        X := Round((wDst * (hSrc - wSrc)) / 2 / hSrc);
        Y := 0;
        wDst := Round(wDst * wSrc / hSrc);
      end else begin
        X := 0;
        Y := Round((hDst * (wSrc - hSrc)) / 2 / wSrc);
        hDst := Round(hDst * hSrc / wSrc);
      end;
      Result := dzStretchBlt(DstHandle, Rect(X, Y, X + wDst - 1, Y + hDst - 1), _SrcBmp);
    end;
  finally
    SetStretchBltMode(DstHandle, OrigBltMode);
    SetBrushOrgEx(DstHandle, OrigBrushOrigin.X, OrigBrushOrigin.Y, nil);
  end;
end;

function dzBitBlt(_DestHandle: Hdc; _DestPos: TPoint; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := BitBlt(_DestHandle, _DestPos.X, _DestPos.Y, _Src.Width, _Src.Height,
    _Src.Canvas.Handle, 0, 0, _Rop);
end;

function dzBitBlt(_DestCvn: TCanvas; _DestPos: TPoint; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := dzBitBlt(_DestCvn.Handle, _DestPos, _Src, _Rop);
end;

function dzBitBlt(_DestHandle: Hdc; _DestRect: TRect; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := BitBlt(_DestHandle, _DestRect.Left, _DestRect.Top, _DestRect.Right, _DestRect.Bottom,
    _Src.Canvas.Handle, 0, 0, _Rop);
end;

function dzBitBlt(_DestCnv: TCanvas; _DestRect: TRect; _Src: TBitmap; _Rop: DWORD): LongBool;
begin
  Result := dzBitBlt(_DestCnv.Handle, _DestRect, _Src, _Rop);
end;

function TCanvas_GetClipRect(_Canvas: TCanvas): TRect;
var
  RGN: THandle;
  Res: Integer;
begin
  RGN := CreateRectRgn(0, 0, 0, 0);
  if RGN = 0 then
    raise Exception.Create(_('CreateRectRgn failed'));
  try
    Res := GetClipRgn(_Canvas.Handle, RGN);
    if Res = -1 then
      raise Exception.Create(_('GetClipRgn failed'));
    GetRgnBox(RGN, Result);
  finally
    DeleteObject(RGN);
  end;
end;

function TCanvas_SetClipRect(_Canvas: TCanvas; _Rect: TRect): Boolean;
var
  RGN: THandle;
  Res: Integer;
begin
  Result := False;
  RGN := CreateRectRgn(_Rect.Left, _Rect.Top, _Rect.Right, _Rect.Bottom);
  if RGN = 0 then
    raise Exception.Create(_('CreateRectRgn failed'));
  try
    Res := SelectClipRgn(_Canvas.Handle, RGN);
    if Res = Error then
      raise Exception.Create(_('SelectClipRgn failed'));
    Result := (Res <> NULLREGION);
  finally
    DeleteObject(RGN);
  end;
end;

function TCanvas_DrawText(_Canvas: TCanvas; const _Text: string; var _Rect: TRect; _Flags: TDrawTextFlagSet): Integer;
var
  Flags: LongWord;
begin
  Flags := 0;
  if dtfLeft in _Flags then
    Flags := Flags or DT_LEFT;
  if dtfRight in _Flags then
    Flags := Flags or DT_RIGHT;
  if dtfCenter in _Flags then
    Flags := Flags or DT_CENTER;
  if dtfWordBreak in _Flags then
    Flags := Flags or DT_WORDBREAK;
  if dtfNoClip in _Flags then
    Flags := Flags or DT_NOCLIP;
  if dtfCalcRect in _Flags then
    Flags := Flags or DT_CALCRECT;
  Result := Windows.DrawText(_Canvas.Handle, PChar(_Text), -1, _Rect, Flags);
end;

type
  TCanvasSaveDC = class(TInterfacedObject)
  private
    FCanvas: TCanvas;
    FSavedDC: Integer;
  public
    constructor Create(_Canvas: TCanvas; _SavedDC: Integer);
    destructor Destroy; override;
  end;

{ TCanvasSaveDC }

constructor TCanvasSaveDC.Create(_Canvas: TCanvas; _SavedDC: Integer);
begin
  inherited Create;
  FCanvas := _Canvas;
  FSavedDC := _SavedDC;
end;

destructor TCanvasSaveDC.Destroy;
begin
  Windows.RestoreDC(FCanvas.Handle, FSavedDC);
  inherited;
end;

function TCanvas_SaveDC(_Canvas: TCanvas): IInterface;
var
  SavedDC: Integer;
begin
  SavedDC := Windows.SaveDC(_Canvas.Handle);
  Result := TCanvasSaveDC.Create(_Canvas, SavedDC);
end;

procedure TCanvas_DrawArrow(_Canvas: TCanvas; _From, _To: TPoint; _ArrowHeadLength: Integer = 15);
// taken from: http://www.efg2.com/Lab/Library/Delphi/Graphics/Arrow.Txt
var
  xbase: Integer;
  xLineDelta: Integer;
  xLineUnitDelta: Double;
  xNormalDelta: Integer;
  xNormalUnitDelta: Double;
  ybase: Integer;
  yLineDelta: Integer;
  yLineUnitDelta: Double;
  yNormalDelta: Integer;
  yNormalUnitDelta: Double;
begin
  _Canvas.MoveTo(_From.X, _From.Y);
  _Canvas.LineTo(_To.X, _To.Y);

  xLineDelta := _To.X - _From.X;
  yLineDelta := _To.Y - _From.Y;

  xLineUnitDelta := xLineDelta / Sqrt(Sqr(xLineDelta) + Sqr(yLineDelta));
  yLineUnitDelta := yLineDelta / Sqrt(Sqr(xLineDelta) + Sqr(yLineDelta));

  // (xBase,yBase) is where arrow line is perpendicular to base of triangle.
  xbase := _To.X - Round(_ArrowHeadLength * xLineUnitDelta);
  ybase := _To.Y - Round(_ArrowHeadLength * yLineUnitDelta);

  xNormalDelta := yLineDelta;
  yNormalDelta := -xLineDelta;
  xNormalUnitDelta := xNormalDelta / Sqrt(Sqr(xNormalDelta) + Sqr(yNormalDelta));
  yNormalUnitDelta := yNormalDelta / Sqrt(Sqr(xNormalDelta) + Sqr(yNormalDelta));

  // Draw the arrow tip
  _Canvas.Polygon([_To,
    Point(xbase + Round(_ArrowHeadLength * xNormalUnitDelta),
      ybase + Round(_ArrowHeadLength * yNormalUnitDelta)),
      Point(xbase - Round(_ArrowHeadLength * xNormalUnitDelta),
      ybase - Round(_ArrowHeadLength * yNormalUnitDelta))]);
end;

procedure TCanvas_DrawTriangle(_Canvas: TCanvas; _Tip: TPoint; _Height: Integer);
var
  BaselineY: Integer;
  BaselineLeft: Integer;
  BaselineRight: Integer;
begin
  BaselineY := _Tip.Y + _Height;
  BaselineLeft := _Tip.X - Abs(_Height);
  BaselineRight := _Tip.X + Abs(_Height);
  _Canvas.Polygon([_Tip, Point(BaselineLeft, BaselineY), Point(BaselineRight, BaselineY)]);
end;

{ TdzRgbTriple }

function TdzRgbTriple.GetBrightness(_Channel: TRgbBrightnessChannelEnum): Byte;
begin
  case _Channel of
    rcbAverage: Result := Round((Red + Green + Blue) / 3);
    rcbFastLuminance: Result := GetFastLuminance;
    rcbRed: Result := Red;
    rcbGreen: Result := Green;
    rcbBlue: Result := Blue;
  else //  rcbLuminance: ;
{$IFDEF dzUseGraphics32}
    Result := Round(GetLuminance * HLSMAX);
{$ELSE}
    Result := GetLuminance;
{$ENDIF}
  end;
end;

function TdzRgbTriple.GetColor: TColor;
begin
  Result := RGB(Red, Green, Blue);
end;

procedure TdzRgbTriple.SetColor(_Color: TColor);
begin
  _Color := ColorToRGB(_Color);
  Red := GetRValue(_Color);
  Green := GetGValue(_Color);
  Blue := GetBValue(_Color);
end;

procedure TdzRgbTriple.SetBrightness(_Value: Byte);
begin
  Red := _Value;
  Green := _Value;
  Blue := _Value;
end;

function TdzRgbTriple.GetFastLuminance: Byte;
begin
  Result := Round(0.299 * Red + 0.587 * Green + 0.114 * Blue);
end;

class function TdzRgbTriple.GetFastLuminance(_Red, _Green, _Blue: Byte): Byte;
begin
  Result := Round(0.299 * _Red + 0.587 * _Green + 0.114 * _Blue);
end;

{$IFDEF dzUseGraphics32}

procedure TdzRgbTriple.GetHls(out _Hls: THlsRec);
begin
  GR32.RGBtoHSL(GR32.Color32(GetColor), _Hls.Hue, _Hls.Saturation, _Hls.Luminance);
end;

procedure TdzRgbTriple.SetHls(const _Hls: THlsRec);
begin
  SetColor(GR32.WinColor(GR32.HSLtoRGB(_Hls.Hue, _Hls.Saturation, _Hls.Luminance)));
end;

{$ELSE}

procedure TdzRgbTriple.GetHls(out _Hls: THlsRec);
begin
  ColorRGBToHLS(GetColor, _Hls.Hue, _Hls.Luminance, _Hls.Saturation);
end;

procedure TdzRgbTriple.SetHls(const _Hls: THlsRec);
begin
  SetColor(ColorHLSToRGB(_Hls.Hue, _Hls.Luminance, _Hls.Saturation));
end;

{$ENDIF}

// untested from http://www.swissdelphicenter.ch/en/showcode.php?id=2349
//function RGB2HSV (R,G,B : Byte) : THSV;
//var
//  Min_, Max_, Delta : Double;
//  H , S , V : Double ;
//begin
//  H := 0.0 ;
//  Min_ := Min (Min( R,G ), B);
//  Max_ := Max (Max( R,G ), B);
//  Delta := ( Max_ - Min_ );
//  V := Max_ ;
//  If ( Max_ <> 0.0 ) then
//    S := 255.0 * Delta / Max_
//  else
//    S := 0.0 ;
//  If (S <> 0.0) then
//    begin
//      If R = Max_ then
//        H := (G - B) / Delta
//      else
//        If G = Max_ then
//          H := 2.0 + (B - R) / Delta
//        else
//          If B = Max_ then
//            H := 4.0 + (R - G) / Delta
//    End
//  else
//    H := -1.0 ;
//  H := H * 60 ;
//  If H < 0.0 then H := H + 360.0;
//  with Result Do
//    begin
//      Hue := H ;             // Hue -> 0..360
//      Sat := S * 100 / 255; // Saturation -> 0..100 %
//      Val := V * 100 / 255; // Value - > 0..100 %
//    end;
//end;

//procedure Swap(var _a, _b: Byte);
//var
//  t: Byte;
//begin
//  t := _a;
//  _a := _b;
//  _b := t;
//end;

// untested: Delphi implmementations of
// http://lolengine.net/blog/2013/01/13/fast-rgb-to-hsv
//procedure RGB2HSV(_r, _g, _b: Byte; out _h, _s, _v: Single);
//var
//  k: Single;
//  chroma: Single;
//begin
//  k := 0;
//
//  if _g < _b then begin
//    Swap(_g, _b);
//    k := -1;
//  end;
//
//  if _r < _g then begin
//    Swap(_r, _g);
//    k := -2 / 6 - k;
//  end;
//
//  chroma := _r - min(_g, _b);
//  _h := Abs(k + (_g - _b) / (6 * chroma + 1e-20));
//  _s := chroma / (_r + 1e-20);
//  _v := _r;
//end;

//procedure RGBtoHSL(_r, _g, _b: Byte; out _h, _s, _l: Single);
//var
//  k: Single;
//  lightness: Integer;
//  chroma: Integer;
//begin
//  k := 0.0;
//  if (_g < _b) then begin
//    Swap(_g, _b);
//    k := 6.0;
//  end;
//  if (_r < _g) then begin
//
//    Swap(_r, _g);
//    k := 2.0 - k;
//  end;
//  lightness := _r + min(_g, _b);
//  chroma := _r - min(_g, _b);
//  if (chroma <> 0) then begin
//    _h := Abs((_g - _b) / chroma - k) * 1.0 / 6.0;
//    _s := chroma / (255 - Abs(lightness - 255));
//  end else begin
//    _h := 0.0;
//    _s := 0.0;
//  end;
//  _l = lightness * 1.0 / 510.0;
//end;

function TdzRgbTriple.GetLuminance: Byte;
var
  Hls: THlsRec;
begin
  GetHls(Hls);
{$IFDEF dzUseGraphics32}
  Result := Round(Hls.Luminance * HLSMAX);
{$ELSE}
  Result := Hls.Luminance;
{$ENDIF}
end;

procedure TdzRgbTriple.SetGray(_Value: Byte);
begin
  Red := _Value;
  Green := _Value;
  Blue := _Value;
end;

{ tdzRgbQuad }

function TdzRgbQuad.GetBrightness(_Channel: TRgbBrightnessChannelEnum): Word;
begin
  case _Channel of
    rcbAverage: Result := Round((Red + Green + Blue) / 3);
    rcbFastLuminance: Result := GetFastLuminance;
    rcbRed: Result := Red;
    rcbGreen: Result := Green;
    rcbBlue: Result := Blue;
  else //  rcbLuminance: ;
    Result := GetLuminance;
  end;
end;

function TdzRgbQuad.GetColor: TColor;
begin
  Result := RGB(Red, Green, Blue);
end;

function TdzRgbQuad.GetFastLuminance: Word;
begin
  Result := Round(0.299 * Red + 0.587 * Green + 0.114 * Blue);
end;

procedure TdzRgbQuad.GetHls(out _Hue, _Luminance, _Saturation: Word);
begin
  ColorRGBToHLS(GetColor, _Hue, _Luminance, _Saturation)
end;

function TdzRgbQuad.GetLuminance: Word;
var
  Hue: Word;
  Saturation: Word;
begin
  GetHls(Hue, Result, Saturation)
end;

procedure TdzRgbQuad.SetBrightness(_Value: Byte);
begin
  Red := _Value;
  Green := _Value;
  Blue := _Value;
end;

procedure TdzRgbQuad.SetColor(_Color: TColor);
begin
  _Color := ColorToRGB(_Color);
  Red := GetRValue(_Color);
  Green := GetGValue(_Color);
  Blue := GetBValue(_Color);
end;

procedure TdzRgbQuad.SetGray(_Value: Byte);
begin
  Red := _Value;
  Green := _Value;
  Blue := _Value;
end;

procedure TdzRgbQuad.SetHls(_Hue, _Luminance, _Saturation: Word);
begin
  SetColor(ColorHLSToRGB(_Hue, _Luminance, _Saturation));
end;

procedure TBitmap_LoadJpg(_bmp: TBitmap; const _JpgFn: string);
var
  jpg: TJPEGImage;
begin
  jpg := TJPEGImage.Create;
  try
    jpg.LoadFromFile(_JpgFn);
    _bmp.Assign(jpg);
  finally
    FreeAndNil(jpg);
  end;
end;

procedure TBitmap_SaveJpg(_bmp: TBitmap; const _JpgFn: string);
var
  jpg: TJPEGImage;
begin
  jpg := TJPEGImage.Create;
  try
    jpg.Assign(_bmp);
    jpg.SaveToFile(_JpgFn);
  finally
    FreeAndNil(jpg);
  end;
end;

function MakeGrayPalette(_NumColors: TNumColors): HPALETTE;
var
  i: Integer;
  lp: TMaxLogPalette;
  Grey: Byte;
begin
  lp.palVersion := $300;
  lp.palNumEntries := _NumColors;
  for i := 0 to _NumColors - 1 do begin
    Grey := i * 255 div _NumColors;
    lp.palPalEntry[i].peRed := Grey;
    lp.palPalEntry[i].peGreen := Grey;
    lp.palPalEntry[i].peBlue := Grey;
    lp.palPalEntry[i].peFlags := PC_RESERVED;
  end;
  Result := CreatePalette(pLogPalette(@lp)^);
end;

procedure TBitmap_AssignBgr8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);
var
  Y: Integer;
  ScanLine: PdzRgbTripleArray;
  BytesPerLine: Integer;
//  ms: TMemoryStream;
//  bfh: TBitmapFileHeader;
//  bih: TBitmapInfoHeader;
begin
  Assert(_bmp.PixelFormat = pf24bit, 'unexpected PixelFormat (expected pf24bit');

  BytesPerLine := 3 * _bmp.Width;

//  bfh.bfType := $4D42; // 'BM'
//  bfh.bfSize := BytesPerLine * _Bmp.Height;
//  bfh.bfReserved1 := 0;
//  bfh.bfReserved2 := 0;
//  bfh.bfOffBits := SizeOf(bfh);

//  bih.biSize := SizeOf(bih);
//  bih.biWidth := _Bmp.Width;
//  bih.biHeight := -_Bmp.Height; // origin is upper left corner -> negative
//  bih.biPlanes := 1;
//  bih.biBitCount := 24;
//  bih.biCompression := BI_RGB;
//  bih.biSizeImage := 0; // The size, in bytes, of the image. This may be set to zero for BI_RGB bitmaps.
//  bih.biXPelsPerMeter := 1000;
//  bih.biYPelsPerMeter := 1000;
//  bih.biClrUsed := 0; // The number of color indexes in the color table that are actually used by the bitmap. If this value is zero, the bitmap uses the maximum number of colors corresponding to the value of the biBitCount member for the compression mode specified by biCompression.
//  bih.biClrImportant := 0;

//  ms := TMemoryStream.Create;
//  ms.WriteBuffer(bfh, SizeOf(bfh));
//  ms.WriteBuffer(_Buffer^, bfh.bfSize);

  // Unfortunately the y coordinates of TBitmap are reversed (the picture is upside down).
  // So we can only copy the whole picture in one go, if the buffer is also upside down
  // (many cameras have this feature). If not, we have to copy it one line at a time.
  if _YIsReversed then begin
    ScanLine := _bmp.ScanLine[_bmp.Height - 1];
    Move(_Buffer^, ScanLine^, _bmp.Height * BytesPerLine);
  end else begin
    // At least with GBR8 the bytes have the right order so we can copy the whole line in one go
    for Y := 0 to _bmp.Height - 1 do begin
      ScanLine := _bmp.ScanLine[Y];
      Move(_Buffer^, ScanLine^, BytesPerLine);
      Inc(_Buffer, BytesPerLine);
    end;
  end;
end;

procedure TBitmap_AssignRgb8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);
var
  Y: Integer;
  X: Integer;
  ScanLine: PdzRgbTripleArray;
  h: Integer;
begin
  Assert(_bmp.PixelFormat = pf24bit, 'unexpected PixelFormat (expected pf24bit');

  h := _bmp.Height;
  for Y := 0 to h - 1 do begin
    if _YIsReversed then begin
      ScanLine := _bmp.ScanLine[h - Y - 1];
    end else begin
      ScanLine := _bmp.ScanLine[Y];
    end;
    for X := 0 to _bmp.Width - 1 do begin
      // unfortunately the bytes in the buffer have a different order (RGB) than in the
      // Bitmap (BGR) so we must copy each byte separately
      ScanLine[X].Red := _Buffer^;
      Inc(_Buffer);
      ScanLine[X].Green := _Buffer^;
      Inc(_Buffer);
      ScanLine[X].Blue := _Buffer^;
      Inc(_Buffer);
    end;
  end;
end;

procedure TBitmap_AssignMono824(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);
var
  Y: Integer;
  X: Integer;
  ScanLine: PdzRgbTripleArray;
  h: Integer;
  Value: Byte;
begin
  Assert(_bmp.PixelFormat = pf24bit, 'unexpected PixelFormat (expected pf24bit');
  h := _bmp.Height;
  for Y := 0 to h - 1 do begin
    if _YIsReversed then begin
      ScanLine := _bmp.ScanLine[h - Y - 1];
    end else begin
      ScanLine := _bmp.ScanLine[Y];
    end;
    for X := 0 to _bmp.Width - 1 do begin
      // gray scale: Set all colours to the same value
      Value := _Buffer^;
      ScanLine[X].Red := Value;
      ScanLine[X].Green := Value;
      ScanLine[X].Blue := Value;
      Inc(_Buffer);
    end;
  end;
end;

procedure TBitmap_AssignMono8(_Buffer: PByte; _bmp: TBitmap; _YIsReversed: Boolean);
var
  Y: Integer;
  ScanLine: PByte;
begin
  Assert(_bmp.PixelFormat = pf8bit, 'unexpected PixelFormat (expected pf8bit');

  // Unfortunately the y coordinates of TBitmap are reversed (the picture is upside down).
  // So we can only copy the whole picture in one go, if the buffer is also upside down
  // (many cameras have this feature). If not, we have to copy it one line at a time.
  if _YIsReversed then begin
    ScanLine := _bmp.ScanLine[_bmp.Height - 1];
    Move(_Buffer^, ScanLine^, _bmp.Height * _bmp.Width);
  end else begin
    for Y := 0 to _bmp.Height - 1 do begin
      ScanLine := _bmp.ScanLine[Y];
      Move(_Buffer^, ScanLine^, _bmp.Width);
      Inc(_Buffer, _bmp.Width);
    end;
  end;
end;

function ColorBrightness(_Red, _Green, _Blue: Byte): Byte;
begin
  Result := TdzRgbTriple.GetFastLuminance(_Red, _Green, _Blue);
end;

function ColorBrightness(_Color: TColor): Byte;
var
  RGB: TdzRgbTriple;
begin
  RGB.SetColor(_Color);
  Result := RGB.GetFastLuminance;
end;

function BestForegroundForColor(_Red, _Green, _Blue: Byte): TColor;
begin
  if ColorBrightness(_Red, _Green, _Blue) < 123 then
    Result := clWhite
  else
    Result := clBlack;
end;

function BestForegroundForColor(_Color: TColor): TColor;
begin
  if ColorBrightness(_Color) < 123 then
    Result := clWhite
  else
    Result := clBlack;
end;

function TryStr2Color(const _s: string; out _Color: TColor): Boolean;
var
  c: Integer;
begin
  Result := IdentToColor(_s, c);
  if not Result then
    Result := TryStr2Int(_s, c);
  if Result then
    _Color := c;
end;

end.
