/*
 *
 *    WaveBuffer.cpp                                         
 *                              (c) HAL 2010-           
 *
 *  This class provides I/O functions for wave files.
 * It keeps raw wave as a 64-bit array buffer.
 *
 * These files are distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include "WaveBuffer.h"

#include "RiffFile.h"
#include "stdint.h"

using namespace std;
using namespace stand::io;

const WaveBuffer::Format WaveBuffer::DefaultFormat =
{
    1,
    1,
    44100,
    88200,
    2,
    16,
};

WaveBuffer::WaveBuffer(const char *filename)
{
    _data = NULL;
    _length = 0;
    setHeader(DefaultFormat);
    read(filename);
}

WaveBuffer::WaveBuffer(const double *w, unsigned int l)
{
    _data = NULL;
    _length = 0;
    setHeader(DefaultFormat);
    setWave(w, l);
}

WaveBuffer::WaveBuffer(unsigned int l)
{
    _data = NULL;
    setHeader(DefaultFormat);
    if(l)
    {
        setLength(l);
    }
    else
    {
        _length = 0;
    }
}

WaveBuffer::~WaveBuffer()
{
    _destroy();
}

void WaveBuffer::_destroy()
{
    delete[] _data;
    _data = NULL;
    _length = 0;
}

bool WaveBuffer::read(const char *filename)
{
    qDebug("WaveBuffer::read(%s)", filename);
    RiffFile f;
    if(!f.read(filename))
    {
        return false;
    }
    if(!f.is("WAVE"))
    {
        qDebug(" WaveBuffer::read(%s); // This is not WAVE file", filename);
        return false;
    }
    const RiffChunk *c;
    if((c = f.chunk("fmt ")) == NULL)
    {
        qDebug(" WaveBuffer::read(%s); // fmt not fount", filename);
        return false;
    }
    _setHeader(c);


    if((c = f.chunk("data")) == NULL)
    {
        qDebug(" WaveBuffer::read(%s); // data not found", filename);
        return false;
    }
    _setData(c);

    return true;
}

void WaveBuffer::_setHeader(const RiffChunk *fmt)
{
    if(fmt->length() < 16)
    {
        _header = DefaultFormat;
    }
    else
    {
        memcpy(&_header, fmt->data(), sizeof(Format));
    }
    qDebug(" read buffer:\n"
           "  Sampling Frequency = %d[Hz]\n"
           "  Bits per sample = %d[bits]\n"
           "  Number of channels = %d"
           , _header.samplesPerSecond, _header.bitsPerSample, _header.numChannels);
}

void WaveBuffer::_setData(const RiffChunk *data)
{
    int l = data->length() / _header.numChannels / (_header.bitsPerSample / 8);
    _createBuffer(l);
    switch(_header.bitsPerSample)
    {
    case 8:
        {
            for(int i = 0; i < data->length(); i += _header.numChannels)
            {
                _data[i / _header.numChannels] = (data->data()[i] - 128) / 128.0;
            }
        }
        break;
    case 16:
        {
            const int16_t *p = (int16_t *)(data->data());
            for(int i = 0; i < data->length() / 2; i += _header.numChannels)
            {
                _data[i / _header.numChannels] = p[i] / (double)(1 << 15);
            }
        }
        break;
    case 24:
        {
            for(int i = 0; i < data->length(); i += _header.numChannels * (24 / 8))
            {
                int32_t val = 0;
                char *p = (char *)(&val);
                p[3] = data->data()[i + 2];
                p[2] = data->data()[i + 1];
                p[1] = data->data()[i + 0];
                _data[i / 3 / _header.numChannels] = val / (double)(1 << 31);
            }
        }
        break;
    case 32:
        {
            const int32_t *p = (int32_t *)(data->data());
            for(int i = 0; i < data->length() / 4; i += _header.numChannels)
            {
                _data[i / _header.numChannels] = data->data()[i] / (double)(1 << 31);
            }
        }
        break;
    default:
        return;
    }
}


void WaveBuffer::setWave(const double *w, unsigned int l)
{
    if(!w || !l)
    {
        qDebug("WaveBuffer::setWave(%08x, %ud); // invalid args", w, l);
        return;
    }
    _createBuffer(l);
    for(unsigned int i = 0; i < l; i++)
    {
        _data[i] = w[i];
    }
}

bool WaveBuffer::write(const char *filename)
{
    qDebug("WaveBuffer::write(%s)", filename);
    if(empty())
    {
        qDebug(" This WaveBuffer class does not contain any data.");
        return false;
    }
    FILE *fp = fopen(filename, "wb");
    if(!fp)
    {
        qDebug(" Could not open File; %s", filename);
        return false;
    }
    qint32 waveSize = _length * (_header.bitsPerSample / 8);
    qint8 *buf = new qint8[waveSize];
    qint32 size = 36 + waveSize; //44 + waveSize; 計算ミスった…死にたい…
    qint32 chunksize = 16;

    switch(_header.bitsPerSample)
    {
    case 8:
        _writeData8(buf);
        break;
    case 24:
        _writeData24(buf);
        break;
    case 32:
        _writeData32(buf);
        break;
    default:
        qDebug(" Unsupported bitrate; %d", _header.bitsPerSample);
        _header.bitsPerSample = 16;
    case 16:
        _writeData16(buf);
    }

    fprintf(fp, "RIFF");
    fwrite(&size, 4, 1, fp);

    fprintf(fp, "WAVEfmt ");
    fwrite(&chunksize, 4, 1, fp);
    fwrite(&_header.formatID, 2, 1, fp);
    fwrite(&_header.numChannels, 2, 1, fp);
    fwrite(&_header.samplesPerSecond, 4, 1, fp);
    fwrite(&_header.bytesPerSecond, 4, 1, fp);
    fwrite(&_header.blockAlign, 2, 1, fp);
    fwrite(&_header.bitsPerSample, 2, 1, fp);

    fprintf(fp, "data");
    fwrite(&waveSize, 4, 1, fp);
    fwrite(buf, sizeof(qint8), waveSize, fp);

    fclose(fp);
    delete[] buf;

    return true;
}

void WaveBuffer::_writeData8(qint8 *p)
{
    for(unsigned int i = 0; i < _length; i++)
    {
        p[i] = (qint8)(_data[i] * 127.0) + 127;
    }
}

void WaveBuffer::_writeData16(qint8 *p)
{
    qint16 *p16 = (qint16 *)p;
    for(unsigned int i = 0; i < _length; i++)
    {
        p16[i] = _data[i] * ((2 << 15) - 1);
    }
}

void WaveBuffer::_writeData24(qint8 *p)
{
    for(unsigned int i = 0; i < _length; i++)
    {
        qint32 val = _data[i] * ((2 << 23) - 1);
        qint8 *v = (qint8 *)(&val);
        p[i*3] = v[1];
        p[i*3+1] = v[2];
        p[i*3+2] = v[3];
    }
}

void WaveBuffer::_writeData32(qint8 *p)
{
    qint32 *p32 = (qint32 *)p;
    for(unsigned int i = 0; i < _length; i++)
    {
        p32[i] = _data[i] * ((2 << 31) - 1);
    }
}

double WaveBuffer::normalize()
{
    if(!_data)
    {
        return 0.0;
    }
    double maxVal = 0.0;
    for(int i = 0; i < _length; i++)
    {
        double val = fabs(_data[i]);
        if(val > maxVal)
        {
            maxVal = val;
        }
    }
    if(maxVal == 0.0)
    {
        return 0.0;
    }
    for(int i = 0; i < _length; i++)
    {
        _data[i] /= maxVal;
    }
    return maxVal;
}

void WaveBuffer::_createBuffer(unsigned int l)
{
    _destroy();
    _length = l;
    _data = new double[l];
}
