#include "MidiFile.h"

#include "MidiChunk.h"
#include "MidiEvent.h"
#include "ExclusiveEvent.h"
#include "MetaEvent.h"
#include "../../utility/Utility.h"

#include <stdint.h>
#include <string.h>

#include <QDebug>

using namespace stand::io::midi;

const char *MidiFile::getDeltaTime(int &d, const char *p)
{
    unsigned int c = -1, current = 0;
    do
    {
        c++;
        current = current << 7;
        current += ((unsigned char)(p[c]) & 0x7F);
    }
    while(p[c] & 0x80);
    d = current;
    return &(p[c + 1]);
}

MidiFile::MidiFile()
{
}

MidiFile::~MidiFile()
{
    _destroy();
}

void MidiFile::_destroy()
{
    for(int i = 0; i < _tracks.size(); i++)
    {
        for(int j = 0; j < _tracks[i].size(); j++)
        {
            delete _tracks[i][j];
        }
    }
    _tracks.clear();

    for(int i = 0; i < _chunks.size(); i++)
    {
        delete _chunks[i];
    }
    _chunks.clear();
}

bool MidiFile::read(const char *path)
{
    if(!path)
    {
        qDebug("MidiFile::read(0); // invalid arg.");
        return false;
    }
    FILE *fp = fopen(path, "rb");
    if(!fp)
    {
        qDebug("MidiFile::read(%s); // File could not be opened.", path);
        return false;
    }

    _destroy();

    fread(_header, sizeof(char), 4, fp);
    _header[4] = '\0';
    if(strcmp(_header, "MThd"))
    {
        qDebug("MidiFile::read(%s); // This file is not MIDI file.", path);
        fclose(fp);
        return false;
    }

    int32_t size;
    fread(&size , sizeof(int), 1, fp);
    size = stand::utility::FromBigEndian(size);
    if(size != 6)
    {
        fclose(fp);
        return false;
    }

    fread(&_formatId, sizeof(short), 1, fp);
    _formatId = stand::utility::FromBigEndian(_formatId);
    fread(&_numTracks, sizeof(short), 1, fp);
    _numTracks = stand::utility::FromBigEndian(_numTracks);
    fread(&_timeBase, sizeof(short), 1, fp);
    _timeBase = stand::utility::FromBigEndian(_timeBase);

    while(!feof(fp))
    {
        MidiChunk *c = new MidiChunk();
        if(c->set(fp))
        {
            _chunks.push_back(c);
        }
        else
        {
            delete c;
        }
    }

    _tracks.resize(_chunks.size());
    for(int i = 0; i < _chunks.size(); i++)
    {
        _interpret(_chunks[i], i);
    }

    fclose(fp);
    return true;
}

void MidiFile::_interpret(MidiChunk *chunk, int index)
{
    const char *p = chunk->data();
    int tick = 0;
    while((long)(p) - (long)(chunk->data()) < (long)(chunk->length()) - 2)
    {
        int diff;
        p = getDeltaTime(diff, p);
        tick += diff;
        MidiEvent *e = NULL;
        unsigned char type = p[0];
        if(type == 0xff)
        {
            e = new MetaEvent();
        }
        else if(type == 0xf0 || type == 0xff)
        {
            e = new ExclusiveEvent();
        }
        else
        {
            e = new MidiEvent();
        }
        if(e)
        {
            p = e->set(p, tick);
            _tracks[index].push_back(e);
            if(e->type() == 0x2f && e->length() == 0)
            {
                break;
            }
        }
        else
        {
            break;
        }
    }
}
