#include "NormalDistribution.h"

#include "Matrix.h"
#include "MathSet.h"

#include <stdio.h>
#include <math.h>

using namespace stand::math;
using namespace stand::math::matrix;

NormalDistribution::NormalDistribution(double *mu, double **sigma, int dim)
{
    _mu = NULL;
    _sigma = _sigmaInverse = NULL;
    set(mu, sigma, dim);
}

NormalDistribution::~NormalDistribution()
{
    _destroy();
}

void NormalDistribution::_create(int dim)
{
    _destroy();
    _dim = dim;
    _mu = new double[dim];
    CreateMatarix(&_sigma, dim, dim);
    CreateMatarix(&_sigmaInverse, dim, dim);
}

void NormalDistribution::_destroy()
{
    delete[] _mu;
    DestroyMatrix(&_sigma);
    DestroyMatrix(&_sigmaInverse);
}

double NormalDistribution::f(const double *x, int dim)
{
    if(!x || dim <= 0 || !_mu || !_sigma || !_sigmaInverse)
    {
        fprintf(stderr, "NormalDistribution::f(%d, %d); // Invalid args or matrix not set.\n", x, dim);
        return -1;
    }
    // 与えられたベクトルから逆行列をかけて内積を求める．
    double *xTSigmaInverse = new double[dim];
    for(int i = 0; i < dim; i++)
    {
        xTSigmaInverse[i] = 0;
        for(int j = 0; j < dim; j++)
        {
            xTSigmaInverse[i] += (x[j] - _mu[j]) * _sigmaInverse[i][j];
        }
    }
    double exped = 0;
    for(int i = 0; i < dim; i++)
    {
        exped += xTSigmaInverse[i] * (x[i] - _mu[i]);
    }
    exped *= -0.5;

    delete[] xTSigmaInverse;

    // exp の係数を計算する．
    double val = exped - log(pow(2.0 * PI, (double)dim / 2.0)) - 0.5 * log(_determinant);

    return val;
}

void NormalDistribution::set(const double *mu, double **sigma, int dim)
{
    if(!mu || !sigma || dim <= 0)
    {
        return;
    }
    if(dim != _dim)
    {
        _create(dim);
    }

    // 値をコピーする．
    for(int i = 0; i < dim; i++)
    {
        _mu[i] = mu[i];
        for(int j = 0; j < dim; j++)
        {
            _sigma[i][j] = sigma[i][j];
        }
    }
    update();
}

void NormalDistribution::set(double **data, int dim, int N)
{
    double *mu = new double[dim];
    double **sigma;
    CreateMatarix(&sigma, dim, dim);
    stand::math::average(mu, data, N, dim);
    stand::math::matrix::Covariance(sigma, data, mu, dim, N);

    set(mu, sigma, dim);

    delete[] mu;
    DestroyMatrix(&sigma);
}

void NormalDistribution::set(int dim)
{
    if(dim <= 0 || dim == _dim)
    {
        return;
    }
    _create(dim);
}

void NormalDistribution::update()
{
    // 逆行列を計算する．
    GaussianElimination(_sigmaInverse, _sigma, _dim);

    // 行列式を計算する．
    _determinant = Determinant(_sigma, _dim);
}

void NormalDistribution::write(FILE *fp) const
{
    if(!fp)
    {
        return;
    }
    fprintf(fp, "Average Vector: \n");
    for(int i = 0; i < _dim; i++)
    {
        fprintf(fp, "%f\n", _mu[i]);
    }
    fprintf(fp, "Covariance Matrix: \n");
    for(int i = 0; i < _dim; i++)
    {
        for(int j = 0; j < _dim; j++)
        {
            fprintf(fp, "%f\t", _sigma[j][i]);
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "Covariance Inverse Matrix: \n");
    for(int i = 0; i < _dim; i++)
    {
        for(int j = 0; j < _dim; j++)
        {
            fprintf(fp, "%f\t", _sigmaInverse[j][i]);
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "Determinant = %e \n", _determinant);
}
