#ifndef NORMALDISTRIBUTION_H
#define NORMALDISTRIBUTION_H

#include <stdio.h>

namespace stand
{
namespace math
{
const double PI = 3.1415926535897932384;

/*!
 *  @brief 多次元正規分布を保持，計算するクラス．
 */
class NormalDistribution
{
public:
    explicit NormalDistribution(double *mu = 0, double **sigma = 0, int dim = 0);
    ~NormalDistribution();

    /*!
     *  @brief 保持している正規分布において，与えられたベクトル x を出力する確率を計算する．
     *  @param[in] x 確率を計算したいベクトル．
     *  @param[in] dim ベクトルの次元数．セーフガード．
     */
    double f(const double *x, int dim);

    /*!
     *  @brief 正規分布を設定する．
     *  @param[in] mu 平均ベクトル．
     *  @param[in] sigma 共分散行列．
     *  @param[in] dim 次元数．
     */
    void set(const double *mu, double **sigma, int dim);

    /*!
     * @brief 正規分布を設定する．
     * @param[in] data サンプル．
     * @param[in] dim ベクトルの次元数
     * @param[in] N サンプル数
     */
    void set(double **data, int dim, int N);

    /*!
     *  @brief 空の正規分布を生成します．
     *  @param[in] dim 次元数
     */
    void set(int dim);

    void update();

    int dimension()
    {
        return _dim;
    }

    double *average()
    {
        return _mu;
    }

    double **convariance()
    {
        return _sigma;
    }

    void write(FILE *fp) const;

private:
    void _destroy();
    void _create(int dim);

    int _dim;               //! @brief 正規分布の次元数
    double *_mu;            //! @brief 正規分布の平均値
    double **_sigma;        //! @brief 正規分布の共分散行列
    double **_sigmaInverse; //! @brief 共分散行列の逆行列
    double _determinant;    //! @brief 共分散行列の行列式
};

}
}

#endif // NORMALDISTRIBUTION_H
