//-----------------------------------------------------------------------------
// Matlab functions implemented for WORLD
// Since these functions are implemented as the same function of Matlab,
// the source code does not follow the style guide (Names of variables 
// and functions).
// Please see the reference of Matlab to show the usage of functions.
// Caution:
//   Since these functions (wavread() and wavwrite())are roughly implemented,
//   we recomend more suitable functions provided by other organizations.
//-----------------------------------------------------------------------------
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "world.h"

namespace stand
{
namespace math
{
namespace dsp
{

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)<(b))?(b):(a))

//-----------------------------------------------------------------------------
// histc() counts the number of values in vector x that fall between the 
// elements in the edges vector (which must contain monotonically 
// nondecreasing values). n is a length(edges) vector containing these counts. 
// No elements of x can be complex.
// http://www.mathworks.co.jp/help/techdoc/ref/histc.html
// Input: 
//   x              : Input vector
//   x_length       : Length of x
//   edges          : Input matrix (1-dimension)
//   edges_length   : Length of edges
// Output:
//   index          : Result counted in vector x
// Caution:
//   Lengths of index and edges must be the same.
//-----------------------------------------------------------------------------
void histc(const double *x, int x_length, double *edges, int edges_length,
  int *index)
{
  int count = 1;

  int i = 0;
  for(;i < edges_length;i++) 
  {
    index[i] = 1;
    if(edges[i] >= x[0]) break;
  }
  for(;i < edges_length;i++)
  {
    if(edges[i] < x[count]) index[i] = count;
    else
    {
      index[i] = count++;
      i--;
    }
    if(count == x_length) break;
  }
  count--;
  for(i++;i < edges_length;i++) index[i] = count;
}

//-----------------------------------------------------------------------------
// interp1() interpolates to find yi, the values of the underlying function Y 
// at the points in the vector or array xi. x must be a vector.
// http://www.mathworks.co.jp/help/techdoc/ref/interp1.html
// Input:
//   x          : Input vector (Time axis)
//   y          : Values at x[n]
//   x_length   : Length of x (Length of y must be the same)
//   xi         : Required vector
//   xi_length  : Length of xi (Length of yi must be the same)
// Output:
//   yi         : Interpolated vector
//-----------------------------------------------------------------------------
void interp1(const double *x, double *y, int x_length, double *xi, int xi_length,
  double *yi)
{
  double *h = (double *)malloc(sizeof(double) * (x_length-1));
  double *p = (double *)malloc(sizeof(double) * xi_length);
  double *s = (double *)malloc(sizeof(double) * xi_length);
  int *k = (int   *)malloc(sizeof(int)   * xi_length);
  
  for(int i = 0;i < x_length-1;i++) h[i] = x[i+1]-x[i];
  for(int i = 0;i < xi_length;i++) 
  {
    p[i] = i; 
    k[i] = 0;
  }

  histc(x, x_length, xi, xi_length, k);

  for(int i = 0;i < xi_length;i++) s[i] = (xi[i] - x[k[i]-1]) / h[k[i]-1];

  for(int i = 0;i < xi_length;i++) 
    yi[i] = y[k[i]-1] + s[i]*(y[k[i]] - y[k[i]-1]);

  free(k);
  free(s);
  free(p);
  free(h); 
}

//-----------------------------------------------------------------------------
// FilterForDecimate() calculates the coefficients of low-pass filter and 
// carries out the filtering. This function is only used for decimate().
// Input: 
//   x          : Input signal
//   x_length   : Length of x
//   r          : Coefficient used for down sampling 
//                (fs after down sampling is fs/r)
// Output: 
//   y          : Output signal
//-----------------------------------------------------------------------------
void FilterForDecimate(double *x, int x_length, int r, 
  double *y)
{
  double w[3] = {0.0, 0.0, 0.0};

  double a[3], b[2]; // filter Coefficients
  switch(r)
  {
  case 11: // fs : 44100 (default)
    a[0] = 2.450743295230728;
    a[1] = -2.06794904601978;
    a[2] = 0.59574774438332101;
    b[0] = 0.0026822508007163792;
    b[1] = 0.0080467524021491377;
    break;
  case 12: // fs : 48000
    a[0] = 2.4981398605924205;
    a[1] = -2.1368928194784025;
    a[2] = 0.62187513816221485;
    b[0] = 0.0021097275904709001;
    b[1] = 0.0063291827714127002;
    break;
  case 10: 
    a[0] = 2.3936475118069387;
    a[1] = -1.9873904075111861;
    a[2] = 0.5658879979027055;
    b[0] = 0.0034818622251927556;
    b[1] = 0.010445586675578267;
    break;
  case 9:
    a[0] = 2.3236003491759578;
    a[1] = -1.8921545617463598;
    a[2] = 0.53148928133729068;
    b[0] = 0.0046331164041389372;
    b[1] = 0.013899349212416812;
    break;
  case 8: // fs : 32000
    a[0] = 2.2357462340187593;
    a[1] = -1.7780899984041358;
    a[2] = 0.49152555365968692;
    b[0] = 0.0063522763407111993;
    b[1] = 0.019056829022133598;
    break;
  case 7:
    a[0] = 2.1225239019534703;
    a[1] = -1.6395144861046302;
    a[2] = 0.44469707800587366;
    b[0] = 0.0090366882681608418;
    b[1] = 0.027110064804482525;
    break;
  case 6: // fs : 24000 and 22050
    a[0] = 1.9715352749512141;
    a[1] = -1.4686795689225347;
    a[2] = 0.3893908434965701;
    b[0] = 0.013469181309343825;
    b[1] = 0.040407543928031475;
    break;
  case 5: 
    a[0] = 1.7610939654280557;
    a[1] = -1.2554914843859768;
    a[2] = 0.3237186507788215;
    b[0] = 0.021334858522387423;
    b[1] = 0.06400457556716227;
    break;
  case 4: // fs : 16000
    a[0] = 1.4499664446880227;
    a[1] = -0.98943497080950582;
    a[2] = 0.24578252340690215;
    b[0] = 0.036710750339322612;
    b[1] = 0.11013225101796784;
    break;
  case 3:
    a[0] = 0.95039378983237421;
    a[1] = -0.67429146741526791;
    a[2] = 0.15412211621346475;
    b[0] = 0.071221945171178636;
    b[1] = 0.21366583551353591;
    break;
  case 2: // fs : 8000
    a[0] = 0.041156734567757189;
    a[1] = -0.42599112459189636;
    a[2] = 0.041037215479961225;
    b[0] = 0.16797464681802227;
    b[1] = 0.50392394045406674;
  }

  double wt;
  for(int i = 0;i < x_length;i++)
  {
    wt = x[i] + a[0]*w[0] + a[1]*w[1] + a[2]*w[2];
    y[i] = b[0]*wt + b[1]*w[0] + b[1]*w[1] + b[0]*w[2];
    w[2] = w[1]; 
    w[1] = w[0]; 
    w[0] = wt;
  }
}

//-----------------------------------------------------------------------------
// decimate() carries out down sampling by both IIR and FIR filters.
// Filter coeffiencts are based on FilterForDecimate().
// Input:
//   x          : Input signal
//   x_length   : Length of x
//   r          : Coefficient used for down sampling 
//                (fs after down sampling is fs/r)
// Output:
//   y          : Output signal
//-----------------------------------------------------------------------------
void decimate(const double *x, int x_length, int r,
  double *y)
{
  int nfact = 9; // 多分これは固定でOK
  double *tmp1 = (double *)malloc(sizeof(double) * (x_length + nfact*2));
  double *tmp2 = (double *)malloc(sizeof(double) * (x_length + nfact*2));

  for(int i = 0;i < nfact;i++) tmp1[i] = 2*x[0] - x[nfact-i];
  for(int i = nfact;i < nfact+x_length;i++) tmp1[i] = x[i-nfact];
  for(int i = nfact+x_length;i < 2*nfact+x_length;i++) 
    tmp1[i] = 2*x[x_length-1] - x[x_length-2 - (i-(nfact+x_length))];

  FilterForDecimate(tmp1, 2*nfact+x_length, r, tmp2);
  for(int i = 0;i < 2*nfact+x_length;i++) 
    tmp1[i] = tmp2[2*nfact+x_length - i - 1];
  FilterForDecimate(tmp1, 2*nfact+x_length, r, tmp2);
  for(int i = 0;i < 2*nfact+x_length;i++) 
    tmp1[i] = tmp2[2*nfact+x_length - i - 1];

  int nout = (int)(x_length/r) + 1;
  int nbeg = r - (r*nout - x_length);

  int count = 0;
  for(int i = nbeg;i < x_length+nfact;i+=r, count++) 
    y[count] = tmp1[i+nfact-1];

  free(tmp1); free(tmp2);
  return;
}

//-----------------------------------------------------------------------------
// matlab_round() calculates rounding.
// Input: 
//   x    : Input value
// Output:
//   y    : Output value (rounded)
//-----------------------------------------------------------------------------
int matlab_round(double x)
{
  return x>0 ? (int)(x+0.5) : (int)(x-0.5);
}

//-----------------------------------------------------------------------------
// diff() calculates differences and approximate derivatives
// http://www.mathworks.co.jp/help/techdoc/ref/diff.html
// Input: 
//   x          : Input signal
//   x_length   : Length of x
// Output:
//   y          : Output signal
//-----------------------------------------------------------------------------
void diff(double *x, int x_length, 
  double *y)
{
  for(int i = 0;i < x_length-1;i++) y[i] = x[i+1] - x[i];
  return;
}

//-----------------------------------------------------------------------------
// interp1Q() is the special case of interp1().
// We can use this function, provided that All periods of x-axis is the same.
// Input: 
//   x          : Origin of the x-axis
//   shift      : Period of the x-axis
//   y          : Values at x[n]
//   x_length   : Length of x (Length of y must be the same)
//   xi         : Required vector
//   xi_length  : Length of xi (Length of yi must be the same)
// Output:
//   yi         : Interpolated vector
// Caution:
//   Length of xi and yi must be the same.
//-----------------------------------------------------------------------------
void interp1Q(double x, double shift, double *y, int x_length, double *xi, 
  int xi_length, 
  double *yi)
{
  double *xiFraction = (double *)malloc(xi_length*sizeof(double));
  double *deltaY     = (double *)malloc(x_length*sizeof(double));
  int *xiBase = (int *)malloc(xi_length*sizeof(int));

  double deltaX = shift;
  for(int i = 0;i < xi_length;i++)
  {
    xiBase[i] = (int)floor((xi[i] - x) / deltaX);
    xiFraction[i] = (double)(xi[i]-x)/deltaX - (double)xiBase[i];
  }
  diff(y, x_length, deltaY);
  deltaY[x_length-1] = 0.0;

  for(int i = 0;i < xi_length;i++)
    yi[i] = y[xiBase[i]] + deltaY[xiBase[i]]*xiFraction[i]; 

  free(xiFraction);
  free(xiBase);
  free(deltaY);
}

//-----------------------------------------------------------------------------
// randn() generates pseudorandom numbers based on xorshift method.
// Output:
//   A generated pseudorandom number
//-----------------------------------------------------------------------------
double randn(void) 
{
  static unsigned int x = 123456789;
  static unsigned int y = 362436069;
  static unsigned int z = 521288629;
  static unsigned int w = 88675123; 
  unsigned int t;
   t = x^(x << 11);
  x = y; y = z; z = w;

  unsigned int tmp = 0;
  for(int i = 0;i < 12;i++)
  {
     t = x ^ (x << 11);
    x = y; y = z; z = w;
    w = (w ^ (w >> 19)) ^ (t ^ (t >> 8));
    tmp += w>>4;
  }
  return (double)tmp / 268435456.0 - 6.0f;
}

//-----------------------------------------------------------------------------
// fftfilt() carries out the convolution on the frequency domain.
// Input:
//   x          : Input signal
//   x_length   : Length of x
//   h          : Impulse response
//   h_length   : Length of h
//   fft_size   : Length of FFT
// Output:
//   y          : Calculated result.
//-----------------------------------------------------------------------------
void fftfilt(double *x, int x_length, double *h, int h_length, int fft_size, 
  double *y)
{
  double *input1 = (double *)malloc(sizeof(double) * fft_size);
  double *input2 = (double *)malloc(sizeof(double) * fft_size);
  fft_complex *specx = (fft_complex *)malloc(sizeof(fft_complex) * fft_size);
  fft_complex *spech = (fft_complex *)malloc(sizeof(fft_complex) * fft_size);

  fft_plan forwardFFT1 = 
    fft_plan_dft_r2c_1d(fft_size, input1, specx, FFT_ESTIMATE);
  fft_plan forwardFFT2 = 
    fft_plan_dft_r2c_1d(fft_size, input2, spech, FFT_ESTIMATE);
  fft_plan inverseFFT = fft_plan_dft_c2r_1d(fft_size, specx, y, FFT_ESTIMATE);

  for(int i = 0;i < x_length;i++) input1[i] = x[i]/(double)fft_size;
  for(int i = x_length; i < fft_size;i++) input1[i] = 0;
  for(int i = 0;i < h_length;i++) input2[i] = h[i];
  for(int i = h_length; i < fft_size;i++) input2[i] = 0;

  fft_execute(forwardFFT1);
  fft_execute(forwardFFT2);

  double tmp_real, tmp_imag;
  for(int i = 0;i <= fft_size/2;i++)
  {
    tmp_real = specx[i][0]*spech[i][0] - specx[i][1]*spech[i][1];
    tmp_imag = specx[i][0]*spech[i][1] + specx[i][1]*spech[i][0];
    specx[i][0] = tmp_real;
    specx[i][1] = tmp_imag;
  }
  fft_execute(inverseFFT);

  free(input1); free(input2);
  free(specx); free(spech);
  fft_destroy_plan(forwardFFT1);
  fft_destroy_plan(forwardFFT2);
  fft_destroy_plan(inverseFFT);
}

//-----------------------------------------------------------------------------
// inv() calculates the inverse matrix of input square matrix.
// Input:
//   r     : Input square matrix;
//   n     : Number of dimensions of the input
// Output:
//   invr  : Calculated inverse matrix.
//-----------------------------------------------------------------------------
void inv(double **r, int n, 
  double **invr)
{
  int i,j,k;

  for(i = 0;i < n;i++)
    for(j = 0;j < n;j++) invr[i][j] = 0.0;
  for(i = 0;i < n;i++) invr[i][i] = 1.0;

  double tmp;
  for(i = 0;i < n;i++)
  {
    tmp = r[i][i]; r[i][i] = 1.0;
    for(j = 0;j <= i;j++) invr[i][j] /= tmp;
    for(;j < n;j++) r[i][j]  /= tmp;
    for(j = i+1;j < n;j++)
    {
      tmp = r[j][i];
      for(k = 0;k <= i;k++) invr[j][k] -= invr[i][k]*tmp;
      for(k--;k < n;k++) r[j][k] -= r[i][k]*tmp;
    }
  }

  for(i = n-1;i >= 0;i--)
  {
    for(j = 0;j < i;j++)
    {
      tmp = r[j][i];
      for(k = 0;k < n;k++) invr[j][k] -= invr[i][k]*tmp;
    }
  }
}

//-----------------------------------------------------------------------------
// matlab_std() calculates the standard deviation of the input vector.
// Input:
//   x          : Input vector
//   x_length   : Length of x
// Output:
//   Calculated standard deviation
//-----------------------------------------------------------------------------
double matlab_std(double *x, int x_length)
{
  double average = 0.0;
  for(int i = 0;i < x_length;i++) average += x[i];
  average /= (double)x_length;

  double s = 0.0;
  for(int i = 0;i < x_length;i++) s += pow(x[i] - average, 2.0);
  s /= (double)(x_length-1);

  return sqrt(s);
}

//-----------------------------------------------------------------------------
// wavwrite() write a .wav file.
// Input:
//   x          : Input signal
//   waveLength : Signal length of x [sample]
//   fs         : Sampling frequency [Hz]
//   Nbit       : Quantization bit [bit]
//   filename   : Name of the output signal.
// Caution:
//   The variable nbit is not used in this function.
//   This function only supports the 16 bit.
//-----------------------------------------------------------------------------
void wavwrite(double *x, int x_length, int fs, int nbit, char *filename)
{
  FILE *fp = fopen(filename, "wb");
  if(fp == NULL) 
  {
    printf("File cannot be opened.\n");
    return;
  }

  char text[4] = {'R', 'I', 'F', 'F'};
  long long_number = 36 + x_length*2;
  fwrite(text, 1, 4, fp);
  fwrite(&long_number, 4, 1, fp);

  text[0] = 'W'; text[1] = 'A'; text[2] = 'V'; text[3] = 'E'; 
  fwrite(text, 1, 4, fp);
  text[0] = 'f'; text[1] = 'm'; text[2] = 't'; text[3] = ' '; 
  fwrite(text, 1, 4, fp);

  long_number = 16;
  fwrite(&long_number, 4, 1, fp);
  short short_number = 1;
  fwrite(&short_number, 2, 1, fp);
  short_number = 1;
  fwrite(&short_number, 2, 1, fp);
  long_number = (long)fs;
  fwrite(&long_number, 4, 1, fp);
  long_number = (long)fs * 2;
  fwrite(&long_number, 4, 1, fp);
  short_number = 2;
  fwrite(&short_number, 2, 1, fp);
  short_number = 16;
  fwrite(&short_number, 2, 1, fp);

  text[0] = 'd'; text[1] = 'a'; text[2] = 't'; text[3] = 'a'; 
  fwrite(text, 1, 4, fp);
  long_number = x_length*2;
  fwrite(&long_number, 4, 1, fp);

  short sTmp;
  for(int i = 0;i < x_length; i++)
  {
    sTmp = (short)MAX(-32768, MIN(32767, x[i]*32767));
    fwrite(&sTmp, 2, 1, fp);
  }

  fclose(fp);
}

//-----------------------------------------------------------------------------
// wavread() read a .wav file.
// Input:
//   filename     : Filename to write a file.
//   fs           : Sampling frequency [Hz]
//   nbit         : Quantization bit [bit]
//   waveLength   : Signal length of the output signal [sample].
// Output:
//   Output .wav file (double *)
//-----------------------------------------------------------------------------
double * wavread(char* filename, int *fs, int *nbit, int *wav_length)
{
  FILE *fp = fopen(filename, "rb");
  if(NULL == fp) 
  {
    printf("File not found.\n");
    return NULL;
  }

  char dataCheck[5]; 
  fread(dataCheck, sizeof(char), 4, fp); // "RIFF"
  dataCheck[4] = '\0';
  if(0 != strcmp(dataCheck,"RIFF"))
  {
    fclose(fp);
    printf("RIFF error.\n");
    return NULL;
  }
  fseek(fp, 4, SEEK_CUR);
  fread(dataCheck, sizeof(char), 4, fp); // "WAVE"
  if(0 != strcmp(dataCheck,"WAVE"))
  {
    fclose(fp);
    printf("WAVE error.\n");
    return NULL;
  }
  fread(dataCheck, sizeof(char), 4, fp); // "fmt "
  if(0 != strcmp(dataCheck,"fmt "))
  {
    fclose(fp);
    printf("fmt error.\n");
    return NULL;
  }
  fread(dataCheck, sizeof(char), 4, fp); //1 0 0 0
  if(!(16 == dataCheck[0] && 0 == dataCheck[1] && 
    0 == dataCheck[2] && 0 == dataCheck[3]))
  {
    fclose(fp);
    printf("fmt (2) error.\n");
    return NULL;
  }
  fread(dataCheck, sizeof(char), 2, fp); //1 0
  if(!(1 == dataCheck[0] && 0 == dataCheck[1]))
  {
    fclose(fp);
    printf("Format ID error.\n");
    return NULL;
  }
  fread(dataCheck, sizeof(char), 2, fp); //1 0
  if(!(1 == dataCheck[0] && 0 == dataCheck[1]))
  {
    fclose(fp);
    printf("This function cannot support a stereo file\n");
    return NULL;
  }
  // Sampling frequency
  unsigned char forIntNumber[4];
  fread(forIntNumber, sizeof(char), 4, fp);
  *fs = 0;
  for(int i = 3;i >= 0;i--) *fs = *fs*256 + forIntNumber[i];
  // Quantization
  fseek(fp, 6, SEEK_CUR);
  fread(forIntNumber, sizeof(char), 2, fp);
  *nbit = forIntNumber[0];

  // Skip until "data" is found. 2011/3/28
  while(0 != fread(dataCheck, sizeof(char), 1, fp))
  {
    if(dataCheck[0] == 'd') 
    {
      fread(&dataCheck[1], sizeof(char), 3, fp);
      if(0 != strcmp(dataCheck,"data")) fseek(fp, -3, SEEK_CUR);
      else break;
    }
  }
  if(0 != strcmp(dataCheck,"data"))
  {
    fclose(fp);
    printf("data error.\n");
    return NULL;
  }

  fread(forIntNumber, sizeof(char), 4, fp); // "data"
  *wav_length = 0;
  for(int i = 3;i >= 0;i--) *wav_length = *wav_length*256 + forIntNumber[i];
  *wav_length /= (*nbit/8);

  double *waveForm = (double *)malloc(sizeof(double) * *wav_length);
  if(waveForm == NULL) return NULL;

  int quantizationByte = *nbit/8;
  double zeroLine = pow(2.0,*nbit-1);
  double tmp, signBias;
  for(int i = 0;i < *wav_length;i++)
  {
    signBias = tmp = 0.0;
    fread(forIntNumber, sizeof(char), quantizationByte, fp); // "data"
    if(forIntNumber[quantizationByte-1] >= 128)
    {
      signBias = pow(2.0,*nbit-1);
      forIntNumber[quantizationByte-1] = 
        forIntNumber[quantizationByte-1] & 0x7F;
    }
    for(int j = quantizationByte-1;j >= 0;j--) 
      tmp = tmp*256.0 + (double)(forIntNumber[j]);
    waveForm[i] = (double)((tmp - signBias) / zeroLine);
  }
  fclose(fp);
  return waveForm;
}

}
}
}
