//-----------------------------------------------------------------------------
// WORLD: High-quality voice analysis, modification and synthesis system.
//
// Version 0.1.0: This version used Ooura's FFT instead of FFTW. (2012/04/02)
//                Ooura's FFT http://www.kurims.kyoto-u.ac.jp/~ooura/index-j.html 
// Version 0.1.1: Almost all of files were rewritten based on Google C++ 
//                Style Guide (2012/09/02)
//                http://google-styleguide.googlecode.com/svn/trunk/cppguide.xml
//
// Specification (This is not a bug)
// decimateForF0 :  This function uses an IIR filter for downsampling.
//          We can't calculate accurate results at 0...4 and N-5...N-1 sample.
//-----------------------------------------------------------------------------
#include "fft.h"

namespace stand
{
namespace math
{
namespace dsp
{

// This is for Linux.
#if (defined (__linux__) || defined(__CYGWIN__))
 #include <stdint.h>
 #ifndef ECHOPRT
 #define ECHOPRT ECHOE
 #endif 
#endif

// Macro
const double PI = 3.1415926535897932384;
const int MAX_FFT_LENGTH = 2048;
const double FLOOR_F0 = 71.0;
const double DEFAULT_F0 = 150.0;
const double LOW_LIMIT = 65.0;
// Note to me (fs: 44100)
// 71 Hz is the limit to maintain the FFT size at 2048.
// If we use 70 Hz as FLOOR_F0, the FFT size of 4096 is required.

//-----------------------------------------------------------------------------
// Struct for DIO
//-----------------------------------------------------------------------------
typedef struct{
  double f0_floor;
  double f0_ceil;
  double channels_in_octave;
  double frame_period; // (msec)
  int speed; // (1, 2, ..., 12)
} DioOption;

//-----------------------------------------------------------------------------
// Structs on FFT
//-----------------------------------------------------------------------------
// Forward FFT in the real sequence
typedef struct{
  int fft_size;
  double *waveform;
  fft_complex *spectrum;
  fft_plan forward_fft;
} ForwardRealFFT;

// Inverse FFT in the real sequence
typedef struct{
  int fft_size;
  double *waveform;
  fft_complex *spectrum;
  fft_plan inverse_fft;
} InverseRealFFT;

// Minimum phase analysis from logarithmic power spectrum
typedef struct{
  int fft_size;
  double *log_spectrum;
  fft_complex *minimum_phase_spectrum;
  fft_complex *cepstrum;
  fft_plan inverse_fft;
  fft_plan forward_fft;
} MinimumPhaseAnalysis;

// These functions are used to speed up the processing.
// Forward FFT
void InitializeForwardRealFFT(int fft_size, ForwardRealFFT *forward_real_fft);
void DestroyForwardRealFFT(ForwardRealFFT *forward_real_fft);

// Inverse FFT
void InitializeInverseRealFFT(int fft_size, InverseRealFFT *inverse_real_fft);
void DestroyInverseRealFFT(InverseRealFFT *inverse_real_fft);

// Minimum phase analysis (This analysis uses FFT)
void InitializeMinimumPhaseAnalysis(int fft_size, 
  MinimumPhaseAnalysis *minimum_phase);
void GetMinimumPhaseSpectrum(MinimumPhaseAnalysis *minimum_phase);
void DestroyMinimumPhaseAnalysis(MinimumPhaseAnalysis *minimum_phase);

//-----------------------------------------------------------------------------
// WORLD functions
//-----------------------------------------------------------------------------
// DIO (F0 estimation)
// 0.1.0
void Dio(const double *x, int x_length, int fs, double frame_period,
  double *time_axis, double *f0);
// 0.1.1 new version
void Dio(const double *x, int x_length, int fs, DioOption option,
  double *time_axis, double *f0);
// This function  0.1.1
void InitializeDioOption(DioOption *option);
int GetSamplesForDIO(int fs, int x_length, double frame_period);

// STAR (spectral envelope estimation)
void Star(const double *x, int x_length, int fs, double *time_axis, double *f0,
     double **spectrogram);
int GetFFTSizeForStar(int fs);

// PLATINUM (Residual signal extraction)
void Platinum(const double *x, int x_length, int fs, double *time_axis, double *f0,
  double **spectrogram, double **residual_spectrogram);

// Aperiodicity estimation
void CalculateAperiodicity(double *aperiodicity, int number_of_bands, 
  int fft_size, double f0, int fs, double target_f0, double *periodic_spec);
int GetBands_v3(int fs);
void AperiodicityRatio_v3(const double *x, int x_length, int fs, double *f0,
  double frame_period, double **aperiodicity, double *target_f0_output);

// Synthesis
void Synthesis(double *f0, int f0_length, double **spectrogram, 
  double **residual_spectrogram, int fft_size, double frame_period, int fs, 
  int y_length, double *y);
void synthesis_ap(double *f0, int f0_length, double **spectrogram, 
  int fft_size, double **aperiodicity, int number_of_bands, double target_f0, 
  double frame_period, int fs, int y_length, double *synthesisOut);


//-----------------------------------------------------------------------------
// Matlab functions (Please see matlabfunctions.cpp)
//-----------------------------------------------------------------------------
void histc(const double *x, int x_length, double *edges, int edges_length,
  int *index);
void interp1(const double *x, double *y, int x_length, double *xi, int xi_length,
  double *yi);
void decimate(const double *x, int x_length, int r, double *y);
int matlab_round(double x);
void diff(double *x, int x_length, double *y);
void interp1Q(double x, double shift, double *y, int x_length, double *xi, 
  int xi_length, double *yi);
double randn(void);
void fftfilt(double *x, int x_length, double *h, int h_length, int fft_size, 
  double *y);
void inv(double **r, int n, double **invr);
double matlab_std(double *x, int x_length);
double * wavread(char* filename, int *fs, int *nbit, int *wav_length);
void wavwrite(double *x, int x_length, int fs, int nbit, char *filename);

}
}
}
