#include "TreeItemModel.h"

#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QTextStream>
#include <QTextCodec>

#include "TreeItem.h"

using namespace stand::model;

xml::TreeXmlFactory TreeItemModel::_factory;

TreeItemModel::TreeItemModel(QObject *parent) :
    QAbstractItemModel(parent)
{
    _root = NULL;
}

TreeItemModel::~TreeItemModel()
{
    onDestroy();
}

void TreeItemModel::onDestroy()
{
    delete _root;
    _root = NULL;
}

Qt::ItemFlags TreeItemModel::flags(const QModelIndex &index) const
{
    if(!index.isValid())
    {
        return 0;
    }
    return static_cast<TreeItem *>(index.internalPointer())->flags();
}

QModelIndex TreeItemModel::index(int row, int column, const QModelIndex &parent) const
{
    if (!hasIndex(row, column, parent))
    {
        qDebug("TreeItemModel(%d, %d, %s); // no item exists.", row, column, ((TreeItem*)(parent.internalPointer()))->name().toLocal8Bit().data());
        //return QModelIndex();
    }

    TreeItem *parentItem;

    if (!parent.isValid())
    {
        parentItem = _root;
    }
    else
    {
        parentItem = static_cast<TreeItem*>(parent.internalPointer());
    }

    TreeItem *childItem = parentItem->child(row);

    if(childItem)
    {
        return createIndex(row, column, childItem);
    }
    else
    {
        return QModelIndex();
    }
}

QModelIndex TreeItemModel::parent(const QModelIndex &index) const
{
    if(!index.isValid())
    {
        return QModelIndex();
    }
    TreeItem *c, *p;
    c = static_cast<TreeItem *>(index.internalPointer());
    p = c->parent();
    if(p == _root)
    {
        return QModelIndex();
    }
    return createIndex(p->row(), 0, p);
}

int TreeItemModel::rowCount(const QModelIndex &parent) const
{
    TreeItem *p;
    if(parent.column() > 0)
    {
        return 0;
    }
    if(!parent.isValid() || !parent.internalPointer())
    {
        p = _root;
    }
    else
    {
        p = static_cast<TreeItem *>(parent.internalPointer());
    }
    return p->childCount();
}

int TreeItemModel::columnCount(const QModelIndex &parent) const
{
    if(parent.isValid())
    {
        return static_cast<TreeItem *>(parent.internalPointer())->columnCount();
    }
    return _root->columnCount();
}

QVariant TreeItemModel::data(const QModelIndex &index, int role) const
{
    if(index.isValid())
    {
        return static_cast<TreeItem *>(index.internalPointer())->data(index.column(), role);
    }
    return QVariant();
}

QVariant TreeItemModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if(orientation == Qt::Horizontal && role == Qt::DisplayRole)
    {
        return QVariant(_root->name());
    }
    if(orientation == Qt::Vertical && role == Qt::DisplayRole)
    {
        TreeItem *i = _root->child(section);
        if(i)
        {
            return QVariant(i->name());
        }
    }
    return QVariant();
}

bool TreeItemModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if(!index.isValid())
    {
        return false;
    }

    TreeItem *item = static_cast<TreeItem *>(index.internalPointer());
    if(!item)
    {
        return false;
    }
    item->setData(index.column(), value, role);
    emit (dataChanged(index, index));
    return true;
}

bool TreeItemModel::write(const QString &path) const
{
    QFile file(QDir::toNativeSeparators(path));
    if(!file.open(QIODevice::WriteOnly))
    {
        qDebug("TreeItemModel::write(%s);// File could not be open.", path.toUtf8().data());
        return false;
    }

    QDomDocument doc;
    QDomNode header = doc.createProcessingInstruction("xml", "version=\"1.0\" encoding=\"UTF-8\"");
    QDomElement root = doc.createElement("stand");
    root.setAttribute("type", type());
    if(_root)
    {
        root.appendChild(_root->domElement(doc));
    }
    doc.appendChild(root);


    doc.insertBefore(header, doc.firstChild());
    QTextStream out(&file);
    out.setCodec(QTextCodec::codecForName("UTF-8"));
    doc.save(out, 1);

    return true;
}

bool TreeItemModel::read(const QString &path)
{
    QFile file(QDir::toNativeSeparators(path));
    if(!file.exists())
    {
        qDebug("TreeItemModel::read(%s); // File not found.", QFileInfo(path).absoluteFilePath().toUtf8().data());
        return false;
    }
    if(!file.open(QIODevice::ReadOnly))
    {
        qDebug("TreeItemModel::read(%s);// File could not be open.", QFileInfo(path).absoluteFilePath().toUtf8().data());
        return false;
    }

    QString errorMsg;
    int errorLine, errorColumn;
    QDomDocument doc;
    if(!doc.setContent(&file, true, &errorMsg, &errorLine, &errorColumn))
    {
        qDebug("TreeItemModel::read(%s); // Parse error.", QFileInfo(path).absoluteFilePath().toUtf8().data());
        qDebug(" error at line %d, column %d; %s", errorLine, errorColumn, errorMsg.toUtf8().data());
        return false;
    }

    QDomElement root = doc.documentElement();
    if(root.tagName() != "stand" || root.attribute("type") != type())
    {
        qDebug("TreeItemModel::read(%s); // Format is invalid.", QFileInfo(path).absoluteFilePath().toUtf8().data());
        qDebug(" Root tag is %s, and attribute type is %s.", root.tagName().toUtf8().data(), root.attribute("type").toUtf8().data());
        return false;
    }
    QDomNode node = root.firstChild();
    TreeItem *newRoot = new TreeItem("");

    newRoot->setDomNode(node, factory());

    onDestroy();
    _root = newRoot;

    onRead();
    return true;
}

bool TreeItemModel::import(const QString &path, importer::Importer *importer)
{
    if(!importer)
    {
        return false;
    }

    TreeItem *newRoot = new TreeItem(type());

    if(!importer->import(newRoot, QDir::toNativeSeparators(path)))
    {
        delete newRoot;
        return false;
    }

    onDestroy();
    _root = newRoot;
    onRead();

    return true;
}
