#ifndef UTILITY_H
#define UTILITY_H

#include <QDir>

class QWidget;

namespace stand
{
namespace io
{
namespace utau
{
struct UtauPhoneme;
}
}
namespace utility
{
/*!
 * @brief ディレクトリを作成します．作成したいディレクトリがある場合は何もしません．
 * @param[in] w エラー時，メッセージを出すウィジェット．
 * @param[in] dir 作成したいディレクトリパス．
 * @param[in] confirm 指定したパスが存在しない場合作成の許可をユーザに求めるかどうかのフラグ．
 * @returns true: 作成が成功したか，ディレクトリが存在した． false: ディレクトリが存在せず作成もできなかった．
 */
bool makeDirectory(QWidget *w, const QDir &dir, bool confirm = false);

/*!
 * @brief UTAU 音源の音素片１つ分のサンプル数を計算します．
 * @param[in] waveLength 波形全体の長さ．
 * @param[in] samplingFrequency 標本化周波数．
 * @param[in] leftBlank 原音設定の左ブランク．
 * @param[in] rightBlank 原音設定の右ブランク．
 * @return 原音設定に対応する信号のサンプル数．
 */
int lengthForUtauSetting(int waveLength, int samplingFrequency, float leftBlank, float rightBlank);

double msLengthForUtauPhoneme(const char *path, const io::utau::UtauPhoneme *p);

/*!
 * @brief 信号波形を端点で正規化します．左端右端どちらか振幅包絡の大きい方の値で正規化されます.
 * @param[in/out] x 正規化する信号列．
 * @param[in] length 信号長．
 */
void normalizeByAnchor(double *x, unsigned int length);

/*!
 * @brief ビッグエンディアンからリトルエンディアンに変更します．
 */
int FromBigEndian(int i);

/*!
 * @brief ビッグエンディアンからリトルエンディアンに変更します．
 */
short FromBigEndian(short s);

/*!
 * @brief ノート番号が黒鍵であるかどうかを調べます．
 */
bool IsBlackKey(unsigned char note);

/*!
 * @brief ノート番号に対応する文字列を返します．
 */
QString NoteName(unsigned char note);

/*!
 * @brief 文字列に対応するノート番号を返します．
 * @note 今のところ実装が重いので注意．
 *       てか２つ関数あるんだけどどういうこと？
 */
int NoteNumber(const QString &name);
int NoteFromString(const QString &name);

/*!
 * @brief ノート番号に対応する周波数を返します．
 */
double NoteFrequency(int note);
/*!
 * @brief ノート番号に対応する周波数を返します．（ノート番号を実数指定する版）
 */
double NoteFrequency(double note);

/*!
 * @brief 周波数に対応するノート番号を返します．
 * @note 戻り値のノート番号は小数点以下を含みます．
 */
double NoteAt(double frequency);

/*!
 * @brief BPM 一定のときミリ秒に対応するティック時刻を返します．
 *        ms -> ticks 変換．
 * @param[in] ms ミリ秒．
 * @param[in] bpm BPM
 * @return BPMが bpm のとき ms(msec.) に対応するティック時間．
 */
double TicksFromMs(double ms, double bpm);
/*!
 * @brief BPM 一定のときティック時間に対応するミリ秒を返します．
 *        ticks -> ms 変換．
 * @param[in] tick ティック時間．
 * @param[in] bpm BPM
 * @return BPM が bpm のとき tick に対応するミリ秒単位の時間．
 */
double MsFromTick(int tick, double bpm);

}
}

#endif // UTILITY_H
