#include "ViewSettings.h"

#include <QPoint>

#include "../configure.h"
#include "Utility.h"

#include "ControlItem.h"
#include "MeasureItem.h"
#include "TrackItem.h"
#include "SequenceModel.h"

using namespace stand::view;

ViewSettings::ViewSettings(int preMeasure, int noteHeight, int beatWidth, QObject *parent) :
    QObject(parent)
{
    _preMeasure = preMeasure;
    _noteHeight = noteHeight;
    _beatWidth = beatWidth;

    _isSnap = false;
    _snapDiv = 4;

    _currentBarTick = 0;
    _controlResolution = 16;
    _tempo = NULL;
    _measure = NULL;
    _undoStack = NULL;
    _controlIdPrevious = -1;
}

ViewSettings::~ViewSettings()
{
    emit AboutToDelete();
}

void ViewSettings::setBeatWidth(int b)
{
    _beatWidth = b;
    emit StateChanged();
}

void ViewSettings::setNoteHeight(int n)
{
    _noteHeight = n;
    emit StateChanged();
}

void ViewSettings::setPreMeasure(int p)
{
    _preMeasure = p;
    emit StateChanged();
}

unsigned char ViewSettings::noteAt(int y) const
{
    return 127 - y / _noteHeight;
}

int ViewSettings::yAt(int note) const
{
    int c = 127 - note;
    return c * _noteHeight;
}

int ViewSettings::yAt(double frequency) const
{
    double n = 127.0 - stand::utility::NoteAt(frequency);
    return (n + 0.5) * _noteHeight;
}

double ViewSettings::frequencyAt(int y) const
{
    double retval = 127.0 - (double)y / (double)(_noteHeight) + 0.5;
    return stand::utility::NoteFrequency(retval);
}

int ViewSettings::xAt(int tick) const
{
    QPoint p(4, 4);
    if(_measure)
    {
        p = _measure->data(0, 0).toPoint();
    }
    int retval = _beatWidth * (tick + _preMeasure * p.x() * 4 * stand::sequence::DefaultTicksPerBeat / std::max(1, p.y()) ) / stand::sequence::DefaultTicksPerBeat;
    return retval;
}

int ViewSettings::tickAt(int x) const
{
    QPoint p(4, 4);
    if(_measure)
    {
        p = _measure->data(0, 0).toPoint();
    }
    int tick = x * stand::sequence::DefaultTicksPerBeat / _beatWidth - _preMeasure * p.x() * 4 * stand::sequence::DefaultTicksPerBeat / p.y();
    return tick;
}

void ViewSettings::setModel(stand::model::SequenceModel *model)
{
    _tracks.clear();
    _controls.clear();
    if(model)
    {
        _tempo = model->tempo();
        _measure = model->measure();
        _preMeasure = model->premeasure();
        _undoStack = model->undoStack();
        _controlIdPrevious = -1;
        stand::model::TrackItem *t;
        for(int i = 0; (t = model->track(i)); i++)
        {
            _tracks.push_back(t);
        }
        setTrackId(0);
        setControlId(0);
    }
    else
    {
        _tempo = NULL;
        _measure = NULL;
        _undoStack = NULL;
    }
    emit resetAll();
    emit StateChanged();
}

void ViewSettings::setTrackId(int id)
{
    if(0 <= id && id < _tracks.size())
    {
        _trackId = id;
        _controls.clear();
        _controls = _tracks[id]->controls();
        _controlSize = 0;
        for(int i = 0; i < _controls.size(); i++)
        {
            if(_controls[i]->data(stand::model::ControlItem::ViewType, 0).toInt() == stand::model::ControlItem::TypeNormal)
            {
                _controlSize++;
            }
        }
    }
    else
    {
        _trackId = -1;
    }
    emit trackChanged();
}

void ViewSettings::setControlId(int id)
{
    if(0 <= id && id < _controlSize && _controlId != id)
    {
        _controlIdPrevious = _controlId;
        _controlId = id;
        emit controlTrackChanged();
    }
}

int ViewSettings::numeratorAt(int tick) const
{
    if(!_measure)
    {
        return 4;
    }
    return _measure->data(measureAt(tick), 0).toPoint().x();
}

int ViewSettings::denominatorAt(int tick) const
{
    if(!_measure)
    {
        return 4;
    }
    return _measure->data(measureAt(tick), 0).toPoint().y();
}

double ViewSettings::msAt(int tick) const
{
    if(!_tempo || _tempo->columnCount() <= 0)
    {
        return stand::utility::MsFromTick(tick, stand::sequence::DefaultBeatsPerMinutes);
    }
    const QList<stand::utility::ControlPoint> &t = _tempo->contour();
    if(t.empty())
    {
        return stand::utility::MsFromTick(tick, stand::sequence::DefaultBeatsPerMinutes);
    }

    int current = t.at(0).tick;
    double ms = 0.0;

    for(int i = 1; i < t.size(); i++)
    {
        if(tick <= t[i].tick)
        {
            int diff = t[i].tick - tick;
            ms += stand::utility::MsFromTick(diff, t[i].value);
            return ms;
        }
        ms += stand::utility::MsFromTick(t[i].tick - current, t[i].value);
        current = t[i].tick;
    }
    ms += stand::utility::MsFromTick(t.last().tick - current, t.last().value);
    return ms;
}

int ViewSettings::tickFromMs(double ms) const
{
    if(!_tempo || _tempo->columnCount() <= 0)
    {
        return stand::utility::TicksFromMs(ms, stand::sequence::DefaultBeatsPerMinutes);
    }
    const QList<stand::utility::ControlPoint> &t = _tempo->contour();
    if(t.empty())
    {
        return stand::utility::TicksFromMs(ms, stand::sequence::DefaultBeatsPerMinutes);
    }

    double current = 0.0;
    int tick = 0;

    for(int i = 1; i < t.size(); i++)
    {
        double msThis = ms + stand::utility::MsFromTick(t[i].tick - tick, t[i].value);
        if(ms <= msThis)
        {
            double diff = ms - current;
            tick += stand::utility::TicksFromMs(diff, t[i].value);
            return tick;
        }
        current += stand::utility::MsFromTick(t[i].tick - current, t[i].value);
        tick = t[i].tick;
    }
    tick += stand::utility::MsFromTick(ms - current, t.last().value);

    return tick;
}

bool ViewSettings::isBar(int tick) const
{
    if(!_measure)
    {
        return (tick % (4 * stand::sequence::DefaultTicksPerBeat) == 0);
    }
    const QList<stand::model::MeasureItem::Rhythm> &t = _measure->contour();
    if(t.empty())
    {
        return (tick % (4 * stand::sequence::DefaultTicksPerBeat) == 0);
    }
    int current = 0;
    int currentTick = 0;
    int step = stand::sequence::DefaultTicksPerBeat * 4 * t[0].numerator / t[0].denominator;;

    for(int i = 1; i < t.size(); i++)
    {
        currentTick += (t[i].measure - current) * step;
        if(tick < currentTick)
        {
            if(tick < 0)
            {
                return ((-tick % step) == 0);
            }
            return ((tick - currentTick) % step == 0);
        }
        step = stand::sequence::DefaultTicksPerBeat * 4 * t[i].numerator / t[i].denominator;
        current = t[i].measure;
    }
    if(tick < 0)
    {
        return ((-tick % step) == 0);
    }
    return ((tick - currentTick) % step == 0);
}

int ViewSettings::tickByMeasure(int measureCount) const
{
    if(!_measure)
    {
        return measureCount * stand::sequence::DefaultTicksPerBeat * 4;
    }
    const QList<stand::model::MeasureItem::Rhythm> &t = _measure->contour();
    if(t.empty())
    {
        return measureCount * stand::sequence::DefaultTicksPerBeat * 4;
    }

    if(measureCount < 0)
    {
        return measureCount * stand::sequence::DefaultTicksPerBeat * 4 * _measure->data(0, 0).toPoint().x() / _measure->data(0, 0).toPoint().y();
    }

    int currentTick = 0;
    int currentMeasure = 0;

    for(int i = 1; i < t.size(); i++)
    {
        if(measureCount < t[i].measure)
        {
            currentTick += (measureCount - currentMeasure) * stand::sequence::DefaultTicksPerBeat * 4 * t[i-1].numerator / t[i-1].denominator;
            return currentTick;
        }
        currentTick += (t[i].measure - currentMeasure) * stand::sequence::DefaultTicksPerBeat * 4 * t[i-1].numerator / t[i-1].denominator;
        currentMeasure = t[i].measure;
    }
    currentTick += (measureCount - currentMeasure) * stand::sequence::DefaultTicksPerBeat * 4 * t.last().numerator / t.last().denominator;

    return currentTick;
}

int ViewSettings::measureAt(int tick) const
{
    if(!_measure)
    {
        return tick / (stand::sequence::DefaultTicksPerBeat * 4) * (stand::sequence::DefaultTicksPerBeat * 4);
    }
    const QList<stand::model::MeasureItem::Rhythm> &t = _measure->contour();
    if(t.empty())
    {
        return tick / (stand::sequence::DefaultTicksPerBeat * 4) * (stand::sequence::DefaultTicksPerBeat * 4);
    }

    int current = 0;
    int currentTick = 0;
    int step = stand::sequence::DefaultTicksPerBeat * 4 * t[0].numerator / t[0].denominator;;

    for(int i = 1; i < t.size(); i++)
    {
        int nextTick = currentTick + (t[i].measure - current) * step;
        if(tick <= nextTick)
        {
            if(tick < 0)
            {
                return (tick / step);
            }
            return (current + (tick - currentTick) / step);
        }
        step = stand::sequence::DefaultTicksPerBeat * 4 * t[i].numerator / t[i].denominator;
        current = t[i].measure;
        currentTick = nextTick;
    }
    return current + (tick - currentTick) / step;
}

void ViewSettings::widgetResized(int w, int)
{
    setWidth(w);
}
